import torch
import numpy as np
import torch.nn as nn
from attrdict import AttrDict

from .modules import CrossAttnEncoder2, Decoder, PoolingEncoder, SelfAttn, build_mlp
import time

class MAB(nn.Module):
    def __init__(self, dim_out=128, num_heads=8):
        super().__init__()
        self.dim_out = dim_out
        self.num_heads = num_heads
        self.fc_q = nn.Linear(dim_out, dim_out)
        self.fc_k = nn.Linear(dim_out, dim_out)
        self.fc_v = nn.Linear(dim_out, dim_out)
        self.fc_out = nn.Linear(dim_out, dim_out)
        self.fc_real_out = nn.Linear(dim_out, dim_out)
        self.ln1 = nn.LayerNorm(dim_out)
        self.ln2 = nn.LayerNorm(dim_out)

    def scatter(self, x):
        return torch.cat(torch.split(x, x.shape[-1] // self.num_heads, dim=-1), dim=0)

    def gather(self, x):
        return torch.cat(torch.split(x, x.shape[0] // self.num_heads, dim=0), dim=-1)

    def attend(self, q, k, v, mask=None):
        
        q_, k_, v_ = self.scatter(q), self.scatter(k), self.scatter(v)
        A_logits = q_ @ k_.swapaxes(-2, -1) / np.sqrt(self.dim_out)
        A = torch.nn.functional.softmax(A_logits, dim=-1)
        
        return self.gather(A @ v_)

    def __call__(self, q, v, mask=None):
        q, k, v = self.fc_q(q), self.fc_k(v), self.fc_v(v)
        out = self.ln1(q + self.attend(q, k, v, mask))
        out = self.ln2(out + nn.functional.relu(self.fc_out(out)))
        out = self.fc_real_out(out)
        return out
    
class ISAB(nn.Module):
    def __init__(self, dim_out=128, num_heads=8):
        super().__init__()
        self.mab0 = MAB(dim_out=dim_out, num_heads=num_heads)
        self.mab1 = MAB(dim_out=dim_out, num_heads=num_heads)

    def __call__(self, context, generate_sample, mask_context=None, mask_generate=None):

        h = self.mab0(context, generate_sample, mask_generate)
        return self.mab1(generate_sample, h, mask_context)
    
class SetGenerate(nn.Module):
    def __init__(self, dim_out=128, dim_hidden=128, num_heads=8):
        super().__init__()
        self.isab = ISAB(dim_out=dim_out, num_heads=num_heads)

    def __call__(self, r, generate_num, mask=None):
        generator = torch.Generator().manual_seed(int(time.time()))
        r_front_shape, r_tail_shape = r.shape[:-2], r.shape[-1]
        generate_initial = torch.randn((*r_front_shape, generate_num, r_tail_shape), generator=generator).to(r.device)
        generate_mask = torch.ones((r.shape[0], generate_num), dtype=torch.bool)
        out = self.isab(r, generate_initial, mask, generate_mask)
        return out
    
def autoregressive(**kwargs):
    return SetGenerate(**kwargs)

class MPANP(nn.Module):
    def __init__(self,
            dim_x=1,
            dim_y=1,
            dim_hid=128,
            enc_v_depth=4,
            enc_qk_depth=2,
            enc_pre_depth=4,
            enc_post_depth=2,
            dec_depth=3):

        super().__init__()

        self.net_qk = build_mlp(dim_x, dim_hid, dim_hid, enc_qk_depth)
        self.net_v_pre = build_mlp(dim_x+dim_y, dim_hid, dim_hid, enc_v_depth-2) 
        self.self_attn = SelfAttn(dim_hid, dim_hid)
        self.net_v_post = build_mlp(dim_hid, dim_hid, dim_hid, enc_post_depth)
                
        self.enc1 = CrossAttnEncoder2(
                dim_x=dim_x,
                dim_y=dim_y,
                dim_hid=dim_hid)
        
        self.auto_regressive = autoregressive(dim_out=dim_hid*2)

        self.dec = Decoder(
            dim_x=dim_x,
            dim_y=dim_y,
            dim_enc=dim_hid,
            dim_hid=dim_hid,
            depth=dec_depth)
            
    def _auto_regressive(self, r_i_k, r_i_v, mask_kv, num_samples, num_generates):
        r_i_kv = torch.cat([r_i_k, r_i_v], -1) # [batch, context, r_dim * 2]
        r_i_gen = self.auto_regressive(r_i_kv, num_generates, mask_kv)
        
        r_i_k_gen, r_i_v_gen = torch.split(r_i_gen, r_i_gen.shape[-1] // 2, dim=-1)
        return r_i_k_gen, r_i_v_gen
        
    def predict(self, xc, yc, xt, num_samples=None, num_generates=40, mask=None):
        
        r_i_q = self.net_qk(xt)  # [batch_size, target, dim_hid]
        r_i_k_base = self.net_qk(xc) # [batch_size, context, dim_hid]
        
        r_i_v_base = self.net_v_pre(torch.cat([xc, yc], -1))
        r_i_v_base = self.self_attn(r_i_v_base) # [batch_size, context, dim_hid]
        #r_i_v_base = r_i_v_base.mean(-2)
        #r_i_v_base = self.net_v_post(r_i_v_base)
        
        r_i_k_gen, r_i_v_gen = \
            self._auto_regressive(r_i_k_base, r_i_v_base, mask, num_samples, num_generates)
        
        r_i_k = torch.cat([r_i_k_base, r_i_k_gen], -2) # [batch_size, context + num_generates, dim_hid]
        r_i_v = torch.cat([r_i_v_base, r_i_v_gen], -2)  # [batch_size, context + num_generates, dim_hid]
        
        r_ctx_gen = self.enc1(r_i_q, r_i_k, r_i_v) # [batch_size, target, dim_hid]

        return self.dec(r_ctx_gen, xt)
        
    def forward(self, batch, num_samples=None, reduce_ll=True):
        outs = AttrDict()
        py = self.predict(batch.xc, batch.yc, batch.x)
        ll = py.log_prob(batch.y).sum(-1)

        if self.training:
            outs.loss = -ll.mean()
        else:
            num_ctx = batch.xc.shape[-2]
            if reduce_ll:
                outs.ctx_ll = ll[...,:num_ctx].mean()
                outs.tar_ll = ll[...,num_ctx:].mean()
            else:
                outs.ctx_ll = ll[...,:num_ctx]
                outs.tar_ll = ll[...,num_ctx:]

        return outs