import numpy


def generate_noise(rng, type, shape, params=None):
    """
    Generate noise with specified distribution type, ensuring all have variance = 1.

    Args:
        rng: Random number generator
        type: String specifying distribution type ("normal", "uniform", or "laplace")
        shape: Shape of the noise array to generate

    Returns:
        Noise array with variance 1 and shape (n, 1)
    """
    if type == "normal":
        # Normal distribution with mean=0, variance=1
        if params is None:
            # Default to standard normal if no parameters provided
            params = {"mean": 0, "std": 1}
        print(params)
        noise = rng.normal(params["mean"], params["std"], size=shape)
    elif type == "uniform":
        # Uniform distribution on [-√3, √3] has variance 1
        # For a uniform distribution on [a, b], variance = (b-a)²/12
        # Setting (b-a)²/12 = 1 and solving for symmetric bounds gives a = -√3, b = √3
        if params is not None and "std" in params:
            std = params["std"]
            mean = params.get("mean", 0)
            params = {
                "low": -std * numpy.sqrt(3) + mean,
                "high": std * numpy.sqrt(3) + mean,
            }
        if params is None:
            # Default to uniform on [-√3, √3] if no parameters provided
            params = {"low": -numpy.sqrt(3), "high": numpy.sqrt(3)}
        print(params)
        noise = rng.uniform(params["low"], params["high"], size=shape)
    elif type == "laplace":
        # Laplace distribution with mean=0, variance=1
        # For Laplace with scale=b, variance = 2b²
        # Setting 2b² = 1 gives b = 1/√2
        noise = rng.laplace(0, 1 / numpy.sqrt(2), size=shape)
    elif type == "normal_mixture":
        # Normal mixture model (1D only for now)
        mixture_params = [
            {"weight": 0.5, "mean": -1.5, "std": 0.5},
            {"weight": 0.5, "mean": 1.5, "std": 0.5},
        ]

        # Normalize weights to sum to 1
        weights = numpy.array([comp["weight"] for comp in mixture_params])
        weights = weights / numpy.sum(weights)

        # Generate component assignments for each sample
        components = rng.choice(len(mixture_params), size=shape, p=weights)

        # Generate samples from each component
        noise = numpy.zeros(shape)
        for i, comp in enumerate(mixture_params):
            # Get indices for this component
            idx = components == i
            count = numpy.sum(idx)
            if count > 0:
                # Generate noise from this component
                noise[idx] = rng.normal(comp["mean"], comp["std"], size=count)

        # Calculate empirical mean and standard deviation
        empirical_mean = numpy.mean(noise)
        empirical_std = numpy.std(noise)

        # Center and scale to ensure mean=0, variance=1
        noise = (noise - empirical_mean) / empirical_std
    else:
        raise ValueError(f"Unknown type of noise: {type}")

    return noise.reshape(-1, 1)
