from collections import defaultdict

# Define all edges (including dotted ones)
edges = [
    ("Plcg", "PIP3"), ("Plcg", "PIP2"), ("PIP3", "PIP2"),
    ("PKC", "PKA"), ("PKC", "Raf"), ("PKC", "Jnk"), ("PKC", "P38"),
    ("PKA", "Raf"), ("PKA", "Mek"), ("PKA", "Jnk"), ("PKA", "P38"),
    ("PKA", "Erk"), ("PKA", "Akt"), ("Raf", "Mek"), ("Mek", "Erk"),
    ("Erk", "Akt"), ("Plcg", "PKC"), ("PIP2", "PKC"), ("PIP3", "Akt"),
]

# Build adjacency and reverse‐adjacency
children = defaultdict(list)
parents  = defaultdict(list)
nodes    = set()

for u, v in edges:
    children[u].append(v)
    parents[v].append(u)
    nodes.update([u, v])

def compute_ancestors(node, parents, visited=None):
    if visited is None:
        visited = set()
    for p in parents[node]:
        if p not in visited:
            visited.add(p)
            compute_ancestors(p, parents, visited)
    return visited

ancestors = {n: compute_ancestors(n, parents) for n in nodes}

def find_paths(u, v, graph, path=None):
    if path is None:
        path = [u]
    else:
        path = path + [u]
    if u == v:
        return [path]
    paths = []
    for w in graph[u]:
        if w not in path:
            for p in find_paths(w, v, graph, path):
                paths.append(p)
    return paths

# Direct edges without any confounder (no common ancestor)
direct_no_confound = [
    (u, v)
    for u, v in edges
    if not (ancestors[u] & ancestors[v])
]

# Shortest mediated path (>=1 mediator) without confounding
shortest_paths = {}
for u in nodes:
    for v in nodes:
        if u == v:
            continue
        if ancestors[u] & ancestors[v]:
            continue
        all_paths = find_paths(u, v, children)
        mediated = [p for p in all_paths if len(p) >= 3]
        if not mediated:
            continue
        best = min(mediated, key=lambda p: len(p) - 2)
        shortest_paths[(u, v)] = best

print("Direct edges without confounding:")
for u, v in sorted(direct_no_confound):
    print(f"(\"{u}\", \"{v}\")")

print("\nstart, end # mediators, mediator path")
for (u, v), path in sorted(shortest_paths.items(), key=lambda item: len(item[1]) - 2):
    mediators = path[1:-1]
    print(f"(\"{u}\", \"{v}\"), # {len(mediators)}: {' → '.join(mediators)}")

