import collections
import copy
import difflib
import re 
import string


import json
import math
import numpy as np
import pandas as pd

import config


def normalize_answer(s):
    """Lower text and remove punctuation, articles and extra whitespace."""
    def remove_articles(text):
        return re.sub(r'\b(a|an|the)\b', ' ', text)

    def white_space_fix(text):
        return ' '.join(text.split())

    def remove_punc(text):
        exclude = set(string.punctuation)
        return ''.join(ch for ch in text if ch not in exclude)

    def lower(text):
        return text.lower()

    return white_space_fix(remove_articles(remove_punc(lower(s))))


def em_score_fn(x, y):
    return int(set(x) == set(y))


def digit_to_word(number):
    # create a dictionary that maps each digit to its English word equivalent
    word_dict = {
        0: 'zero', 1: 'one', 2: 'two', 3: 'three', 4: 'four',
        5: 'five', 6: 'six', 7: 'seven', 8: 'eight', 9: 'nine'
    }

    tens_dict = {
        2: 'twenty', 3: 'thirty', 4: 'forty', 5: 'fifty',
        6: 'sixty', 7: 'seventy', 8: 'eighty', 9: 'ninety'
    }
    
    # if the input number is a single digit, return its English word equivalent
    if number in word_dict:
        return word_dict[number]
    
    # if the input number is a two-digit number, convert its tens and ones digits to English words
    elif number < 100:
        tens_digit = number // 10
        ones_digit = number % 10
        if tens_digit == 1:
            # for numbers between 10 and 19, use special English words
            special_dict = {
                10: 'ten', 11: 'eleven', 12: 'twelve', 13: 'thirteen', 14: 'fourteen',
                15: 'fifteen', 16: 'sixteen', 17: 'seventeen', 18: 'eighteen', 19: 'nineteen'
            }
            return special_dict[number]
        else:
            # for other two-digit numbers, use a combination of tens and ones digits
            if ones_digit == 0:
                return tens_dict[tens_digit]
            else:
                return tens_dict[tens_digit] + ' ' + word_dict[ones_digit]
    
    # if the input number is a three-digit number, convert its hundreds digit and the remaining two digits to English words
    elif number < 1000:
        hundreds_digit = number // 100
        remainder = number % 100
        if remainder == 0:
            # for numbers like 100, 200, 300, etc., use the word 'hundred'
            return word_dict[hundreds_digit] + ' hundred'
        else:
            # for other three-digit numbers, use a combination of hundreds, tens, and ones digits
            return word_dict[hundreds_digit] + ' hundred ' + digit_to_word(remainder)
    
    # if the input number is a four-digit number or larger, convert its thousands digit and the remaining digits to English words
    elif number < 1000000:
        thousands = number // 1000
        remainder = number % 1000
        if remainder == 0:
            # for numbers like 1000, 2000, 3000, etc., use the word 'thousand'
            return digit_to_word(thousands) + ' thousand'
        else:
            # for other four-digit numbers or larger, use a combination of thousands, hundreds, tens, and ones digits
            return digit_to_word(thousands) + ' thousand ' + digit_to_word(remainder)
    
    # if the input number is too large, raise an error
    else:
        return ' '.join([digit_to_word(int(num)) for num in str(number)])


def replace_numbers(text):
    #numbers = re.findall(r'^(\d+(?:[\.\,]\d{3})?)$', text)
    numbers = re.findall(r'\b\d+(?:,\d{3})*(?:\.\d+)?\b', text)
    numbers_wo_punctuation = [re.sub(r'[^\w\s]','',number) for number in numbers]
    words = list(map(digit_to_word, map(int, numbers_wo_punctuation)))
    number_to_word = dict(zip(numbers, words))

    for number, word in number_to_word.items():
        text = text.replace(number, word)
    return text


def convert_to_normal_dict(dic):
    return {k: dict(v) for k, v in dic.items()}


def flat_dict_of_dict_of_int(dic):
    flat_list = [value for inner_dict in dic.values() for value in inner_dict.values()]
    return flat_list


def flat_dict_of_dict(dic):
    ret = {}
    for inner_dic in dic.values():
        ret.update(inner_dic)
    return ret


def group(df, cond):
    key_to_subdf = {}
    for idx, g in df.groupby(cond):
        key_to_subdf[idx] = g
    return key_to_subdf

    
def safe_div(a, b):
    if b == 0:
        return 0
    else:
        return a / b


def partial_true_positive_for_freeform_values(list1, list2):
    # list1: predicted values.
    # list2: gold values.
    list1 = list(map(normalize_answer, list1))
    list2 = list(map(normalize_answer, list2))
    result = []
    for string1 in list1:
        best_match = ("", 0)
        string1 = string1.split(' ')
        for string2 in list2:
            string2 = string2.split(' ')
            ratio = difflib.SequenceMatcher(None, string1, string2).ratio()
            if ratio > best_match[1]:
                best_match = (' '.join(string2), ratio)

        # Not matched anything in the reference (list2).
        if best_match[1] == 0:
            continue

        if list2:
            result.append(best_match[1])
            if best_match[0] in list2:
                list2.remove(best_match[0])  # Each string in list2 can only be used once
    num_tp = sum(result)
    num_match = len(result)
    return num_tp, num_match


def true_positive_for_categorical_values(preds, golds):
    preds = list(map(normalize_answer, preds))
    golds = list(map(normalize_answer, golds))
    num_tp = 0
    for pred in preds:
        if pred in golds:
            num_tp += 1
            golds.remove(pred)

    # We return 2 num_tp here because we use the unified function 
    # `true_positive_fn` to compute both categorical and freeform values.
    return num_tp, num_tp


def true_positive_fn(slot_type):
    if slot_type == 'categorical':
        return true_positive_for_categorical_values
    elif slot_type == 'extractive':
        return partial_true_positive_for_freeform_values
    else:
        raise ValueError(f'Unknown slot type: {slot_type}')


def merge_turn_fn(row):
    ems = row['em'].tolist()
    precisions = row['precision'].tolist()
    recalls = row['recall'].tolist()
    valid_precisions = row['valid_precision'].tolist()
    valid_recalls = row['valid_recall'].tolist()

    precisions = [p for p, v in zip(precisions, valid_precisions) if v]
    recalls = [r for r, v in zip(recalls, valid_recalls) if v]

    em = np.prod(ems) if ems else 0
    precision = np.mean(precisions) if precisions else 0
    recall = np.mean(recalls) if recalls else 0

    return pd.Series({
        'em': em,
        'precision': precision,
        'recall': recall,
        'valid_precision': np.any(valid_precisions),
        'valid_recall': np.any(valid_recalls),
    })


def merge_dialog_fn(row):
    ems = row['em'].tolist()
    precisions = row['precision'].tolist()
    recalls = row['recall'].tolist()
    valid_precisions = row['valid_precision'].tolist()
    valid_recalls = row['valid_recall'].tolist()

    precisions = [p for p, v in zip(precisions, valid_precisions) if v]
    recalls = [r for r, v in zip(recalls, valid_recalls) if v]

    em = np.mean(ems) if ems else 0
    precision = np.mean(precisions) if precisions else 0
    recall = np.mean(recalls) if recalls else 0

    return pd.Series({
        'em': em,
        'precision': precision,
        'recall': recall,
        'valid_precision': np.any(valid_precisions),
        'valid_recall': np.any(valid_recalls),
    })


def group(df, cond):
    key_to_subdf = {}
    for idx, g in df.groupby(cond):
        key_to_subdf[idx] = g
    return key_to_subdf


def safe_div(a, b):
    if b == 0:
        return 0
    else:
        return a / b


def get_value_em_f1(ref_df, pred_df, grouping_cond, avg_by='turn_idx'):
    ref_key_to_subdf = group(ref_df, grouping_cond)
    pred_key_to_subdf = group(pred_df, grouping_cond)
    
    all_keys = set(ref_key_to_subdf.keys()) | set(pred_key_to_subdf.keys())
    dialogue_id_idx = grouping_cond.index('dialogue_id')
    all_keys = sorted(all_keys, key=lambda x: x[dialogue_id_idx])

    # Each row contain EM, precision, recall, f1 for a condition.
    rows = []
    for idx, key in enumerate(all_keys):
        ref_subdf = ref_key_to_subdf.get(key, None)
        pred_subdf = pred_key_to_subdf.get(key, None)
        dialogue_id_idx = grouping_cond.index('dialogue_id')
        turn_idx_idx = grouping_cond.index('turn_idx')

        if avg_by == 'referent':
            referent_idx = grouping_cond.index('referent')
    
        em, total_num_tp, total_num_fp, total_num_fn = 0, 0, 0, 0

        if ref_subdf is not None and pred_subdf is not None:
            # Prediction something in the turns with at least one annotations.

            # Gets all slot types for the turn.
            slot_types = (set(ref_subdf['slot_type'].tolist()) 
                          | set(pred_subdf['slot_type'].tolist()))

            # Compute true positives, false positives, and false negatives 
            # for `categorical` and `extractive` slot types, respectively.
            for slot_type in slot_types:
                slot_type_ref_subdf = \
                    ref_subdf[ref_subdf['slot_type'] == slot_type]
                slot_type_pred_subdf = \
                    pred_subdf[pred_subdf['slot_type'] == slot_type]
                ref_values = sum(slot_type_ref_subdf['values'].tolist(), [])
                pred_values = sum(slot_type_pred_subdf['values'].tolist(), [])

                #assert ref_values or pred_values
                if not (ref_values or pred_values):
                    continue

                # We might predict categorical slots but the reference only has 
                # extractive slots.
                if not ref_values or not pred_values:
                    total_num_fp += len(pred_values)
                    total_num_fn += len(ref_values)
                    continue

                em = em_score_fn(pred_values, ref_values)
                num_tp, num_match = \
                    true_positive_fn(slot_type)(pred_values, ref_values)

                total_num_tp += num_tp
                # FN = num reference - num match.
                total_num_fn += (len(ref_values) - num_match)
                # FP = num predict - num match.
                total_num_fp += (len(pred_values) - num_match)

        # Underprediction.
        elif ref_subdf is not None and pred_subdf is None:
            refs = sum(ref_subdf['values'].tolist(), [])
            total_num_fn += len(refs)
            
        # Overprediction.
        elif ref_subdf is None and pred_subdf is not None:
            preds = sum(pred_subdf['values'].tolist(), [])
            total_num_fp += (len(preds))

        else:
            continue

        valid_precision = (total_num_tp + total_num_fp) > 0
        valid_recall = (total_num_tp + total_num_fn) > 0
        precision = safe_div(total_num_tp, total_num_tp + total_num_fp)
        recall = safe_div(total_num_tp, total_num_tp + total_num_fn)

        row = {
            'dialogue_id': key[dialogue_id_idx],
            'turn_idx': key[turn_idx_idx],
            'em': em,
            'precision': precision,
            'recall': recall,
            'valid_precision': valid_precision,
            'valid_recall': valid_recall,
        }
        if avg_by == 'referent':
            assert 'referent' in grouping_cond
            row['referent'] = key[referent_idx]
        rows.append(row)

    df = pd.DataFrame.from_dict(rows, orient='columns')

    df = df.groupby(['dialogue_id', avg_by]).apply(merge_turn_fn).reset_index()
    df = df.groupby(['dialogue_id']).apply(merge_dialog_fn).reset_index()
    #df = df[['em', 'precision', 'recall', 'f1']]
    #df = df[['em', 'precision', 'recall']]
    result = {}
    result['EM'] = df['em'].mean()
    result['Precision'] = p = (df['precision'] * df['valid_precision']).sum() / df['valid_precision'].sum()
    result['Recall'] = r = (df['recall'] * df['valid_recall']).sum() / df['valid_recall'].sum()
    result['F1'] = safe_div(2 * p * r, p + r)
    return result


def get_referent_or_slot_em_f1(ref_df, pred_df, grouping_cond, target_col):
    ref_key_to_subdf = group(ref_df, grouping_cond)
    pred_key_to_subdf = group(pred_df, grouping_cond)
    
    all_keys = set(ref_key_to_subdf.keys()) | set(pred_key_to_subdf.keys())
    dialogue_id_idx = grouping_cond.index('dialogue_id')
    all_keys = sorted(all_keys, key=lambda x: x[dialogue_id_idx])
    rows = []
    for idx, key in enumerate(all_keys):
        ref_subdf = ref_key_to_subdf.get(key, None)
        pred_subdf = pred_key_to_subdf.get(key, None)
        dialogue_id_idx = grouping_cond.index('dialogue_id')
        turn_idx_idx = grouping_cond.index('turn_idx')

        # Because we compute referent or slot. They must be categorical.
        slot_type = 'categorical'

        em, total_num_tp, total_num_fp, total_num_fn = 0, 0, 0, 0
        if ref_subdf is not None and pred_subdf is not None:
            ref_targets = ref_subdf[target_col].tolist()
            pred_targets = pred_subdf[target_col].tolist()
            if not ref_targets or not pred_targets:
                total_num_fp += len(pred_targets)
                total_num_fn += len(ref_targets)
                continue

            em = em_score_fn(pred_targets, ref_targets)
            num_tp, num_match = \
                true_positive_fn(slot_type)(pred_targets, ref_targets)

            total_num_tp += num_tp
            # FN = num reference - num match.
            total_num_fn += (len(ref_targets) - num_match)
            # FP = num predict - num match.
            total_num_fp += (len(pred_targets) - num_match)

        # Underprediction.
        elif ref_subdf is not None and pred_subdf is None:
            ref_targets = ref_subdf[target_col].tolist()
            total_num_fn += len(ref_targets)
            
        # Overprediction.
        elif ref_subdf is None and pred_subdf is not None:
            pred_targets = pred_subdf[target_col].tolist()
            total_num_fp += (len(pred_targets))

        else:
            continue

        valid_precision = (total_num_tp + total_num_fp) > 0
        valid_recall = (total_num_tp + total_num_fn) > 0
        precision = safe_div(total_num_tp, total_num_tp + total_num_fp)
        recall = safe_div(total_num_tp, total_num_tp + total_num_fn)

        row = {
            'dialogue_id': key[dialogue_id_idx],
            'turn_idx': key[turn_idx_idx],
            'em': em,
            'precision': precision,
            'recall': recall,
            'valid_precision': valid_precision,
            'valid_recall': valid_recall,
        }
        rows.append(row)

    df = pd.DataFrame.from_dict(rows, orient='columns')
    df = df.groupby(['dialogue_id', 'turn_idx']).apply(merge_turn_fn).reset_index()
    df = df.groupby(['dialogue_id']).apply(merge_dialog_fn).reset_index()

    result = {}
    result['EM'] = df['em'].mean()
    result['Precision'] = p = (df['precision'] * df['valid_precision']).sum() / df['valid_precision'].sum()
    result['Recall'] = r = (df['recall'] * df['valid_recall']).sum() / df['valid_recall'].sum()
    result['F1'] = safe_div(2 * p * r, p + r)
    return result


def get_ref_tuples(ref_path, label_col):
    with open(ref_path, 'r') as f:
        refs = json.load(f)

    ref_tuples = []
    for ref in refs:
        dialogue_id = ref['dialogue_id']
        for turn in ref['dialogue']:
            turn_idx = turn['turn_idx']
            ref_label = turn[label_col]
            for domain_referent_slot, values in ref_label.items():
                domain, referent, slot = domain_referent_slot.split('-')
                domain_slot = f'{domain}-{slot}'
                is_categorical = \
                    config.domain_slot_to_is_categorical.get(domain_slot, None)
                slot_type = 'categorical' if is_categorical else 'extractive'
                ref_tuples.append({
                    'dialogue_id': dialogue_id,
                    'turn_idx': turn_idx,
                    'domain': domain,
                    'referent': referent,
                    'slot': slot,
                    'slot_type': slot_type,
                    'values': values,
                })
    return ref_tuples

def get_pred_tuples(examples, preds):
    pred_tuples = []
    for ex, pred in zip(examples, preds):
        slots_referent_values_list = pred.split(config.slot_sep)
        for slot_referent_values_list in slots_referent_values_list:
            if config.slot_referent_value_sep not in slot_referent_values_list:
                continue
            if len(slot_referent_values_list.split(config.slot_referent_value_sep)) != 2:
                continue
            slot, referent_values_list = slot_referent_values_list.split(
                config.slot_referent_value_sep)
            slot = slot.strip()

            if slot.lower() == 'NONE'.lower():
                continue

            # Damage Part [srv] Front [rv] Caller [cv] Back [rv] Other Driver
            # referent_values_list: Front [rv] Caller [cv] Back [rv] Other Driver
            # referent_values_list: ["Front [rv] Caller", "Back [rv] Other Driver"]
            referent_values_list = referent_values_list.split(config.cross_referent_sep)
            for referent_values in referent_values_list:
                if config.referent_value_sep not in referent_values:
                    continue
                if len(referent_values.split(config.referent_value_sep)) != 2:
                    continue
                values, referent = referent_values.split(config.referent_value_sep)
                values = values.split(config.value_sep)
                referent = referent.strip()

                values = list(map(lambda x: x.strip(), values))

                if set(values) == {'NONE'}:
                    continue

                domain = ex['domain']
                domain_slot = f'{domain}-{slot}'
                is_categorical = \
                    config.domain_slot_to_is_categorical.get(domain_slot, None)
                slot_type = 'categorical' if is_categorical else 'extractive'

                pred_tuples.append({
                    'dialogue_id': ex['dialogue_id'],
                    'turn_idx': ex['turn_idx'],
                    'domain': domain,
                    'referent': referent,
                    'slot': slot,
                    'slot_type': slot_type,
                    'values': values,
                })
    return pred_tuples


def get_result(ref_df, pred_df):
    """Compute the result for all dialogs."""

    if len(pred_df) == 0:
        return {
            'EM': 0,
            'Precision': 0,
            'Recall': 0,
            'F1': 0,

            'R-EM': 0,
            'R-Precision': 0,
            'R-Recall': 0,
            'R-F1': 0,

            'S-EM': 0,
            'S-Precision': 0,
            'S-Recall': 0,
            'S-F1': 0,

            'RS-EM': 0,
            'RS-Precision': 0,
            'RS-Recall': 0,
            'RS-F1': 0,

            'SV-EM': 0,
            'SV-Precision': 0,
            'SV-Recall': 0,
            'SV-F1': 0,

            'v2-EM': 0,
            'v2-Precision': 0,
            'v2-Recall': 0,
            'v2-F1': 0,
        }

    result = {}

    # TLB-F1 or DST-F1.
    grouping_cond = ['dialogue_id', 'turn_idx', 'referent']
    res = get_value_em_f1(ref_df, pred_df, grouping_cond)
    result.update(res)

    # R-F1.
    grouping_cond = ['dialogue_id', 'turn_idx']
    res = get_referent_or_slot_em_f1(ref_df, pred_df, grouping_cond, 'referent')
    res = {f'R-{k}': v for k, v in res.items()}
    result.update(res)

    ### S-F1
    grouping_cond = ['dialogue_id', 'turn_idx']
    res = get_referent_or_slot_em_f1(ref_df, pred_df, grouping_cond, 'slot')
    res = {f'S-{k}': v for k, v in res.items()}
    result.update(res)

    # RS-F1.
    grouping_cond = ['dialogue_id', 'turn_idx', 'referent']
    res = get_referent_or_slot_em_f1(ref_df, pred_df, grouping_cond, 'slot')
    res = {f'RS-{k}': v for k, v in res.items()}
    result.update(res)

    # SV-F1.
    grouping_cond = ['dialogue_id', 'turn_idx']
    res = get_value_em_f1(ref_df, pred_df, grouping_cond)
    res = {f'SV-{k}': v for k, v in res.items()}
    result.update(res)

    # TLB-F1 or DST-F1 (version 2)
    #grouping_cond = ['dialogue_id', 'turn_idx', 'referent']
    #res = get_value_em_f1(ref_df, pred_df, grouping_cond, avg_by='referent')
    #res = {f'v2-{k}': v for k, v in res.items()}
    #result.update(res)

    return result

def get_target_turn_idxs(ref_path):
    with open(ref_path, 'r') as f:
        refs = json.load(f)

    ref_tuples = []
    dialogue_id_to_target_turn_idxs = collections.defaultdict(list)
    for ref in refs:
        dialogue_id = ref['dialogue_id']
        num_turns = len(ref['dialogue'])

        target_turn_idxs = []
        for dst_percentile in range(1, 5):
            target_idx = math.ceil(num_turns * 0.25 * dst_percentile)
            target_idx = min(target_idx, num_turns - 1)
            assert target_idx < num_turns, f'{target_idx = } {num_turns = }'
            target_turn_idxs.append(target_idx)
        dialogue_id_to_target_turn_idxs[dialogue_id] = target_turn_idxs
    return dialogue_id_to_target_turn_idxs


def get_dialogue_id_to_num_turns(ref_path):
    with open(ref_path, 'r') as f:
        refs = json.load(f)

    dialogue_id_to_num_turns = {}
    for ref in refs:
        dialogue_id = ref['dialogue_id']
        num_turns = len(ref['dialogue'])
        dialogue_id_to_num_turns[dialogue_id] = num_turns
    return dialogue_id_to_num_turns


def naive_accumulation(tuples, dialogue_id_to_num_turns):
    if not tuples:
        return []

    # group tuples by dialogue_id and turn_idx.
    dialogue_id_to_turn_idx_to_tuples = collections.defaultdict(
        lambda: collections.defaultdict(list))
    for tup in tuples:
        dialogue_id = tup['dialogue_id']
        turn_idx = int(tup['turn_idx'])
        dialogue_id_to_turn_idx_to_tuples[dialogue_id][turn_idx].append(tup)

    all_cumulative_tuples = []
    for dialogue_id, turn_idx_to_tuples in dialogue_id_to_turn_idx_to_tuples.items():
        cumulative_domain_referent_slot_to_values = {}

        for turn_idx in range(dialogue_id_to_num_turns[dialogue_id]):
            tuples = turn_idx_to_tuples.get(turn_idx, [])
            for tup in tuples:
                domain_referent_slot = f'{tup["domain"]}-{tup["referent"]}-{tup["slot"]}'
                values = tup['values']
                if domain_referent_slot in cumulative_domain_referent_slot_to_values:
                    values = cumulative_domain_referent_slot_to_values[domain_referent_slot] + values
                cumulative_domain_referent_slot_to_values[domain_referent_slot] = values

            for domain_referent_slot, values in cumulative_domain_referent_slot_to_values.items():
                domain, referent, slot = domain_referent_slot.split('-')
                domain_slot = f'{domain}-{slot}'
                is_categorical = \
                    config.domain_slot_to_is_categorical.get(domain_slot, None)
                slot_type = 'categorical' if is_categorical else 'extractive'
                all_cumulative_tuples.append({
                    'dialogue_id': dialogue_id,
                    'turn_idx': turn_idx,
                    'domain': domain,
                    'referent': referent,
                    'slot': slot,
                    'slot_type': slot_type,
                    'values': copy.deepcopy(values),
                })
                
    return all_cumulative_tuples


def aggregate_frames_state_change(
    json_basename_to_dialogue_id_to_dialog,
    target_turn_id=None,
    skip_op_if_error=False):

    for basename, dialog_id_to_dialog in \
        json_basename_to_dialogue_id_to_dialog.items():
        for dialog_id, dialog in dialog_id_to_dialog.items():
            touch_first_user_turn = False
            prev_frames = None
            if target_turn_id is not None:
                curr_target_turn_id = target_turn_id
                if dialog['turns'][0]['speaker'] == 'USER':
                    curr_target_turn_id = target_turn_id - 1
            for turn_idx, turn in enumerate(dialog['turns']):
                if turn['speaker'] != 'USER':
                    continue

                if not touch_first_user_turn:
                    touch_first_user_turn = True
                    continue

                if prev_frames is None:
                    prev_frames = copy.deepcopy(turn['frames'])
                    continue

                assert len(prev_frames) == len(turn['frames'])

                if curr_target_turn_id is not None:
                    if turn_idx < curr_target_turn_id:
                        prev_frames = copy.deepcopy(turn['frames'])
                        continue
                    assert turn_idx == curr_target_turn_id

                for frame_idx, (prev_frame, curr_frame) in \
                    enumerate(zip(prev_frames, turn['frames'])):
                    assert prev_frame['service'] == curr_frame['service']

                    curr_referent_slot_names = set(curr_frame['state']['slot_values'].keys()) 
                    prev_referent_slot_names = set(prev_frame['state']['slot_values'].keys())
                    referent_slot_names = curr_referent_slot_names.union(prev_referent_slot_names)

                    for referent_slot_name in referent_slot_names:
                        curr_values = curr_frame['state']['slot_values'].get(
                            referent_slot_name, [])
                        prev_values = prev_frame['state']['slot_values'].get(
                            referent_slot_name, [])
                        
                        #print(f'{turn_idx = } {referent_slot_name = } {curr_values = }')
                        updated_curr_values = copy.deepcopy(prev_values)
                        if not curr_values:
                            curr_frame['state']['slot_values'][referent_slot_name] = \
                                updated_curr_values
                            continue

                        concat_values = []
                        #print(f'{referent_slot_name = } ||| {curr_values = }')
                        for curr_value in curr_values:
                            # No operation, just add the value.
                            if config.value_op_sep not in curr_value:
                                updated_curr_values.append(curr_value)
                                continue

                            tmp = curr_value.split(config.value_op_sep)
                            curr_value, curr_op = tmp[0], tmp[1]

                            # The operation is either concat or delete.
                            # Remove the value from the previous values.
                            if not skip_op_if_error:
                                assert curr_value in prev_values

                            # We do not consider same operation in CB 
                            # because it already appear in the previous state.
                            if curr_op == config.op_to_token['same']:
                                continue
                                #if curr_value in updated_curr_values:
                                #    continue
                                #else:
                                #    print(f'[same] op: {curr_value = } is not in {updated_curr_values = }')
                                #updated_curr_values.append(curr_value)
                            elif curr_op == config.op_to_token['delete']:
                                if curr_value in updated_curr_values:
                                    updated_curr_values.remove(curr_value)
                                else:
                                    print(f'[delete] op: {curr_value = } is not in {updated_curr_values = }')
                            elif curr_op == config.op_to_token['concat']:
                                concat_values.append(curr_value)  
                            elif skip_op_if_error:
                                continue
                            else:
                                raise ValueError(f'The op is not available! {curr_op}')

                        if concat_values:
                            if not skip_op_if_error:
                                assert len(concat_values) > 1
                            for concat_value in concat_values:
                                if concat_value in updated_curr_values:
                                    updated_curr_values.remove(concat_value)
                            updated_curr_values.append(' '.join(concat_values))
                            
                        curr_frame['state']['slot_values'][referent_slot_name] = \
                            updated_curr_values
                        
                prev_frames = copy.deepcopy(turn['frames'])
    #return json_basename_to_dialogue_id_to_dialog
    