import torch
import torch.nn as nn
import torch.nn.functional as F
from layers.Transformer_EncDec import Encoder, EncoderLayer
from layers.SelfAttention_Family import FullAttention, AttentionLayer
from layers.Embed import DataEmbedding_inverted
import numpy as np

torch.autograd.set_detect_anomaly(True)


class Model(nn.Module):
    """
    Paper link: https://arxiv.org/abs/2310.06625
    """

    def __init__(self, configs):
        super(Model, self).__init__()
        self.task_name = configs.task_name
        self.seq_len = configs.seq_len
        self.pred_len = configs.pred_len
        # Embedding
        self.enc_embedding = DataEmbedding_inverted(configs.seq_len, configs.d_model, configs.embed, configs.freq,
                                                    configs.dropout)
        # Encoder
        self.encoder = Encoder(
            [
                EncoderLayer(
                    AttentionLayer(
                        #output attention matrix
                        FullAttention(False, configs.factor, attention_dropout=configs.dropout,
                                      output_attention=True), configs.d_model, configs.n_heads),
                    configs.d_model,
                    configs.d_ff,
                    dropout=configs.dropout,
                    activation=configs.activation
                ) for l in range(configs.e_layers)
            ],
            norm_layer=torch.nn.LayerNorm(configs.d_model)
        )
        # Decoder
        if self.task_name == 'long_term_forecast' or self.task_name == 'short_term_forecast':
            self.projection = nn.Linear(configs.d_model, configs.pred_len, bias=True)
        if self.task_name == 'imputation':
            self.projection = nn.Linear(configs.d_model, configs.seq_len, bias=True)
        if self.task_name == 'anomaly_detection':
            self.projection = nn.Linear(configs.d_model, configs.seq_len, bias=True)
        if self.task_name == 'classification':
            self.act = F.gelu
            self.dropout = nn.Dropout(configs.dropout)
            print("configs.num_class: ", configs.num_classes)
            self.projection = nn.Linear(configs.d_model * configs.enc_in, configs.num_classes, bias=True)

    def forecast(self, x_enc, x_mark_enc, x_dec, x_mark_dec, return_attn = True):
        # Normalization from Non-stationary Transformer
        means = x_enc.mean(1, keepdim=True).detach()
        x_enc = x_enc - means
        stdev = torch.sqrt(torch.var(x_enc, dim=1, keepdim=True, unbiased=False) + 1e-5)
        x_enc = x_enc / stdev

        _, _, N = x_enc.shape

        # Embedding
     #   print("[DEBUG itransformer] x_enc.shape: ", x_enc.shape)
        enc_out = self.enc_embedding(x_enc, x_mark_enc)
     #   print("[DEBUG itransformer] enc embedding.shape: ", enc_out.shape)
        enc_out, attns, (keys,values) = self.encoder(enc_out, attn_mask=None)
        # print("[DEBUG itransformer] enc_out.shape after enc: ", enc_out.shape)
        # print("[DEBUG itransformer] attns.shape: ", len(attns))
        # print("[DEBUG itransformer] attns[0].shape: ", attns[-1].shape)
        # print("[DEBUG itransformer] keys.shape: ", keys.shape)
        # print("[DEBUG itransformer] values.shape: ", values.shape)
        dec_out = self.projection(enc_out).permute(0, 2, 1)[:, :, :N]
        # print("[DEBUG itransformer] self.projection(enc_out).shape: ", self.projection(enc_out).shape)
        # print("[DEBUG itransformer] self.projection(enc_out).permute(0, 2, 1).shape: ", self.projection(enc_out).permute(0, 2, 1).shape)

        # De-Normalization from Non-stationary Transformer
        dec_out = dec_out * (stdev[:, 0, :].unsqueeze(1).repeat(1, self.pred_len, 1))
        dec_out = dec_out + (means[:, 0, :].unsqueeze(1).repeat(1, self.pred_len, 1))

   #     print("[DEBUG itransformer] dec_out.shape after denormalization: ", dec_out.shape)
        return dec_out, attns, (keys,values) if return_attn else dec_out

    def imputation(self, x_enc, x_mark_enc, x_dec, x_mark_dec, mask, return_attn = True):
        # Normalization from Non-stationary Transformer
        means = x_enc.mean(1, keepdim=True).detach()
        x_enc = x_enc - means
        stdev = torch.sqrt(torch.var(x_enc, dim=1, keepdim=True, unbiased=False) + 1e-5)
        x_enc = x_enc / stdev

        _, L, N = x_enc.shape

        # Embedding
        enc_out = self.enc_embedding(x_enc, x_mark_enc)
        enc_out, attns, (keys,values) = self.encoder(enc_out, attn_mask=None)

        dec_out = self.projection(enc_out).permute(0, 2, 1)[:, :, :N]
        # De-Normalization from Non-stationary Transformer
        dec_out = dec_out * (stdev[:, 0, :].unsqueeze(1).repeat(1, L, 1))
        dec_out = dec_out + (means[:, 0, :].unsqueeze(1).repeat(1, L, 1))
        return dec_out, attns, (keys,values) if return_attn else dec_out

    def anomaly_detection(self, x_enc, return_attn = True):
        # Normalization from Non-stationary Transformer
        means = x_enc.mean(1, keepdim=True).detach()
        x_enc = x_enc - means
        stdev = torch.sqrt(torch.var(x_enc, dim=1, keepdim=True, unbiased=False) + 1e-5)
        x_enc = x_enc / stdev

        _, L, N = x_enc.shape

        # Embedding
        enc_out = self.enc_embedding(x_enc, None)
        enc_out, attns, (keys,values) = self.encoder(enc_out, attn_mask=None)

        dec_out = self.projection(enc_out).permute(0, 2, 1)[:, :, :N]
        # De-Normalization from Non-stationary Transformer
        dec_out = dec_out * (stdev[:, 0, :].unsqueeze(1).repeat(1, L, 1))
        dec_out = dec_out + (means[:, 0, :].unsqueeze(1).repeat(1, L, 1))
        return dec_out, attns, (keys,values) if return_attn else dec_out

    def classification(self, x_enc, x_mark_enc, return_attn = True):
        # Embedding
     #   print("[DEBUG itransformer] x_enc.shape: ", x_enc.shape)
        enc_out = self.enc_embedding(x_enc, None)
     #   print("[DEBUG itransformer] enc embedding.shape: ", enc_out.shape)
        enc_out, attns, (keys,values) = self.encoder(enc_out, attn_mask=None)
     #   print("[DEBUG itransformer] enc_out.shape after enc: ", enc_out.shape)
     #   print("keys.shape: ", keys.shape)
     #   print("values.shape: ", values.shape)

        # Output
        output = self.act(enc_out)  # the output transformer encoder/decoder embeddings don't include non-linearity
     #   print("[DEBUG itransformer] output.shape after act: ", output.shape)
        output = self.dropout(output)
     #   print("[DEBUG itransformer] output.shape after dropout: ", output.shape)
        output = output.reshape(output.shape[0], -1)  # (batch_size, c_in * d_model)
     #   print("[DEBUG itransformer] output.shape after reshape: ", output.shape)
         
        output = self.projection(output)  # (batch_size, num_classes)
    #    print("[DEBUG itransformer] output.shape after projection: ", output.shape)

        return output, attns, (keys,values) if return_attn else output

    def forward(self, x_enc, x_mark_enc, x_dec, x_mark_dec, mask=None, return_attn=True):
        if self.task_name == 'long_term_forecast' or self.task_name == 'short_term_forecast':
            out = self.forecast(x_enc, x_mark_enc, x_dec, x_mark_dec, return_attn=True)
            
            # return dec_out[:, -self.pred_len:, :]  # [B, L, D]
            if return_attn:
                dec_out, attns, (keys, values) = out
                # print("[DEBUG itransformer] keys shape: ", keys.shape)
                # print("[DEBUG itransformer] values shape: ", values.shape)
                # print("[DEBUG itransformer] dec_out shape: ", dec_out.shape)
                return dec_out[:, -self.pred_len:, :], attns, (keys, values)
            return out[:, -self.pred_len:, :]
        if self.task_name == 'imputation':
            dec_out = self.imputation(x_enc, x_mark_enc, x_dec, x_mark_dec, mask)
            if return_attn:
                output, attns, (keys, values) = dec_out
                return output, attns, (keys, values)
            return dec_out  
        if self.task_name == 'anomaly_detection':
            dec_out = self.anomaly_detection(x_enc)
            if return_attn:
                output, attns, (keys, values) = dec_out
                return output, attns, (keys, values)
            return dec_out  # [B, L, D]
        if self.task_name == 'classification':
            dec_out = self.classification(x_enc, x_mark_enc)
            if return_attn:
                output, attns, (keys, values) = dec_out
                return output, attns, (keys, values)
            return dec_out  # [B, N]
        return None
