import torch
import torch.nn as nn

from egnn.egnn_new import EGNN, GNN
from equivariant_diffusion.utils import remove_mean, remove_mean_with_mask


# TODO: remove stuff related to z_sigma since we're not using it.
class EGNN_encoder(nn.Module):
    def __init__(self, in_node_nf, context_node_nf, out_node_nf,
                 n_dims, hidden_nf=64, device='cpu',
                 act_fn=torch.nn.SiLU(), n_layers=4, attention=False,
                 tanh=False, mode='egnn_dynamics', norm_constant=1,
                 inv_sublayers=2, sin_embedding=False, normalization_factor=100, aggregation_method='sum',
                 include_atomic_numbers=False, n_extra_atomic_features=0, encode_h_indep=False):
        '''
        :param in_node_nf: Number of invariant features for input nodes.
        :param context_node_nf: Number of context features for input nodes used for conditional generation.
        :param out_node_nf: Number of latent features for invariant features.
        :param n_dims: Number of equivariant features, should be 3 in our case.
        '''
        super().__init__()

        include_atomic_numbers = int(include_atomic_numbers)
        # TODO: remove hard-coded number of formal charges=3
        num_classes = in_node_nf - include_atomic_numbers - 3 # one hot formal charges are always included

        self.n_extra_atomic_features = n_extra_atomic_features
        self.encode_h_indep = encode_h_indep

        # to differentiate between fingerprint conditioning and other
        self.context_node_nf = context_node_nf
        if context_node_nf == 1024:
            context_node_nf = 0

        if self.n_extra_atomic_features > 0:
            print('Using extra atom features!!')
            in_node_nf = n_extra_atomic_features

        if self.encode_h_indep:
            print('Using MLP to embed atom features independently!!')
            # embed atom features independently without EGNN
            self.node_mlp = nn.Sequential(
                nn.Linear(in_node_nf + context_node_nf, hidden_nf),
                act_fn,
                nn.Linear(hidden_nf, out_node_nf)
            ).to(device)
        else:
            self.node_mlp = None

        self.mode = mode
        self.n_layers = n_layers
        if n_layers > 0:
            if mode == 'egnn_dynamics':
                self.egnn = EGNN(
                    in_node_nf=in_node_nf + context_node_nf, out_node_nf=hidden_nf, 
                    in_edge_nf=1, hidden_nf=hidden_nf, device=device, act_fn=act_fn,
                    n_layers=n_layers, attention=attention, tanh=tanh, norm_constant=norm_constant,
                    inv_sublayers=inv_sublayers, sin_embedding=sin_embedding,
                    normalization_factor=normalization_factor,
                    aggregation_method=aggregation_method,
                    fingerprint_conditioning=(self.context_node_nf == 1024))
                self.in_node_nf = in_node_nf
            elif mode == 'gnn_dynamics':
                self.gnn = GNN(
                    in_node_nf=in_node_nf + context_node_nf + 3, out_node_nf=hidden_nf + 3, 
                    in_edge_nf=0, hidden_nf=hidden_nf, device=device,
                    act_fn=act_fn, n_layers=n_layers, attention=attention,
                    normalization_factor=normalization_factor, aggregation_method=aggregation_method)
            
            # TODO: just removed it now so that i can load but remove it later!!
            if self.node_mlp is None:
                self.final_mlp = nn.Sequential(
                    nn.Linear(hidden_nf, hidden_nf),
                    act_fn,
                    nn.Linear(hidden_nf, out_node_nf * 2 + 1)
                ).to(device)

        self.num_classes = num_classes
        self.include_atomic_numbers = include_atomic_numbers
        self.device = device
        self.n_dims = n_dims
        self._edges_dict = {}
        # self.condition_time = condition_time

        self.out_node_nf = out_node_nf

    def forward(self, t, xh, node_mask, edge_mask, context=None):
        raise NotImplementedError

    def wrap_forward(self, node_mask, edge_mask, context):
        def fwd(time, state):
            return self._forward(time, state, node_mask, edge_mask, context)
        return fwd

    def unwrap_forward(self):
        return self._forward

    def _forward(self, xh, node_mask, edge_mask, context):      
        bs, n_nodes, dims = xh.shape
        h_dims = dims - self.n_dims
        edges = self.get_adj_matrix(n_nodes, bs, self.device)
        edges = [x.to(self.device) for x in edges]
        node_mask = node_mask.view(bs*n_nodes, 1)
        edge_mask = edge_mask.view(bs*n_nodes*n_nodes, 1)
        xh = xh.view(bs*n_nodes, -1).clone() * node_mask
        x = xh[:, 0:self.n_dims].clone()
        if h_dims == 0:
            h = torch.ones(bs*n_nodes, 1).to(self.device)
        else:
            h = xh[:, self.n_dims:].clone()

        if self.context_node_nf != 0 and context is not None:
            # We're conditioning, awesome!
            context = context.view(bs*n_nodes, self.context_node_nf)
            # if we're conditioning on fingerprints, we will embed them inside the EGNN
            if self.context_node_nf != 1024:
                h = torch.cat([h, context], dim=1)
                context = None
        else:
            # this means the diffusion model is conditioned on context but not encoder / decoder
            context = None

        if self.n_layers > 0:
            if self.mode == 'egnn_dynamics':
                h_final, x_final = self.egnn(h, x, edges, node_mask=node_mask, edge_mask=edge_mask, context=context)
                vel = x_final * node_mask  # This masking operation is redundant but just in case
            elif self.mode == 'gnn_dynamics':
                xh = torch.cat([x, h], dim=1)
                output = self.gnn(xh, edges, node_mask=node_mask)
                vel = output[:, 0:3] * node_mask
                h_final = output[:, 3:]
            else:
                raise Exception("Wrong mode %s" % self.mode)
        else:
            h_final, x_final = h, x
            vel = x_final * node_mask

        vel = vel.view(bs, n_nodes, -1)

        if torch.any(torch.isnan(vel)):
            print('Warning: detected nan in encoder fwd, resetting EGNN output to zero.')
            vel = torch.zeros_like(vel)

        if node_mask is None:
            vel = remove_mean(vel)
        else:
            vel = remove_mean_with_mask(vel, node_mask.view(bs, n_nodes, 1))

        if self.node_mlp is not None:
            h_mean = self.embed_node_features(h, node_mask)
            h_mean = h_mean * node_mask if node_mask is not None else h_mean
            h_mean = h_mean.view(bs, n_nodes, -1)
            h_std = torch.ones((bs, n_nodes, self.out_node_nf))

            vel_mean = vel
            vel_std = torch.ones((bs, n_nodes, 1))

            return vel_mean, vel_std, h_mean, h_std

        h_final = self.final_mlp(h_final) # shape: (*, out_node_nf * 2 + 1)
        h_final = h_final * node_mask if node_mask is not None else h_final
        h_final = h_final.view(bs, n_nodes, -1)

        vel_mean = vel
        vel_std = h_final[:, :, :1].sum(dim=1, keepdim=True).expand(-1, n_nodes, -1)
        vel_std = torch.exp(0.5 * vel_std)

        h_mean = h_final[:, :, 1:1 + self.out_node_nf]
        h_std = torch.exp(0.5 * h_final[:, :, 1 + self.out_node_nf:])

        if torch.any(torch.isnan(vel_std)):
            print('Warning: detected nan in vel_std in encoder fwd, resetting to one.')
            vel_std = torch.ones_like(vel_std)
        
        if torch.any(torch.isnan(h_std)):
            print('Warning: detected nan in h_std in encoder fwd, resetting to one.')
            h_std = torch.ones_like(h_std)
        
        # Note: only vel_mean and h_mean are correctly masked
        # vel_std and h_std are not masked, but that's fine:

        # For calculating KL: vel_std will be squeezed to 1D
        # h_std will be masked

        # For sampling: both stds will be masked in reparameterization

        return vel_mean, vel_std, h_mean, h_std

    def embed_node_features(self, h, node_mask):
        bs = h.size(0)

        # The following code ensures that we only predict the nodes we care about
        # i.e. we ignore the nodes that were padded to all atoms in the batch so that they have the same number of nodes
        h_indices_non_zero = node_mask.bool().squeeze()
        h_processed = self.node_mlp(h[h_indices_non_zero])
        output_dim = h_processed.size(1)
        h_final_output = torch.zeros((bs, output_dim)).to(self.device)
        h_final_output[h_indices_non_zero] = h_processed
        h = h_final_output
        return h
    
    def get_adj_matrix(self, n_nodes, batch_size, device):
        if n_nodes in self._edges_dict:
            edges_dic_b = self._edges_dict[n_nodes]
            if batch_size in edges_dic_b:
                return edges_dic_b[batch_size]
            else:
                # get edges for a single sample
                rows, cols = [], []
                for batch_idx in range(batch_size):
                    for i in range(n_nodes):
                        for j in range(n_nodes):
                            rows.append(i + batch_idx * n_nodes)
                            cols.append(j + batch_idx * n_nodes)
                edges = [torch.LongTensor(rows).to(device),
                         torch.LongTensor(cols).to(device)]
                edges_dic_b[batch_size] = edges
                return edges
        else:
            self._edges_dict[n_nodes] = {}
            return self.get_adj_matrix(n_nodes, batch_size, device)
