import rdkit.Chem.AllChem as Chem
from rdkit.Chem.Pharm3D import EmbedLib
from rdkit.Chem import rdDistGeom as molDG


def set_3D_coords_rdkit(mol):
    """
    Creates a set of 3D coordinates for the input molecule, using RDKit's default algorithm,
    described here: https://www.rdkit.org/docs/GettingStartedInPython.html#working-with-3d-molecules.
    Modifies the molecule in place.

    Returns:
        the modified molecule with a set of 3D coordinates that can be obtained by calling mol.GetConformers()[0].GetPositions()
    """
    mol = Chem.AddHs(mol)
    conf_id = Chem.EmbedMolecule(mol, maxAttempts=1000)

    if conf_id == -1:
        # random-coordinate embedding will be done: 
        # instead of generating a distance matrix that satisfies the bounds and then embedding it in 3D, 
        # the atoms will be randomly placed in a box and then their positions will be minimized with the 
        # “distance geometry force field”.
        conf_id = Chem.EmbedMolecule(mol, maxAttempts=1000, useRandomCoords=True)

    if conf_id == -1:
        # removes the constraint that the chirality of specified stereocenters in the molecule is preserved in the conformers
        conf_id = Chem.EmbedMolecule(mol, maxAttempts=1000, enforceChirality=False)

    if conf_id == -1:
        # completely remove Stereochemistry
        Chem.RemoveStereochemistry(mol)
        conf_id = Chem.EmbedMolecule(mol, maxAttempts=1000)

    if conf_id == -1:
        # If nothing works, try this method. Similar approach but different package
        bm = molDG.GetMoleculeBoundsMatrix(mol)
        EmbedLib.EmbedMol(mol, bm)
        EmbedLib.OptimizeMol(mol, bm)


    # if conf_id == -1:
    #     print('Nothing worked. Trying with 10k iterations')
    #     Chem.RemoveStereochemistry(mol)
    #     conf_id = Chem.EmbedMolecule(mol, maxAttempts=10000, ETversion=2)
    # if conf_id == -1:
    #     print('Nothing worked. Trying with 10k iterations and random coordinates')
    #     conf_id = Chem.EmbedMolecule(mol, maxAttempts=10000, useRandomCoords=True)

    Chem.MMFFOptimizeMolecule(mol)
    #mol = Chem.RemoveHs(mol)
    return mol
