import os
import json
import regex as re
from openai import OpenAI
import pandas as pd
import sys


project_path = os.path.abspath(os.path.join(os.path.dirname(__file__), ".."))
print(f"Project path: {project_path}")
if project_path not in sys.path:
    sys.path.append(project_path)


def get_response_openai(history, model_name="gpt-4o"):
    """
    Get a response from GPT-4o using the OpenAI API and check token counts.

    Args:
        history: List of dictionaries with 'role' and 'content' keys

    Returns:
        The model's response text
    """
    # Check if OpenAI API key is set
    api_key = os.environ.get("OPENAI_API_KEY")
    if not api_key:
        raise ValueError(
            "OPENAI_API_KEY environment variable is not set. Please set your OpenAI API key."
        )

    try:
        client = OpenAI()

        response = client.chat.completions.create(
            model=model_name, messages=history, temperature=0, max_tokens=256
        )

        # Print input tokens, output tokens, and total token count
        print(f"Input tokens: {response.usage.prompt_tokens}")
        print(f"Output tokens: {response.usage.completion_tokens}")
        print(f"Total tokens used: {response.usage.total_tokens}")

        return response.choices[0].message.content.strip()
    except Exception as e:
        print(f"Error calling OpenAI API: {e}")
        return "noop"  # Default to noop if API call fails


system_prompt = """You are an expert evaluator analyzing agent behavior in a survival crafting game called Crafter.
You will be given a **reduced version** of the agent's trajectory, focusing only on segments where the agent's status and inventory have been changed.

Your output **must** be a JSON object with the following two fields:

{
 "behavior_analysis": "Describe what the agent has accomplished so far. Mention specific achievements (e.g., placing a table) and what those imply about the agent’s current progression or intent.",
 "next_iteration_advice": "Suggest a specific, actionable next step for the agent that would likely improve its capabilities or unlock new achievements. The advice should always start with 'Focus on...' and be written as a single sentence. It should reflect the agent's current progress and identify a meaningful, skill-expanding next goal."
}

Guidelines:
- Do not include any explanation or text outside of the JSON block.
- Do not list step-by-step logs or inventory diffs — summarize behavior abstractly.
- Consider the agent's current resources and abilities to suggest realistic next goals.
- Make sure the 'next_iteration_advice' sentence is specific and skill-oriented, not vague.

Note: This is a **partial trajectory**, so analyze only what is visible.
"""


def parse_inventory_dict(inv_str):
    try:
        return json.loads(inv_str)
    except Exception:
        return {}


def diff_dict(prev, curr):
    diff = {}
    for k in set(prev) | set(curr):
        delta = curr.get(k, 0) - prev.get(k, 0)
        if delta:
            diff[k] = f"{'+' if delta>0 else '-'}{abs(delta)}"
    return diff


def extract_status_field(obs, key):
    for line in obs.splitlines():
        if line.strip().startswith(f"- {key}:"):
            return int(line.split(":")[-1].split("/")[0].strip())
    return None


def extract_inventory(obs):
    if "Your inventory:" not in obs:
        return "{}"
    inv_block = obs.split("Your inventory:")[-1].split("You see:")[0]
    inv = {}
    for line in inv_block.splitlines():
        m = re.match(r"- (\w+): (\d+)", line.strip())
        if m:
            inv[m.group(1)] = int(m.group(2))
    return json.dumps(inv)


def build_user_prompt(df):
    df = df.reset_index(drop=True)

    # Precompute inventory & status
    inventories = []
    statuses = []
    for row in df.itertuples():
        obs = row.observation_goal
        inventories.append(parse_inventory_dict(extract_inventory(obs)))
        statuses.append(
            {
                k: extract_status_field(obs, k)
                for k in ("health", "food", "drink", "energy")
            }
        )

    prompt_txt = ""
    episode_idx = 1

    prompt_txt += f"[Episode {episode_idx}]\n"

    for i in range(len(df) - 1):
        row = df.iloc[i]
        next_row = df.iloc[i + 1]

        step = row.step
        next_step = next_row.step

        # Check for episode boundary
        if next_step < step:
            episode_idx += 1
            prompt_txt += f"\n[Episode {episode_idx}]\n"
            prev_step = next_step
            continue  # Skip current step (we don’t know its effect)

        # Compute change caused by this action
        inv_delta = diff_dict(inventories[i], inventories[i + 1])
        stat_delta = diff_dict(statuses[i], statuses[i + 1])

        if not inv_delta and not stat_delta:
            continue
        if any("-" in v for (k, v) in stat_delta.items()):
            continue

        inv_str = json.dumps(inv_delta) if inv_delta else "None"
        stat_str = json.dumps(stat_delta) if stat_delta else "None"

        prompt_txt += (
            f"Step {step}:\n"
            f"- Action: {row.action}\n"
            f"- Inventory Change: {inv_str}\n"
            f"- Status Change: {stat_str}\n\n"
        )

    return (
        "Below is a filtered trajectory from a Crafter agent, showing only steps where "
        "the action caused a change in inventory or status. Each new episode is marked accordingly.\n\n"
        + prompt_txt
    )


df = pd.read_csv("example_file.csv")
user_prompt = build_user_prompt(df)

# print(user_prompt)

prompt = [
    {"role": "system", "content": system_prompt},
    {"role": "user", "content": user_prompt},
]

# Call the function
result = get_response_openai(prompt)
print(result)
