import numpy as np
import torch
from .base_model import BaseModel
from . import networks
from .patchnce import PatchNCELoss
import util.util as util
# import 

def sample_SB(G, X0, T, z_dim, tau):
    G.eval()
    Xts = []
    Xt_1s = []
    for t in range(T):
        Xt   = X0 if (t == 0) else (1-1/(T-t+1)) * Xt + (1/(T-t+1)) * Xt_1 + np.sqrt((1/T)*(1-1/(T-t+1))) * np.sqrt(tau) * torch.randn_like(Xt).to(X0.device)
        time = t * torch.ones(size=[X0.shape[0]]).to(X0.device)
        z    = torch.randn(size=[X0.shape[0],z_dim]).to(X0.device)
        Xt_1 = G(Xt, time, z).detach()
        Xts.append(Xt[:,None].detach().clone().cpu())
        Xt_1s.append(Xt_1[:,None].detach().clone().cpu())
    return torch.cat(Xts, dim=1), torch.cat(Xt_1s, dim=1)

class CUTModel(BaseModel):
    """ This class implements CUT and FastCUT model, described in the paper
    Contrastive Learning for Unpaired Image-to-Image Translation
    Taesung Park, Alexei A. Efros, Richard Zhang, Jun-Yan Zhu
    ECCV, 2020

    The code borrows heavily from the PyTorch implementation of CycleGAN
    https://github.com/junyanz/pytorch-CycleGAN-and-pix2pix
    """
    @staticmethod
    def modify_commandline_options(parser, is_train=True):
        """  Configures options specific for CUT model
        """
        parser.add_argument('--CUT_mode', type=str, default="CUT", choices='(CUT, cut, FastCUT, fastcut)')

        parser.add_argument('--lambda_GAN', type=float, default=1.0, help='weight for GAN loss：GAN(G(X))')
        parser.add_argument('--lambda_NCE', type=float, default=1.0, help='weight for NCE loss: NCE(G(X), X)')
        parser.add_argument('--lambda_entropy', type=float, default=0.1, help='weight for NCE loss: NCE(G(X), X)')
        parser.add_argument('--nce_idt', type=util.str2bool, nargs='?', const=True, default=False, help='use NCE loss for identity mapping: NCE(G(Y), Y))')
        parser.add_argument('--nce_layers', type=str, default='0,4,8,12,16', help='compute NCE loss on which layers')
        parser.add_argument('--nce_includes_all_negatives_from_minibatch',
                            type=util.str2bool, nargs='?', const=True, default=False,
                            help='(used for single image translation) If True, include the negatives from the other samples of the minibatch when computing the contrastive loss. Please see models/patchnce.py for more details.')
        parser.add_argument('--netF', type=str, default='mlp_sample', choices=['sample', 'reshape', 'mlp_sample'], help='how to downsample the feature map')
        parser.add_argument('--netF_nc', type=int, default=256)
        parser.add_argument('--nce_T', type=float, default=0.07, help='temperature for NCE loss')
        parser.add_argument('--lmda', type=float, default=0.1, help='temperature for NCE loss')
        parser.add_argument('--num_patches', type=int, default=256, help='number of patches per layer')
        parser.add_argument('--flip_equivariance',
                            type=util.str2bool, nargs='?', const=True, default=False,
                            help="Enforce flip-equivariance as additional regularization. It's used by FastCUT, but not CUT")
        
        parser.set_defaults(pool_size=0)  # no image pooling

        opt, _ = parser.parse_known_args()

        # Set default parameters for CUT and FastCUT
        if opt.CUT_mode.lower() == "cut":
            parser.set_defaults(nce_idt=True, lambda_NCE=1.0)
        elif opt.CUT_mode.lower() == "fastcut":
            parser.set_defaults(
                nce_idt=False, lambda_NCE=10.0, flip_equivariance=True,
                n_epochs=150, n_epochs_decay=50
            )
        else:
            raise ValueError(opt.CUT_mode)

        return parser

    def __init__(self, opt):
        BaseModel.__init__(self, opt)

        # specify the training losses you want to print out.
        # The training/test scripts will call <BaseModel.get_current_losses>
        self.loss_names = ['G_GAN', 'D_real', 'D_fake', 'G', 'NCE','entropy']
        self.visual_names = ['real_A', 'fake_B', 'real_B']
        if self.opt.phase == 'test':
            self.visual_names = ['real', 'fake_1','fake_2','fake_3','fake_4','fake_5']
        self.nce_layers = [int(i) for i in self.opt.nce_layers.split(',')]

        if opt.nce_idt and self.isTrain:
            self.loss_names += ['NCE_Y']
            self.visual_names += ['idt_B']

        if self.isTrain:
            self.model_names = ['G', 'F', 'D','E']
        else:  # during test time, only load G
            self.model_names = ['G']

        # define networks (both generator and discriminator)
        # self.netPre = losses_sesim.VGG16().to(self.device)
        self.netG = networks.define_G(opt.input_nc, opt.output_nc, opt.ngf, opt.netG, opt.normG, not opt.no_dropout, opt.init_type, opt.init_gain, opt.no_antialias, opt.no_antialias_up, self.gpu_ids, opt)
        self.netF = networks.define_F(opt.input_nc, opt.netF, opt.normG, not opt.no_dropout, opt.init_type, opt.init_gain, opt.no_antialias, self.gpu_ids, opt)

        if self.isTrain:
            self.netD = networks.define_D(opt.output_nc, opt.ndf, opt.netD, opt.n_layers_D, opt.normD, opt.init_type, opt.init_gain, opt.no_antialias, self.gpu_ids, opt)
            self.netE = networks.define_D(opt.output_nc*4, opt.ndf, opt.netD, opt.n_layers_D, opt.normD,
                                          opt.init_type, opt.init_gain, opt.no_antialias, self.gpu_ids, opt)
            # define loss functions
            self.criterionGAN = networks.GANLoss(opt.gan_mode).to(self.device)
            self.criterionNCE = []

            for nce_layer in self.nce_layers:
                self.criterionNCE.append(PatchNCELoss(opt).to(self.device))

            self.criterionIdt = torch.nn.L1Loss().to(self.device)
            self.optimizer_G = torch.optim.Adam(self.netG.parameters(), lr=opt.lr, betas=(opt.beta1, opt.beta2))
            self.optimizer_D = torch.optim.Adam(self.netD.parameters(), lr=opt.lr, betas=(opt.beta1, opt.beta2))
            self.optimizer_E = torch.optim.Adam(self.netE.parameters(), lr=opt.lr, betas=(opt.beta1, opt.beta2))
            self.optimizers.append(self.optimizer_G)
            self.optimizers.append(self.optimizer_D)
            self.optimizers.append(self.optimizer_E)
            
    def data_dependent_initialize(self, data,data2):
        """
        The feature network netF is defined in terms of the shape of the intermediate, extracted
        features of the encoder portion of netG. Because of this, the weights of netF are
        initialized at the first feedforward pass with some input images.
        Please also see PatchSampleF.create_mlp(), which is called at the first forward() call.
        """
        bs_per_gpu = data["A"].size(0) // max(len(self.opt.gpu_ids), 1)
        self.set_input(data,data2)
        self.real_A = self.real_A[:bs_per_gpu]
        self.real_B = self.real_B[:bs_per_gpu]
        self.forward()                     # compute fake images: G(A)
        if self.opt.isTrain:
            # self.optimizer_D.zero_grad()
            # self.set_requires_grad(self.netE, True)
            # self.set_requires_grad(self.netD, True)
            # self.set_requires_grad(self.netG, False)
            self.compute_G_loss().backward()
            self.compute_D_loss().backward()
            # self.optimizer_E.zero_grad()
            self.compute_E_loss().backward()  # calculate gradients for D
            # self.optimizer_G.zero_grad()
            # self.set_requires_grad(self.netG, True)
            # self.compute_G_loss()#.backward()                   # calculate graidents for G
            if self.opt.lambda_NCE > 0.0:
                self.optimizer_F = torch.optim.Adam(self.netF.parameters(), lr=self.opt.lr, betas=(self.opt.beta1, self.opt.beta2))
                self.optimizers.append(self.optimizer_F)

    def optimize_parameters(self):
        # forward
        self.forward()
        self.netG.train()
        self.netE.train()
        self.netD.train()
        self.netF.train()
        # update D
        self.set_requires_grad(self.netD, True)
        # self.set_requires_grad(self.netS, True)
        # self.set_requires_grad(self.netG, False)
        self.optimizer_D.zero_grad()
        self.loss_D = self.compute_D_loss()
        self.loss_D.backward()
        self.optimizer_D.step()
        
        self.set_requires_grad(self.netE, True)
        # # self.set_requires_grad(self.netG, False)
        self.optimizer_E.zero_grad()
        self.loss_E = self.compute_E_loss()
        self.loss_E.backward()
        self.optimizer_E.step()
        
        # update G
        self.set_requires_grad(self.netD, False)
        self.set_requires_grad(self.netE, False)
        # self.set_requires_grad(self.netG, True)
        self.optimizer_G.zero_grad()
        if self.opt.netF == 'mlp_sample':
            self.optimizer_F.zero_grad()
        self.loss_G = self.compute_G_loss()
        self.compute_G_loss()
        self.loss_G.backward()
        self.optimizer_G.step()
        if self.opt.netF == 'mlp_sample':
            self.optimizer_F.step()
        
        
        
        
    def set_input(self, input,input2=None):
        
        

        
        """Unpack input data from the dataloader and perform necessary pre-processing steps.
        Parameters:
            input (dict): include the data itself and its metadata information.
        The option 'direction' can be used to swap domain A and domain B.
        """
        AtoB = self.opt.direction == 'AtoB'
        self.real_A = input['A' if AtoB else 'B'].to(self.device)
        self.real_B = input['B' if AtoB else 'A'].to(self.device)
        if input2 is not None:
            self.real_A2 = input2['A' if AtoB else 'B'].to(self.device)
            self.real_B2 = input2['B' if AtoB else 'A'].to(self.device)
        
        self.image_paths = input['A_paths' if AtoB else 'B_paths']

    def forward(self):
        
        tau = self.opt.tau
        T = self.opt.num_timesteps
        incs = np.array([0] + [1/(i+1) for i in range(T-1)])
        times = np.cumsum(incs)
        times = times / times[-1]
        times = 0.5 * times[-1] + 0.5 * times
        times = np.concatenate([np.zeros(1),times])
        times = torch.tensor(times).float().cuda()
        self.times = times
        bs =  self.real_A.size(0)
        time_idx = (torch.randint(T, size=[1]).cuda() * torch.ones(size=[bs]).cuda()).long()
        self.time_idx = time_idx
        self.timestep     = times[time_idx]
        
        with torch.no_grad():
            self.netG.eval()
            for t in range(self.time_idx.int().item()+1):
                # print(t)
                if t > 0:
                    delta = times[t] - times[t-1]
                    denom = times[-1] - times[t-1]
                    inter = (delta / denom).reshape(-1,1,1,1)
                    scale = (delta * (1 - delta / denom)).reshape(-1,1,1,1)
                Xt       = self.real_A if (t == 0) else (1-inter) * Xt + inter * Xt_1.detach() + (scale * tau).sqrt() * torch.randn_like(Xt).to(self.real_A.device)
                time_idx = (t * torch.ones(size=[self.real_A.shape[0]]).to(self.real_A.device)).long()
                time     = times[time_idx]
                z        = torch.randn(size=[self.real_A.shape[0],4*self.opt.ngf]).to(self.real_A.device)
                Xt_1     = self.netG(Xt, time_idx, z)
                
                Xt2       = self.real_A2 if (t == 0) else (1-inter) * Xt2 + inter * Xt_12.detach() + (scale * tau).sqrt() * torch.randn_like(Xt2).to(self.real_A.device)
                time_idx = (t * torch.ones(size=[self.real_A.shape[0]]).to(self.real_A.device)).long()
                time     = times[time_idx]
                z        = torch.randn(size=[self.real_A.shape[0],4*self.opt.ngf]).to(self.real_A.device)
                Xt_12    = self.netG(Xt2, time_idx, z)
                
                
                if self.opt.nce_idt:
                    XtB = self.real_B if (t == 0) else (1-inter) * XtB + inter * Xt_1B.detach() + (scale * tau).sqrt() * torch.randn_like(XtB).to(self.real_A.device)
                    time_idx = (t * torch.ones(size=[self.real_A.shape[0]]).to(self.real_A.device)).long()
                    time     = times[time_idx]
                    z        = torch.randn(size=[self.real_A.shape[0],4*self.opt.ngf]).to(self.real_A.device)
                    Xt_1B = self.netG(XtB, time_idx, z)
            if self.opt.nce_idt:
                self.XtB = XtB.detach()
            self.Xt = Xt.detach()
            self.Xt2 = Xt2.detach()
                # self.Xt2 = Xt2
                
        
        z_in    = torch.randn(size=[2*bs,4*self.opt.ngf]).to(self.real_A.device)
        z_in2    = torch.randn(size=[bs,4*self.opt.ngf]).to(self.real_A.device)
        """Run forward pass"""
        self.real = torch.cat((self.real_A, self.real_B), dim=0) if self.opt.nce_idt and self.opt.isTrain else self.real_A
        
        self.realt = torch.cat((self.Xt, self.XtB), dim=0) if self.opt.nce_idt and self.opt.isTrain else self.Xt
        
        if self.opt.flip_equivariance:
            self.flipped_for_equivariance = self.opt.isTrain and (np.random.random() < 0.5)
            if self.flipped_for_equivariance:
                self.real = torch.flip(self.real, [3])
                self.realt = torch.flip(self.realt, [3])
        
        self.fake = self.netG(self.realt,self.time_idx,z_in)
        self.fake_B2 =  self.netG(self.Xt2,self.time_idx,z_in2)
        self.fake_B = self.fake[:self.real_A.size(0)]
        if self.opt.nce_idt:
            self.idt_B = self.fake[self.real_A.size(0):]
        if self.opt.phase == 'test':
            tau = self.opt.tau
            T = self.opt.num_timesteps
            incs = np.array([0] + [1/(i+1) for i in range(T-1)])
            times = np.cumsum(incs)
            times = times / times[-1]
            times = 0.5 * times[-1] + 0.5 * times
            times = np.concatenate([np.zeros(1),times])
            times = torch.tensor(times).float().cuda()
            self.times = times
            bs =  self.real.size(0)
            time_idx = (torch.randint(T, size=[1]).cuda() * torch.ones(size=[bs]).cuda()).long()
            self.time_idx = time_idx
            self.timestep     = times[time_idx]
            visuals = []
            with torch.no_grad():
                self.netG.eval()
                for t in range(self.opt.num_timesteps):
                    # print(t)
                    if t > 0:
                        delta = times[t] - times[t-1]
                        denom = times[-1] - times[t-1]
                        inter = (delta / denom).reshape(-1,1,1,1)
                        scale = (delta * (1 - delta / denom)).reshape(-1,1,1,1)
                    Xt       = self.real_A if (t == 0) else (1-inter) * Xt + inter * Xt_1.detach() + (scale * tau).sqrt() * torch.randn_like(Xt).to(self.real_A.device)
                    time_idx = (t * torch.ones(size=[self.real_A.shape[0]]).to(self.real_A.device)).long()
                    time     = times[time_idx]
                    z        = torch.randn(size=[self.real_A.shape[0],4*self.opt.ngf]).to(self.real_A.device)
                    Xt_1     = self.netG(Xt, time_idx, z)
                    visuals.append(Xt_1)
            self.fake_1 = visuals[0]
            self.fake_2 = visuals[1]

            self.fake_3 = visuals[2]
            self.fake_4 = visuals[3]
            self.fake_5 = visuals[4]
        
        # self.real = torch.cat((self.real_A, self.real_B), dim=0) if self.opt.nce_idt and self.opt.isTrain else self.real_A
        # self.realt = torch.cat((self.Xt, self.XtB), dim=0) if self.opt.nce_idt and self.opt.isTrain else self.Xt
        # if self.opt.flip_equivariance:
        #     self.flipped_for_equivariance = self.opt.isTrain and (np.random.random() < 0.5)
        #     if self.flipped_for_equivariance:
        #         self.real = torch.flip(self.real, [3])
        #         self.realt = torch.flip(self.realt,[3])
        # # print(z.shape)
        # # ts2 = torch.cat([self.timestep]*2,dim=0)
        # self.fake = self.netG(self.realt,self.timestep,z_in)
        # # self.fake_B2 = self.netG(self.Xt2,self.timestep,z_in2)
        # self.fake_B = self.fake[:self.real_A.size(0)]
        # if self.opt.nce_idt:
        #     self.idt_B = self.fake[self.real_A.size(0):]
        
        
    def compute_D_loss(self):
        """Calculate GAN loss for the discriminator"""
        bs =  self.real_A.size(0)
        # z_in    = torch.randn(size=[2*bs,self.opt.style_dim]).to(self.real_A.device)
        # z_in2    = torch.randn(size=[bs,self.opt.style_dim]).to(self.real_A.device)
        # ts2 = torch.cat([self.timestep]*2,dim=0)
        # self.fake = self.netG(self.realt,ts2,z_in)
        # self.fake_B2 = self.netG(self.Xt2,self.timestep,z_in2)
        # self.fake_B = self.fake[:self.real_A.size(0)]
        
        fake = self.fake_B.detach()
        std = torch.rand(size=[1]).item() * self.opt.std
        # X1_noise   = self.real_B + std * torch.randn_like(self.real_B).to(self.Xt.device)
        # Xt_1_noise = fake + std * torch.randn_like(self.fake_B).to(self.Xt.device)
        # Fake; stop backprop to the generator by detaching fake_B
        pred_fake = self.netD(fake,self.time_idx)
        self.loss_D_fake = self.criterionGAN(pred_fake, False).mean()
        # Real
        self.pred_real = self.netD(self.real_B,self.time_idx)
        loss_D_real = self.criterionGAN(self.pred_real, True)
        self.loss_D_real = loss_D_real.mean()

        # combine loss and calculate gradients
        self.loss_D = (self.loss_D_fake + self.loss_D_real) * 0.5
        return self.loss_D
    def compute_E_loss(self):
        
        bs =  self.real_A.size(0)
        # z_in    = torch.randn(size=[2*bs,self.opt.style_dim]).to(self.real_A.device)
        # z_in2    = torch.randn(size=[bs,self.opt.style_dim]).to(self.real_A.device)
        # ts2 = torch.cat([self.timestep]*2,dim=0)
        # self.fake = self.netG(self.realt,ts2,z_in)
        # self.fake_B2 = self.netG(self.Xt2,self.timestep,z_in2)
        # self.fake_B = self.fake[:self.real_A.size(0)]
        
        """Calculate GAN loss for the discriminator"""
        
        XtXt_1 = torch.cat([self.Xt,self.fake_B.detach()], dim=1)
        XtXt_2 = torch.cat([self.Xt2,self.fake_B2.detach()], dim=1)
        temp = torch.logsumexp(self.netE(XtXt_1, self.time_idx, XtXt_2).reshape(-1), dim=0).mean()
        self.loss_E = -self.netE(XtXt_1, self.time_idx, XtXt_1).mean() +temp + temp**2
        # opt_E.zero_grad()
        # self.loss_E.backward()
        # self.loss_E = -self.netE(XtXt_1,self.timestep, XtXt_1).mean() + torch.logsumexp(self.netE(XtXt_1,self.timestep, XtXt_2).reshape(-1), dim=0)
        
        return self.loss_E
    def compute_G_loss(self):
        bs =  self.real_A.size(0)
        # z_in    = torch.randn(size=[2*bs,self.opt.style_dim]).to(self.real_A.device)
        # z_in2    = torch.randn(size=[bs,self.opt.style_dim]).to(self.real_A.device)
        # ts2 = torch.cat([self.timestep]*2,dim=0)
        # self.fake = self.netG(self.realt,ts2,z_in)
        # self.fake_B2 = self.netG(self.Xt2,self.timestep,z_in2)
        # self.fake_B = self.fake[:self.real_A.size(0)]
        # if self.opt.nce_idt:
        #     self.idt_B = self.fake[self.real_A.size(0):]
        tau = self.opt.tau
        # T = self.opt.num_timesteps
        
        
        
        """Calculate GAN and NCE loss for the generator"""
        fake = self.fake_B
        std = torch.rand(size=[1]).item() * self.opt.std
        # Xt_1_noise = fake + std * torch.randn_like(self.fake_B).to(self.Xt.device)
        # Xt_1_noise = fake + std * torch.randn_like(self.fake_B).to(self.Xt.device)
        # Xt_1_noise = fake #+ std * torch.randn_like(self.fake_B).to(self.Xt.device)
        # First, G(A) should fake the discriminator
        if self.opt.lambda_GAN > 0.0:
            # print(fake.shape)
            # pred_fake = fake.mean(dim=[1,2,3])
            pred_fake = self.netD(fake,self.time_idx)
            # print(pred_fake.shape)
            self.loss_G_GAN = self.criterionGAN(pred_fake, True).mean() * self.opt.lambda_GAN
        else:
            self.loss_G_GAN = 0.0
        self.loss_entropy = 0
        if self.opt.lambda_entropy > 0.0:
            XtXt_1 = torch.cat([self.Xt, self.fake_B], dim=1)
            XtXt_2 = torch.cat([self.Xt2, self.fake_B2], dim=1)
            # params   = list(self.netG.parameters() )
            bs = self.opt.batch_size

            # XtXt_1   = torch.cat([Xt,Xt_1], dim=1)
            ET_XY    = self.netE(XtXt_1, self.time_idx, XtXt_1).mean() - torch.logsumexp(self.netE(XtXt_1, self.time_idx, XtXt_2).reshape(-1), dim=0)
            self.loss_entropy = -(self.opt.num_timesteps-self.time_idx[0])/self.opt.num_timesteps*self.opt.tau*ET_XY
            self.loss_entropy += self.opt.tau*torch.mean((self.Xt-self.fake_B)**2)
        if self.opt.lambda_NCE > 0.0:
            self.loss_NCE = self.calculate_NCE_loss(self.real_A, fake)
        else:
            self.loss_NCE, self.loss_NCE_bd = 0.0, 0.0

        if self.opt.nce_idt and self.opt.lambda_NCE > 0.0:
            self.loss_NCE_Y = self.calculate_NCE_loss(self.real_B, self.idt_B)
            loss_NCE_both = (self.loss_NCE + self.loss_NCE_Y) * 0.5
        else:
            loss_NCE_both = self.loss_NCE
            
        
        loss_SB =  loss_NCE_both + self.loss_entropy
        

        
        # loss_ADV = 
        self.loss_G = self.loss_G_GAN + loss_SB
        return self.loss_G
        # params = [self.netG.parameters(), self.netF.parameters()]
#         params   = list(self.netG.parameters() ) + list(self.netF.parameters())
#         # print(len(params))
#         # params   = list(self.netG.parameters())
#         grad_ENT = torch.autograd.grad(outputs=fake, inputs=params, grad_outputs=grad_ENT, allow_unused=True, retain_graph=True)
#         grad_OT  = torch.autograd.grad(outputs=loss_NCE_both, inputs=params, grad_outputs=torch.ones_like(loss_NCE_both), allow_unused=True, retain_graph=True)
#         grad_ADV = torch.autograd.grad(outputs=self.loss_G_GAN, inputs=params, grad_outputs=torch.ones_like(self.loss_G_GAN), allow_unused=True)
        
        
        
#         # grad_SB  = torch.autograd.grad(outputs=loss_SB, inputs=params, grad_outputs=torch.ones_like(loss_SB), allow_unused=True, retain_graph=True)
#         # grad_ADV = torch.autograd.grad(outputs=self.loss_G_GAN, inputs=params, grad_outputs=torch.ones_like(self.loss_G_GAN), allow_unused=True)
        
#         for k,p in enumerate(params):
#             if grad_ENT[k] is not None:
#                 # grad_SB_norm = torch.norm(grad_SB[k])
#                 # grad_ADV_norm = torch.norm(grad_ADV[k])
#                 # factor = grad_ADV_norm / (grad_SB_norm + 1e-10)
                
#                 grad_SB = (self.times[-1] - self.timestep[0])*tau*grad_ENT[k] + grad_OT[k]
#                 grad_SB_norm = torch.norm(grad_SB)
#                 grad_ADV_norm = torch.norm(grad_ADV[k])
#                 factor = torch.minimum(grad_ADV_norm, grad_SB_norm) / (grad_SB_norm + 1e-10)
                
#                 p.grad = self.opt.lmda * factor * grad_SB + grad_ADV[k]
        
        # return self.loss_G

    def calculate_NCE_loss(self, src, tgt):
        n_layers = len(self.nce_layers)
        z    = torch.randn(size=[self.real_A.size(0),4*self.opt.ngf]).to(self.real_A.device)
        feat_q = self.netG(tgt, self.time_idx*0, z, self.nce_layers, encode_only=True)

        if self.opt.flip_equivariance and self.flipped_for_equivariance:
            feat_q = [torch.flip(fq, [3]) for fq in feat_q]
        
        feat_k = self.netG(src, self.time_idx*0,z,self.nce_layers, encode_only=True)
        feat_k_pool, sample_ids = self.netF(feat_k, self.opt.num_patches, None)
        feat_q_pool, _ = self.netF(feat_q, self.opt.num_patches, sample_ids)

        total_nce_loss = 0.0
        for f_q, f_k, crit, nce_layer in zip(feat_q_pool, feat_k_pool, self.criterionNCE, self.nce_layers):
            loss = crit(f_q, f_k) * self.opt.lambda_NCE
            total_nce_loss += loss.mean()

        return total_nce_loss / n_layers
