# first line: 59
@utils.cache
def test_hitting_time(seed=None, verbose=False,
        algorithm="local", num_random_walks=1000, max_len=10000,
        compute_truth=True, **kwargs):
    """
        Tests the hitting time estimation algorithms on a graph.

        Parameters
        ----------
        seed : int
            The seed to use for the random number generator.
        verbose : bool
            If True, prints the results of the test.
        algorithm : str
            The algorithm to use for the hitting time estimation. Can be "local", "sampling", "exact" or "eff_res".
        kwargs : dict
            The parameters to pass to the graph generation function.
    """
    G = networks.get_graph(**kwargs, seed=seed)
    u, v = sample_nodes(G, **kwargs, seed=seed)

    # The ground truth.
    # We compute it using a linear system solver.
    if compute_truth:
        true_hitting_time = ht.exact_ht(G, u, v, sparse=True)
    else:
        true_hitting_time = -1

    walltime = None
    num_samples = None

    # The algorithms to test.
    if algorithm == "exact":
        start_time = time.time()
        hitting_time = ht.exact_ht(G, u, v)
        walltime = time.time() - start_time

    if algorithm == "exact-sparse":
        start_time = time.time()
        hitting_time = ht.exact_ht(G, u, v, sparse=True)
        walltime = time.time() - start_time

    # Cutoff algorithm 
    if algorithm == "cutoff":
        hitting_time, num_samples, walltime = ht.ht_via_cutoff(G, u, v,
                max_len=max_len, num_random_walks=num_random_walks)

    # Collision time algorithm 
    if algorithm == "local-delete":
        hitting_time, num_samples, walltime = ht.estimate_local_ht(G, u, v,
                num_random_walks=num_random_walks, max_len=max_len)

    elif algorithm == "sampling":
        hitting_time, num_samples, walltime = ht.sampling_ht(G, u, v,
                num_random_walks=num_random_walks)

    absolute_error = abs(hitting_time - true_hitting_time)
    relative_error = absolute_error / true_hitting_time

    return {"hitting_time": hitting_time, "true_hitting_time": true_hitting_time,
            "time": walltime, "absolute_error": absolute_error,
            "relative_error": relative_error, "error": relative_error,
            "num_samples": num_samples, "deg_u": G.degree(u),
            "deg_v": G.degree(v), "ix_u": u, "ix_v": v}
