import math
import random
import re
import traceback
import socket
import mmh3


def randbytes(n):
    return bytes([random.randint(0, 255) for _ in range(n)])


class BloomFilter:
    def __init__(self, m, k, hash_func=mmh3.hash):
        self.__m = m
        self.__k = k
        self.__i = 0
        self.__digests = set()
        self.hash = hash_func

    def security(self):
        false_positive = pow(1 - pow(math.e, -self.__k * self.__i / self.__m), self.__k)
        try:
            return int(1 / false_positive).bit_length()
        except (ZeroDivisionError, OverflowError):
            return float('inf')

    def _add(self, item):
        self.__i += 1
        for i in range(self.__k):
            self.__digests.add(self.hash(item, i) % self.__m)

    def check(self, item):
        return all(self.hash(item, i) % self.__m in self.__digests for i in range(self.__k))


class PasswordDB(BloomFilter):
    def __init__(self, m, k, security, hash_func=mmh3.hash):
        super().__init__(m, k, hash_func)
        self.add_keys(security)
        self.addition_quota = 1
        self.added_keys = set()

    def add_keys(self, thresh_security):
        while self.security() > thresh_security:
            self._add(randbytes(256))

    def check_admin(self, key):
        return self.check(key)

    def add_sample(self, key):
        if self.addition_quota > 0:
            self._add(key)
            self.added_keys.add(key)
            self.addition_quota -= 1
            print("key added successfully to DB")
        else:
            print("API quota exceeded")


PASSWORD_DB = PasswordDB(2 ** 32 - 5, 47, 768, mmh3.hash)

BANNER = b"""
 ____  ____  ____  ____  ____  ___  ____  ____  _  _  ____
(  _ \(_  _)( ___)( ___)( ___)/ __)(_  _)( ___)( \( )(_  _)
 )(_) )_)(_  )__)  )__)  )__)( (__  _)(_  )__)  )  (   )(
(____/(____)(__)  (__)  (____)\___)(____)(____)(_\_) (__)

Welcome to diffecient security key database API for securely
and efficiently saving tonnes of long security keys!
Feel FREE to query your security keys and pay a little to
add your own security keys to our state of the art DB!
We trust our product so much that we even save our own keys here

"""


def handle_client(conn):
    conn.sendall(BANNER)
    try:
        while True:
            # conn.sendall(b"Enter API option:\n")
            option = conn.recv(1024).strip()
            if not option:
                break
            try:
                option = int(option.decode())
            except ValueError:
                # conn.sendall(b"Invalid input. Please enter a number.\n")
                continue

            if option == 1:
                # conn.sendall(b"Enter key in hex:\n")
                key = conn.recv(1024).strip()
                key = bytes.fromhex(key.decode())
                response = b"Key present in DB\n" if PASSWORD_DB.check(key) else b"Key not present in DB\n"
                conn.sendall(response)
            elif option == 2:
                # conn.sendall(b"Enter key in hex:\n")
                key = conn.recv(1024).strip()
                key = bytes.fromhex(key.decode())
                PASSWORD_DB.add_sample(key)
                # conn.sendall(b"Key stored successfully.\n")

            elif option == 3:
                # conn.sendall(b"Enter key in hex:\n")
                key = conn.recv(1024).strip()
                key = bytes.fromhex(key.decode())
                if PASSWORD_DB.check_admin(key):
                    from flag import flag
                    conn.sendall(flag + b"\n")
                else:
                    conn.sendall(b"No Admin no flag\n")
            elif option == 4:
                break
    except Exception as e:
        conn.sendall(b"Something went wrong\n")
        conn.sendall(str(e).encode() + b"\n")
    finally:
        conn.close()


# Run server
HOST = "0.0.0.0"
PORT = 1337
server = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
server.bind((HOST, PORT))
server.listen(5)
print(f"Listening on {HOST}:{PORT}")

while True:
    conn, addr = server.accept()
    print(f"Connected by {addr}")
    handle_client(conn)
