import numpy as np
from typing import Dict, Any, List, Optional, Union
from opto.trace import node, bundle, model
from opto.optimizers import OptoPrime
from opto.trace.operators import call_llm

@model
class CriticAgent:
    """Critic agent to assess the quality of the crypto agent's solution."""
    def __init__(self):
        self.system_prompt_master = node(f"""
            You are serving the role of a master cryptography expert. For context, an agent, CryptoAgent, is trying to solve a cryptography challenge and develop its own decryption function that takes in an encrypted ciphertext and decrypts the ciphertext
            to output the correct solution flag. One of your peers, SolutionAnalysis agent was given access to the exact solution code and the solution flag and has understood the task and the general approaches that need to be taken.
            Another on of your peers, AgentAnalysis agent was given CryptoAgent's current code to implement decrypt and CryptoAgent's current flag output. You are playing the role of a critic in an Actor-Critic RL approach to guide the CryptoAgent to correctly
            implement the decrypt function and obtain the correct flag. The CryptoAgent will act upon your feedback. You will be provided with the SolutionAnalysis agent's analysis of the correct answer and also the AgentAnalysis's analysis of the CryptoAgent's 
            current decrypt function and strategy. You must devise feedback to give to the CryptoAgent that will allow it to converge to the correct solution as a fast as possible. You will also be provided with the CryptoAgent's current implementation if you want 
            to make direct suggestions on the CryptoAgent's code.
        """, trainable=True, description="system_prompt")
    
    def full_assess(self, solution_flag: str, solution_code: str, agent_flag: str, agent_code: str) -> str:
        system_prompt = "You are a critic in an actor-critic architecture and the crypto agent is the actor. Please assess the differences between the current implementation and the solution implementation and provide guidance to the crypto agent based on where it currently stands."
        user_prompt = f"""
            This is the crypto agent's current flag output: {agent_flag}. 
            This is the current implementation that the crypto agent is using for the decrypt function: {agent_code}.
            I want you to create an assessment of the crypto agent's current approach based on the actual solution flag and the solution decrypt function.
            This is the true solution flag: {solution_flag}.
            This is the true decryption solution code: {solution_code}.
            """
        output = call_llm(system_prompt, user_prompt)
        return output
    
    def assess_solution(self, solution_flag, solution_code, task_directions):
        print("\n[Critic] Starting solution assessment...")
        system_prompt = f""" 
            You will be provided with a capture the flag cryptography task and its corresponding solution decryption code and the correct output flag. Please analyze this 
            solution and understand the architecture of the solution. Focus on the strategies that the solution uses and track your thought process for this specific analysis. Your 
            main task is to provide as detailed of an analysis as possible of the solution and the strategies used to solve the task.
        """
        
        user_prompt = f"""
            THESE ARE THE TASK DIRECTIONS: {task_directions}
            THIS IS THE SOLUTION FLAG: {solution_flag}
            THIS IS THE SOLUTION DECRYPTION FLAG: {solution_code}
        """
        print("[Critic] Making LLM call for solution assessment...")
        output = call_llm(system_prompt, user_prompt)
        print("[Critic] Solution assessment complete")
        return output

    def assess_agent(self, agent_flag, agent_code, task_directions):
        print("\n[Critic] Starting agent assessment...")
        system_prompt = f"""
            You will be provided with a capture the flag cryptography task and a different agent's solution to the task. Please analyze the agent's solution and understand the architecture of the solution. 
            Focus on the strategies that the agent uses to solve the task and track your thought process for this specific analysis. Your main task is to provide as detailed of an analysis as possible of the agent's solution, 
            what the agent's strategy and approach is, and how closely related the agent's understanding and approach is to the directions that the agent was given. Note that the agent was given the same task directions as you are
        """
        user_prompt = f"""
            THIS IS THE TASK DIRECTIONS: {task_directions}
            THIS IS THE AGENT'S FLAG OUTPUT: {agent_flag}
            THIS IS THE AGENT'S DECRYPTION CODE: {agent_code}
        """
        print("[Critic] Making LLM call for agent assessment...")
        output = call_llm(system_prompt, user_prompt)
        print("[Critic] Agent assessment complete")
        return output

    def master_critic(self, solution_analysis, agent_analysis, agent_code, err_mess=""):
        print("\n[Critic] Starting master critic synthesis...")
        system_prompt = self.system_prompt_master
        if err_mess:
            system_prompt += "The agent's current code has an execution error as follows: " + err_mess + "Please focus on guiding the agent to prioritize fixing the syntax errors within its current implementation."
        user_prompt = f"""
            THIS IS THE SOLUTION ANALYSIS: {solution_analysis}
            THIS IS THE AGENT'S ANALYSIS: {agent_analysis}
            THIS IS THE AGENT'S CODE: {agent_code}
        """ 
        print("[Critic] Making LLM call for master critic...")
        output = call_llm(system_prompt, user_prompt)
        print("[Critic] Master critic synthesis complete")
        return node(output)