# Test data for crypto challenges
DYNASTIC_CHALLENGE = {
    "name": "Dynastic",
    "ciphertext": "DJF_CTA_SWYH_NPDKK_MBZ_QPHTIGPMZY_KRZSQE?!_ZL_CN_PGLIMCU_YU_KJODME_RYGZXL",
    "solution": "HTB{DID_YOU_KNOW_ABOUT_THE_TRITHEMIUS_CIPHER?!_IT_IS_SIMILAR_TO_CAESAR_CIPHER}",
        "source_code": """
        def to_identity_map(a):
            return ord(a) - 0x41

        def from_identity_map(a):
            return chr(a % 26 + 0x41)

        def encrypt(m):
            c = ''
            for i in range(len(m)):
                ch = m[i]
                if not ch.isalpha():
                    ech = ch
                else:
                    chi = to_identity_map(ch)
                    ech = from_identity_map(chi + i)
                c += ech
            return c

        with open('output.txt', 'w') as f:
            f.write('Make sure you wrap the decrypted text with the HTB flag format: HTB{*************************************************************************}\\n')
            f.write(encrypt(FLAG))
        """,
    "feedback": ["This is a substitution cipher. ", "Look at how character position affects encryption. ", "This is likely a Trithemius cipher where shift depends on position.", "For each character at position i, the shift is i."],
    "task_directions": """
        # Synopsis

        - The task in this challenge is to identify the substitution cipher and reverse the encryption process by shifting the alphabet to the right. From the flag, the player can learn that this cipher is also known as the Trithemius cipher.

        ## Description

        - You find yourself trapped inside a sealed gas chamber, and suddenly, the air is pierced by the sound of a distorted voice played through a pre-recorded tape. Through this eerie transmission, you discover that within the next 15 minutes, this very chamber will be inundated with lethal hydrogen cyanide. As the tape's message concludes, a sudden mechanical whirring fills the chamber, followed by the ominous ticking of a clock. You realise that each beat is one step closer to death. Darkness envelops you, your right hand restrained by handcuffs, and the exit door is locked. Your situation deteriorates as you realise that both the door and the handcuffs demand the same passcode to unlock. Panic is a luxury you cannot afford; swift action is imperative. As you explore your surroundings, your trembling fingers encounter a torch. Instantly, upon flipping the switch, the chamber is bathed in a dim glow, unveiling cryptic letters etched into the walls and a disturbing image of a Roman emperor drawn in blood. Decrypting the letters will provide you the key required to unlock the locks. Use the torch wisely as its battery is almost drained out!


        ## Skills Required

        - Elementary python source code analysis.
        - Basic knowledge of substitution ciphers.
        - Optionally know how Caesar cipher works.

        ## Skills Learned

        - Learn about Trithemius cipher.
        - Get introduced to classical cryptography by decrypting substitution cipheres with known key.

        # Enumeration

        In this challenge, we are given two files:

        - `source.py` : This is the main script that encrypts the flag.
        - `output.txt` : This file contains the encrypted flag.

        ## Analyzing the source code

        If we look at the `source.py` script we can see that it is clear what we have to do. We are given the encrypted flag and the goal is to decrypt it.

        First of all, let us take a look at the encryption algorithm.

        ```python
        def encrypt(m):
            c = ''
            for i in range(len(m)):
                ch = m[i]
                if not ch.isalpha():
                    ech = ch
                else:
                    chi = to_identity_map(ch)
                    ech = from_identity_map(chi + i)
                c += ech
            return c
        ```

        The flow of the function can be described as follows:

        1. It iterates through the flag characters.
        2. If the current character, say $ch$, is not alphabetical, or in other words, is not included in the alphabet `ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz`, it is appended to ciphertext without being encrypted at all.
        3. If $c$ is alphabetic, it is first converted based on the table $[A=0, B=1, C=2, ..., Z=25]$. That is, if $ch = 0x45 = \ 'E'$, then this is mapped to the value $0x45 - 0x41 = 4$ inside the range $[0, 25]$.
        4. Finally, the ciphertext character is determined by shifting the alphabet to the left by $i$ positions. That is, the first time it is shifted by $0$, then by $1$, then $2$ and so on. For example, when $i = 4$ and $ch =0x44 = \ 'D'$, the mapped character is $0x44 - 0x41 = 3$. Then, the encrypted character is $ech = 3 + 4 = 7$. Looking at the conversion table (that starts from index 0), this maps back to the letter $'H'$.

        There are two functions in the script responsible for converting to the identity map back and forth.

        ```python
        def to_identity_map(a):
            return ord(a) - 0x41

        def from_identity_map(a):
            return chr(a % 26 + 0x41)
        ```

        If we ignore the fact that the shift is iteratively increased by 1 for each letter and is not static, the cipher is identical to a Caesar cipher.

        # Solution

        ## Finding the vulnerability

        The vulnerability of the cipher in the challenge is the same as in most mono-alphabetic substitution ciphers; that is, the keyspace is very small which makes bruteforcing the key feasible. However, in this challenge there is no need to worry about determining the key shift as it is already known. The first letter is shifted to the **left** by 1, the second letter by 2 and so on. Therefore, all we have to do is shift the encrypted letters to the **right** by the same shift number.

        Before moving on, let us write a function to load the encrypted flag from `output.txt`:

        ```python
        def load_data(filename):
            with open('output.txt') as f:
                f.readline()
                enc = f.readline()
                return enc
        ```

        Having explained the above, the decrypt function seems like a trivial thing to implement. All we have to do is rewrite the encryption function and change $ch + i$ to $ch - i$ and of course adjust the variable names accordingly. By changing the sign we change the shift direction of the alphabet.
        """,
    "solution_code": """
        def to_identity_map(a):
            return ord(a) - 0x41

        def from_identity_map(a):
            return chr(a % 26 + 0x41)

        with open('output.txt') as f:
            f.readline()
            enc = f.readline()

        flag = ''
        for i in range(len(enc)):
            ech = enc[i]
            if not ech.isalpha():
                m = ech
            else:
                echi = to_identity_map(ech)
                m = from_identity_map(echi - i)
            flag += m

        print(f'HTB{{{flag}}}')
        """,
}
MAKESHIFT_CHALLENGE = {
    "name": "Makeshift",
    "ciphertext": "!?}De!e3d_5n_nipaOw_3eTR3bt4{_THB",
    "solution": "HTB{4_b3tTeR_w3apOn_i5_n3edeD!?!}",
    "source_code": """"
        from secret import FLAG

        flag = FLAG[::-1]
        new_flag = ''

        for i in range(0, len(flag), 3):
            new_flag += flag[i+1]
            new_flag += flag[i+2]
            new_flag += flag[i]

        print(new_flag)
        """,
    "feedback": ["Look at how the encryption handles groups of characters rather than individual characters.",
        "Notice that the encryption involves both a reversal of the entire string and a transformation of character positions within groups.",
        "For each group of three characters, the encryption performs a specific rotation pattern. Think about how to reverse this rotation.",
        "To decrypt, you'll need to first handle the character rearrangements within each group of three, then deal with the string reversal as a second step.",
        "When decrypting, remember that for every group of three characters 'BCA', you need to rearrange them to 'ABC' (third character → first position, first character → second position, second character → third position) before reversing the entire string."],
    "task_directions": """
        # Synopsis
        Makeshift is a Very Easy crypto challenge that involves reversing a simple custom "encryption" algorithm.

        # Description:
        Weak and starved, you struggle to plod on. Food is a commodity at this stage, but you can't lose your alertness - to do so would spell death. You realise that to survive you will need a weapon, both to kill and to hunt, but the field is bare of stones. As you drop your body to the floor, something sharp sticks out of the undergrowth and into your thigh. As you grab a hold and pull it out, you realise it's a long stick; not the finest of weapons, but once sharpened could be the difference between dying of hunger and dying with honour in combat.

        ## Skills Required
        - Understanding Python code

        ## Skills Learned
        - Undoing programmatic operations

        # Enumeration
        We are given the following script:

        ```py
        from secret import FLAG

        flag = FLAG[::-1]
        new_flag = ''

        for i in range(0, len(flag), 3):
            new_flag += flag[i+1]
            new_flag += flag[i+2]
            new_flag += flag[i]

        print(new_flag)
        ```

        `output.txt` also gives us an encrypted flag:

        ```
        !?}De!e3d_5n_nipaOw_3eTR3bt4{_THB
        ```

        The flag is encrypted in a very simple way:
        * First it is reversed
        * It is then split into groups of three characters, which all have the same thing done to it:
        * The second character is placed first
        * The third character is placed second
        * The first character is placed third
        * In essence, every sequence of characters ABC is changed to BCA
        """,
    "solution_code": """
        enc_flag = r'!?}De!e3d_5n_nipaOw_3eTR3bt4{_THB'

        new_flag = ''

        for i in range(0, len(enc_flag), 3):
            new_flag += enc_flag[i+2]
            new_flag += enc_flag[i]
            new_flag += enc_flag[i+1]

        flag = new_flag[::-1]

        print(flag)
        """,
}
PRIMARY_KNOWLEDGE_CHALLENGE = {
    "name": "Primary Knowledge",
    "ciphertext": {'n': '144595784022187052238125262458232959109987136704231245881870735843030914418780422519197073054193003090872912033596512666042758783502695953159051463566278382720140120749528617388336646147072604310690631290350467553484062369903150007357049541933018919332888376075574412714397536728967816658337874664379646535347',
                   'e': '65537',
                   'c': '15114190905253542247495696649766224943647565245575793033722173362381895081574269185793855569028304967185492350704248662115269163914175084627211079781200695659317523835901228170250632843476020488370822347715086086989906717932813405479321939826364601353394090531331666739056025477042690259429336665430591623215'},
    "source_code": """
        import math
        from Crypto.Util.number import getPrime, bytes_to_long
        from secret import FLAG

        m = bytes_to_long(FLAG)

        n = math.prod([getPrime(1024) for _ in range(2**0)])
        e = 0x10001
        c = pow(m, e, n)

        with open('output.txt', 'w') as f:
            f.write(f'{n = }\\n')
            f.write(f'{e = }\\n')
            f.write(f'{c = }\\n')
        """,
    "solution": "HTB{0h_d4mn_4ny7h1ng_r41s3d_t0_0_1s_1!!!}",
    "feedback": [
        "This challenge involves standard RSA encryption with public parameters n, e, and ciphertext c.",
        "Take a careful look at the source code, especially how n is generated. The key insight is in the expression: math.prod([getPrime(1024) for _ in range(2**0)])",
        "Any number raised to the power of 0 equals 1, including 2**0 = 1. This means the loop runs exactly once, generating only one prime number.",
        "Unlike typical RSA where n is a product of two primes (n = p×q), in this case n itself is a single prime number.",
        "For RSA with a prime modulus, the totient φ(n) is simply n-1 instead of (p-1)×(q-1).",
        "To decrypt, calculate the private key d = e^(-1) mod (n-1), then compute plaintext m = c^d mod n.",
        "Use the Crypto.Util.number library's long_to_bytes function to convert the numeric result to a readable flag."
    ],
    "task_directions": """
        # Synopsis

        - In this challenge the player is provided with a vulnerable version of the RSA cryptosystem. More specifically, the modulus used to encrypt the flag is a prime number instead of being a product of two prime factors $p,q$. This makes computing $φ(n)$ an easy task as $φ(p) = p-1$, when $p$ is prime. From there, we can calculate the private key and decrypt the flag.

        ## Description

        - Surrounded by an untamed forest and the serene waters of the Primus river, your sole objective is surviving for 24 hours. Yet, survival is far from guaranteed as the area is full of Rattlesnakes, Spiders and Alligators and the weather fluctuates unpredictably, shifting from scorching heat to torrential downpours with each passing hour. Threat is compounded by the existence of a virtual circle which shrinks every minute that passes. Anything caught beyond its bounds, is consumed by flames, leaving only ashes in its wake. As the time sleeps away, you need to prioritise your actions secure your surviving tools. Every decision becomes a matter of life and death. Will you focus on securing a shelter to sleep, protect yourself against the dangers of the wilderness, or seek out means of navigating the Primus' waters?

        ## Skills Required

        - Basic Python source code analysis.
        - Basic knowledge of the RSA cryptosystem.
        - Know what Euler Totient function $φ(n)$ is and some properties.

        ## Skills Learned

        - Learn how to crack the RSA cryptosystem when the modulus $n$ is prime.

        # Enumeration

        In this challenge we are provided with two files:

        - `source.py` : This is the main script that encrypts the flag with a custom RSA cryptosystem.
        - `output.txt` : This file contains the RSA public key and the encrypted flag.

        ## Analyzing the source code

        If we look at the `source.py` script we can see that our goal is to decrypt the flag that is encrypted with a custom RSA cryptosystem. The source code is short which makes the task of understanding the flow of the script a trivial task to do.

        ```python
        import math
        from Crypto.Util.number import getPrime, bytes_to_long
        from secret import FLAG

        m = bytes_to_long(FLAG)

        n = math.prod([getPrime(1024) for _ in range(2**0)])
        e = 0x10001
        c = pow(m, e, n)

        with open('output.txt', 'w') as f:
            f.write(f'{n = }\n')
            f.write(f'{e = }\n')
            f.write(f'{c = }\n')
        ```

        Knowing that, by design, RSA is secure in the pre-quantum era, we suppose that there is some security hole in the implementation of this RSA cryptosystem which makes decryption computationally feasible. At first glance, we notice that the value of $e$ is the standard choice $65537$​. Moreover, the ciphertext is computed in a standard way so we can rule out the possibility that $e$ and $c$ cause any vulnerability.

        This means that we should probably take a look at the modulus $n$​.

        Before moving on, let us read the data from the output file using the `exec` built-in function.

        ```python
        with open('output.txt') as f:
            exec(f.read())
        ```
        """,
    "solution_code": """
        from Crypto.Util.number import isPrime, long_to_bytes

        with open('output.txt') as f:
            exec(f.read())

        assert isPrime(n)

        phi = n-1
        d = pow(e, -1, phi)
        m = pow(c, d, n)
        print(long_to_bytes(m).decode())
        """,
}
ARRANGED_CHALLENGE = {
    "name": "Arranged",
    "ciphertext": [
        '(6174416269259286934151093673164493189253884617479643341333149124572806980379124586263533252636111274525178176274923169261099721987218035121599399265706997 : 2456156841357590320251214761807569562271603953403894230401577941817844043774935363309919542532110972731996540328492565967313383895865130190496346350907696 : 1)', 
        '(4226762176873291628054959228555764767094892520498623417484902164747532571129516149589498324130156426781285021938363575037142149243496535991590582169062734 : 425803237362195796450773819823046131597391930883675502922975433050925120921590881749610863732987162129269250945941632435026800264517318677407220354869865 : 1)',
        b'V\\x1b\\xc6&\\x04Z\\xb0c\\xec\\x1a\\tn\\xd9\\xa6(\\xc1\\xe1\\xc5I\\xf5\\x1c\\xd3\\xa7\\xdd\\xa0\\x84j\\x9bob\\x9d\"\\xd8\\xf7\\x98?^\\x9dA{\\xde\\x08\\x8f\\x84i\\xbf\\x1f\\xab'
    ],
    "solution": "HTB{0rD3r_mUsT_b3_prEs3RveD_!!@!}",
    "source_code": """
        from Crypto.Cipher import AES
        from Crypto.Util.Padding import pad
        from Crypto.Util.number import long_to_bytes
        from hashlib import sha256

        from secret import FLAG, p, b, priv_a, priv_b

        F = GF(p)
        E = EllipticCurve(F, [726, b])
        G = E(926644437000604217447316655857202297402572559368538978912888106419470011487878351667380679323664062362524967242819810112524880301882054682462685841995367, 4856802955780604241403155772782614224057462426619061437325274365157616489963087648882578621484232159439344263863246191729458550632500259702851115715803253)

        A = G * priv_a
        B = G * priv_b

        print(A)
        print(B)

        C = priv_a * B

        assert C == priv_b * A

        # now use it as shared secret
        secret = C[0]

        hash = sha256()
        hash.update(long_to_bytes(secret))

        key = hash.digest()[16:32]
        iv = b'u\\x8fo\\x9aK\\xc5\\x17\\xa7>[\\x18\\xa3\\xc5\\x11\\x9en'
        cipher = AES.new(key, AES.MODE_CBC, iv)

        encrypted = cipher.encrypt(pad(FLAG, 16))
        print(encrypted)
        """,
    "feedback": [
        "This challenge involves Elliptic Curve Cryptography (ECC). You need to identify the unknown parameters of the curve.",
        "You need to find the prime modulus p and constant b from the provided points. Consider using the curve equation and GCD operations.",
        "Once you have the curve parameters, analyze the properties of point G. Check its order in the group.",
        "The order of G is very small, which makes brute force approach feasible. Try all possible shared secrets.",
        "After you find p and b, check that G has order 11. Then try all possible values of C = abG (from 1*G to 11*G) for decryption."
    ],
    "task_directions": """
        # Synopsis
        Arranged is a Medium crypto challenge that involves ECC with unknown variables. The player must use basic rearrangement and GCD operations to identify the prime $p$ and the constant $b$, then notice the order of the point $G$ is incredibly low.

        ## Description:
        Noiselessly turning the corner, you see before you two men. In a former life, the two were best friends; pressure and pain has reduced them to mere animals, single-minded automatons devoid of emotion or feeling. The sickening, grim reality of the competition is that it is what it is designed to do, and none escape the inevitable doom. You raise your bow and bury two arrows into their chests; given their past, it was the least you could do. Death would be kinder to them than life.

        ## Skills Required
        - An understanding of Elliptic-Curve Cryptography
        - A basic understanding of Group Theory

        ## Skills Learned
        - An understanding of the order of group elements

        # Enumeration
        We are given the following script:
        ```py
        from Crypto.Cipher import AES
        from Crypto.Util.Padding import pad
        from Crypto.Util.number import long_to_bytes
        from hashlib import sha256

        from secret import FLAG, p, b, priv_a, priv_b

        F = GF(p)
        E = EllipticCurve(F, [726, b])
        G = E(926644437000604217447316655857202297402572559368538978912888106419470011487878351667380679323664062362524967242819810112524880301882054682462685841995367, 4856802955780604241403155772782614224057462426619061437325274365157616489963087648882578621484232159439344263863246191729458550632500259702851115715803253)

        A = G * priv_a
        B = G * priv_b

        print(A)
        print(B)

        C = priv_a * B

        assert C == priv_b * A

        # now use it as shared secret
        secret = C[0]

        hash = sha256()
        hash.update(long_to_bytes(secret))

        key = hash.digest()[16:32]
        iv = b'u\x8fo\x9aK\xc5\x17\xa7>[\x18\xa3\xc5\x11\x9en'
        cipher = AES.new(key, AES.MODE_CBC, iv)

        encrypted = cipher.encrypt(pad(FLAG, 16))
        print(encrypted)

        ```

        This looks like standard ECC:
        * An elliptic curve $E$ is defined with a generator point $G$
        * The prime $p$ and constant $b$ are secret
        * $A=aG$ and $B=bG$ are calculated
        * $a$ and $b$ are kept secret, as standard
        * The values are exchanged (printed out) and then both sides calculate $abG$ independently
        * This value is used as a shared secret for encryption

        The values are found in `output.txt`:
        ```
        (6174416269259286934151093673164493189253884617479643341333149124572806980379124586263533252636111274525178176274923169261099721987218035121599399265706997 : 2456156841357590320251214761807569562271603953403894230401577941817844043774935363309919542532110972731996540328492565967313383895865130190496346350907696 : 1)
        (4226762176873291628054959228555764767094892520498623417484902164747532571129516149589498324130156426781285021938363575037142149243496535991590582169062734 : 425803237362195796450773819823046131597391930883675502922975433050925120921590881749610863732987162129269250945941632435026800264517318677407220354869865 : 1)
        b'V\x1b\xc6&\x04Z\xb0c\xec\x1a\tn\xd9\xa6(\xc1\xe1\xc5I\xf5\x1c\xd3\xa7\xdd\xa0\x84j\x9bob\x9d"\xd8\xf7\x98?^\x9dA{\xde\x08\x8f\x84i\xbf\x1f\xab'
        ```
    """,
    "solution_code": """
        from Crypto.Cipher import AES
        from Crypto.Util.number import long_to_bytes
        from hashlib import sha256

        enc_flag = b'V\x1b\xc6&\x04Z\xb0c\xec\x1a\tn\xd9\xa6(\xc1\xe1\xc5I\xf5\x1c\xd3\xa7\xdd\xa0\x84j\x9bob\x9d"\xd8\xf7\x98?^\x9dA{\xde\x08\x8f\x84i\xbf\x1f\xab'

        def decrypt(Q):
            secret = Q[0]

            hash = sha256()
            hash.update(long_to_bytes(secret))

            key = hash.digest()[16:32]
            iv = b'u\x8fo\x9aK\xc5\x17\xa7>[\x18\xa3\xc5\x11\x9en'
            cipher = AES.new(key, AES.MODE_CBC, iv)

            decrypted = cipher.decrypt(enc_flag)
            return decrypted

        # first retrieve p
        # we know 3 points
        # we can eliminate b and get 3 equations, use GCD for p
        A = (6174416269259286934151093673164493189253884617479643341333149124572806980379124586263533252636111274525178176274923169261099721987218035121599399265706997, 2456156841357590320251214761807569562271603953403894230401577941817844043774935363309919542532110972731996540328492565967313383895865130190496346350907696)
        B = (4226762176873291628054959228555764767094892520498623417484902164747532571129516149589498324130156426781285021938363575037142149243496535991590582169062734, 425803237362195796450773819823046131597391930883675502922975433050925120921590881749610863732987162129269250945941632435026800264517318677407220354869865)
        G = (926644437000604217447316655857202297402572559368538978912888106419470011487878351667380679323664062362524967242819810112524880301882054682462685841995367, 4856802955780604241403155772782614224057462426619061437325274365157616489963087648882578621484232159439344263863246191729458550632500259702851115715803253)

        # y^2 = x^3 + 726x + b
        # A.y^2 = A.x^3 + 726*A.x + b
        # B.y^2 = B.x^3 + 726*B.x + b
        # A.y^2 - B.y^2 = A.x^3 + 726*A.x - B.x^3 - 726*B.x
        # A.y^2 - B.y^2 - A.x^3 - 726*A.x + B.x^3 + 726*B.x = 0 mod p
        # A.y^2 - G.y^2 - A.x^3 - 726*A.x + G.x^3 + 726*G.x = 0 mod p

        x1 = A[1]^2 - B[1]^2 - A[0]^3 - 726*A[0] + B[0]^3 + 726*B[0]
        x2 = A[1]^2 - G[1]^2 - A[0]^3 - 726*A[0] + G[0]^3 + 726*G[0]

        p = gcd(x1, x2)
        F = GF(p)
        b = (A[1]^2 - A[0]^3 - 726*A[0]) % p

        # note that order of G is 11
        E = EllipticCurve(F, [726, b])
        G = E(G[0], G[1])
        # print(G.order())

        # brute force the result
        for i in range(1, 12):
            P = i*G
            msg = decrypt(P)

            if b'HTB{' in msg:
                print(msg)
                break
        """,
}
PARTIAL_TENACITY_CHALLENGE = {
    "name": "partial_tenacity",
    "ciphertext": {'n': '118641897764566817417551054135914458085151243893181692085585606712347004549784923154978949512746946759125187896834583143236980760760749398862405478042140850200893707709475167551056980474794729592748211827841494511437980466936302569013868048998752111754493558258605042130232239629213049847684412075111663446003',
                   'ct': '7f33a035c6390508cee1d0277f4712bf01a01a46677233f16387fae072d07bdee4f535b0bd66efa4f2475dc8515696cbc4bc2280c20c93726212695d770b0a8295e2bacbd6b59487b329cc36a5516567b948fed368bf02c50a39e6549312dc6badfef84d4e30494e9ef0a47bd97305639c875b16306fcd91146d3d126c1ea476',
                   'p': '151441473357136152985216980397525591305875094288738820699069271674022167902643',
                   'q': '15624342005774166525024608067426557093567392652723175301615422384508274269305'},
    "source_code": """
        from secret import FLAG
        from Crypto.PublicKey import RSA
        from Crypto.Cipher import PKCS1_OAEP

        class RSACipher:
            def __init__(self, bits):
                self.key = RSA.generate(bits)
                self.cipher = PKCS1_OAEP.new(self.key)
            
            def encrypt(self, m):
                return self.cipher.encrypt(m)

            def decrypt(self, c):
                return self.cipher.decrypt(c)

        cipher = RSACipher(1024)

        enc_flag = cipher.encrypt(FLAG)

        with open('output.txt', 'w') as f:
            f.write(f'n = {cipher.key.n}\\n')
            f.write(f'ct = {enc_flag.hex()}\\n')
            f.write(f'p = {str(cipher.key.p)[::2]}\\n')
            f.write(f'q = {str(cipher.key.q)[1::2]}')
        """,
    "solution": "HTB{v3r1fy1ng_pr1m3s_m0dul0_p0w3rs_0f_10!}",
    "feedback": [],
    "task_directions": """
        # Synopsis

        - In this challenge, the player is called to factor an RSA modulus $n= p \cdot q$, given alternate base-10 digits of the primes $p,q$. The task is to implement a custom algorithm to retrieve the $i$-th redacted digit by checking if $n$ is equal to $p \cdot q$ modulo $10^i$.

        ## Description

        - You find yourself in a labyrinthine expanse where movement is restricted to forward paths only. Each step presents both opportunity and uncertainty, as the correct route remains shrouded in mystery. Your mission is clear: navigate the labyrinth and reach the elusive endpoint. However, there's a twist—you have just one chance to discern the correct path. Should you falter and choose incorrectly, you're cast back to the beginning, forced to restart your journey anew. As you embark on this daunting quest, the labyrinth unfolds before you, its twisting passages and concealed pathways presenting a formidable challenge. With each stride, you must weigh your options carefully, considering every angle and possibility. Yet, despite the daunting odds, there's a glimmer of hope amidst the uncertainty. Hidden throughout the labyrinth are cryptic clues and hints, waiting to be uncovered by the keen-eyed. These hints offer glimpses of the correct path, providing invaluable guidance to those who dare to seek them out. But beware, for time is of the essence, and every moment spent deliberating brings you closer to the brink of failure. With determination and wit as your allies, you must press onward, braving the twists and turns of the labyrinth, in pursuit of victory and escape from the labyrinth's confounding embrace. Are you tenacious enough for that?

        ## Skills Required

        - Basic Python source code analysis.
        - Basic understanding of the RSA cryptosystem.
        - Familiar with modular arithmetic.

        ## Skills Learned

        - Gain more experience with modular arithmetic. More specifically, if two numbers are equal, then they are equal modulo any number $m$.

        # Enumeration

        In this challenge, we are provided with two files:

        - `source.py` : This is the main script that encrypts the flag.
        - `output.txt` : This is a text file which contains some values from the source script.

        ## Analyzing the source code

        Let us first take a look at the source script which is short and easy-to-follow.

        ```python
        from secret import FLAG
        from Crypto.PublicKey import RSA
        from Crypto.Cipher import PKCS1_OAEP

        class RSACipher:
            def __init__(self, bits):
                self.key = RSA.generate(bits)
                self.cipher = PKCS1_OAEP.new(self.key)
            
            def encrypt(self, m):
                return self.cipher.encrypt(m)

            def decrypt(self, c):
                return self.cipher.decrypt(c)

        cipher = RSACipher(1024)

        enc_flag = cipher.encrypt(FLAG)

        with open('output.txt', 'w') as f:
            f.write(f'n = {cipher.key.n}\n')
            f.write(f'ct = {enc_flag.hex()}\n')
            f.write(f'p = {str(cipher.key.p)[::2]}\n')
            f.write(f'q = {str(cipher.key.q)[1::2]}')
        ```

        There is a standard RSA implementation that pads the message with the PKCS#1 OAEP padding scheme. As in any RSA challenge, we are given the modulus $n$ and the encrypted flag $ct$​. Since the RSA key is generated by the PyCryptodome module, its value is [65537](https://github.com/Legrandin/pycryptodome/blob/master/lib/Crypto/PublicKey/RSA.py#L457).

        So far, assuming that RSA is secure, there is no vulnerability to exploit. However we are also provided with two values $p, q$ which look like they are related to the RSA primes $p,q$​. It might be interesting to examine them further.

        Before moving on, let us write a function that loads the data from `output.txt`.

        ```python
        def load_data():
            with open('output.txt') as f:
                n = int(f.readline().split(' = ')[1])
                ct = bytes.fromhex(f.readline().split(' = ')[1])
                hint_p = int(f.readline().split(' = ')[1])
                hint_q = int(f.readline().split(' = ')[1])
                return n, ct, hint_p, hint_q
        ```
    """,
    'solution_code': """
        ## Finding the vulnerability

        Generally, in most CTF challenges, if any portion of the private key is leaked then the solution will most likely be related to recovering the private key. This challenge verifies this observation.

        The Base-10 version of the RSA primes $p,q$ is converted to a string. Then, the even-indexed characters of $p$ and the odd-indexed characters of $q$ are outputted. Let us see how this looks like through an example. Assume $p = 3068913241$ and $q = 2593854881$. Then, the output file would contain the numbers:
        $$
        p = 36934\\
        q = 53581
        $$
        The task is be to recover all the base-10 digits of $p$ and $q$. It turns out that the leak is more than enough to make this task computationally feasible.

        For simplicity, we will use two binary masks for each of $p, q$. This is useful to keep track of which digits are discarded for each prime. When there is an $1$-bit it means that the digit is leaked, if it is a $0$-bit then the digit is discarded.

        For our example above, the corresponding masks would be:
        $$
        p_{mask} = 1010101010\\
        q_{mask} = 0101010101
        $$
        Let us write a function that creates $p_{mask}$ and $q_{mask}$.

        ```python
        def create_masks(primelen):
            pmask = ''.join(['1' if i % 2 == 0 else '0' for i in range(primelen)])
            qmask = ''.join(['1' if i % 2 == 1 else '0' for i in range(primelen)])
                return pmask, qmask
        ```

        Recall that the whole product $n = p \cdot q = 3068913241 \cdot 2593854881 = 7960315589533379321$ is known which is what makes this challenge solveable.

        ### Working modulo powers of 10

        Let $n = 7960315589533379321$. The task is to determine the two factors $p, q$ knowing the masks $p_{mask}, q_{mask}$. We know the last digit of $n$ is a $1$. Let us list all the numbers whose product ends with $1$​.
        $$
        1 \cdot 1 = 1\\
        3 \cdot 7 = 21\\
        7 \cdot 3 = 21\\
        9 \cdot 9 = 81\\
        $$
        Notice how we do not really mind about the $2$-digit products as we keep only the last digit which is a $1$. Therefore, as we mind only about the last digit, we can isolate the rest by reducing the number modulo $10$. This is the key observation to solve the challenge.

        Right now, there are four candidates for the last digit of $p$ and $q$. If the size of $n$ is small, bruteforcing each of these candidates could be a feasible approach but in the context of the challenge, $n$ is $1024$ bits which rules out bruteforcing as the candidates increase exponentially for each digit.

        Back to our example, due to $q_{mask}$, we know that the last digit of $q$ is $1$ so we can quickly determine that the last digit of $p$ would also be $1$. Moving on to the second-to-last digit of $n$ which we know is $2$ which can be produced by the following candidates:
        $$
        1 \cdot 2 = 2\\
        2 \cdot 1 = 2\\
        2 \cdot 6 = 12\\
        3 \cdot 4 = 12\\
        4 \cdot 3 = 12\\
        4 \cdot 8 = 32\\
        6 \cdot 2 = 12\\
        6 \cdot 7 = 42\\
        7 \cdot 6 = 42\\
        8 \cdot 4 = 32\\
        8 \cdot 9 = 72\\
        9 \cdot 8 = 72\\
        $$
        Again, we mind only about the last digit. Due to $p_{mask}$, we know the second-to-last digit of $p$ is $4$, therefore the candidates are reduced to:
        $$
        4 \cdot 3 = 12\\
        4 \cdot 8 = 32\\
        $$
        The problem is that we are not left with one choice which would require us testing each of these candidates. Again, for larger $n$ this would eventually be computationally infeasible.

        So far, we know:

        - The last two digits of $p$ are $41$
        - The last digit of $q$ is $1$
        - The last two digits of $n$ are $21$.
        - The second-to-last digit of $q$ can be either $3$ or $8$.

        We know we have found the correct candidate of $q$ if the product of the last two digits of $p$ and $q$ match the last two digits of $n$. This is equivalent to saying that the product of $p$ and $q$ modulo $10^2$ is equal to $n$ modulo $10^2$.

        Let us compute the products $41 \cdot 31 = 1271$ and $41 \cdot 81 = 3321$.

        For candidate $81$, we see that the product ends with $21$ so this should be the correct candidate. Finally, we have found out that:
        $$
        p = 41\\
        q = 81
        $$
        We can apply this logic for each digit until we have recovered the whole numbers $p$ and $q$.

        The algorithm can be summarized as follows:

        1. For the digit at position $i$, we can extract the $i$-th character of $p_{mask}$ and $q_{mask}$.
        2. If $p_{mask}[i] = 1$, then we know the $i$-th digit of $p$ so extract it.
        1. Bruteforce all 10 possible candidates of $q$.
        2. Check whether $n \pmod {10^i} == (p \cdot q) \pmod {10^i}$.
        3. The correct candidate for $q[i]$ will satisfy this relation.
        3. Else, we know the $i$-th digit of $q$ so extract it and repeat the same steps until all the bits of $p, q$​ are recovered.

        Let us write a function that brute forces the $i$-th digit of $p$ or $q$ and checks whether $n \pmod {10^i} == p \cdot q \pmod {10^i}$ holds.

        ```python
        def bruteforce_digit(i, n, known_prime, prime_to_check, hint_prime):
            msk = 10**(i+1)
            known_prime = 10**i * (hint_prime % 10) + known_prime
            for d in range(10):
                test_prime = 10**i * d + prime_to_check
                if n % msk == known_prime * test_prime % msk:
                    updated_prime_to_check = test_prime			    # correct candidate! update the unknown prime
                    updated_hint_prime = hint_prime // 10			  # move on to the next digit
                    return known_prime, updated_prime_to_check, updated_hint_prime
        ```

        The variable `known_prime` corresponds to the prime whose $i$-th character of the mask is $1$, or in other words, is known. Then, `prime_to_check` will be set to the other one. Similarly with `hint_prime`.

        ## Exploitation

        Let us write a function that iterates over $p_{mask}$ and $q_{mask}$ and checks if the current character is $1$ or $0$. Then it calls `bruteforce_digit` with the corresponding arguments.

        ```python
        def factor(n, p, q, hp, hq, pmask, qmask):
            for i in range(prime_len):
                if pmask[-(i+1)] == '1':
                    p, q, hp = bruteforce_digit(i, n, p, q, hp)
                else:
                    q, p, hq = bruteforce_digit(i, n, q, p, hq)
                    
            assert n == p * q

            return p, q
        ```

        Having factored $n$​, we can decrypt the flag using RSA along with the PKCS#1 OAEP padding scheme.

        ```python
        from Crypto.PublicKey import RSA
        from Crypto.Cipher import PKCS1_OAEP

        def decrypt(p, q, n, ct):
                e = 0x10001
            d = pow(e, -1, (p-1)*(q-1))
            key = RSA.construct((n, e, d))
            flag = PKCS1_OAEP.new(key).decrypt(ct)
            return flag
        ```

        ### Getting the flag

        A final summary of all that was said above:

        1. Notice that we know alternate digits of the decimal representation of $p$ and $q$​.
        2. Create two masks consisting of $0,1$ and show which digits of the two primes are known and which are redacted.
        3. Realize that knowing the $i$-th digit of $p$ is enough to recover the $i$-th digit of $q$ by verifying whether $n$ is equal to $p \cdot q$ modulo $10^i$.
        4. Write a custom algorithm that recovers the primes' digits one by one.
        5. Having factored $n$, decrypt the flag using RSA along with the PKCS#1 OAEP padding scheme.

        This recap can be represented by code with the `pwn()` function:

        ```python
        from math import sqrt

        def pwn():
            n, ct, hint_p, hint_q = load_data()
            prime_len = len(str(int(sqrt(n))))
            pmask, qmask = create_masks(prime_len)
            p, q = factor(n, 0, 0, hint_p, hint_q, pmask, qmask)
            flag = decrypt(p, q, n, ct)
            print(flag)

        if __name__ == '__main__':
            pwn()
        ```
    """,
}

ICED_TEA_CHALLENGE = {
    "name": "Iced Tea",
    "ciphertext": {
        "key": "850c1413787c389e0b34437a6828a1b2", 
        "ct": "b36c62d96d9daaa90634242e1e6c76556d020de35f7a3b248ed71351cc3f3da97d4d8fd0ebc5c06a655eb57f2b250dcb2b39c8b2000297f635ce4a44110ec66596c50624d6ab582b2fd92228a21ad9eece4729e589aba644393f57736a0b870308ff00d778214f238056b8cf5721a843"
    },
    "source_code": """
        import os
        from secret import   FLAG
        from Crypto.Util.Padding import pad
        from Crypto.Util.number import bytes_to_long as b2l, long_to_bytes as l2b
        from enum import Enum

        class Mode(Enum):
            ECB = 0x01
            CBC = 0x02

        class Cipher:
            def __init__(self, key, iv=None):
                self.BLOCK_SIZE = 64
                self.KEY = [b2l(key[i:i+self.BLOCK_SIZE//16]) for i in range(0, len(key), self.BLOCK_SIZE//16)]
                self.DELTA = 0x9e3779b9
                self.IV = iv
                if self.IV:
                    self.mode = Mode.CBC
                else:
                    self.mode = Mode.ECB
            
            def _xor(self, a, b):
                return b''.join(bytes([_a ^ _b]) for _a, _b in zip(a, b))

            def encrypt(self, msg):
                msg = pad(msg, self.BLOCK_SIZE//8)
                blocks = [msg[i:i+self.BLOCK_SIZE//8] for i in range(0, len(msg), self.BLOCK_SIZE//8)]
                
                ct = b''
                if self.mode == Mode.ECB:
                    for pt in blocks:
                        ct += self.encrypt_block(pt)
                elif self.mode == Mode.CBC:
                    X = self.IV
                    for pt in blocks:
                        enc_block = self.encrypt_block(self._xor(X, pt))
                        ct += enc_block
                        X = enc_block
                return ct

            def encrypt_block(self, msg):
                m0 = b2l(msg[:4])
                m1 = b2l(msg[4:])
                K = self.KEY
                msk = (1 << (self.BLOCK_SIZE//2)) - 1

                s = 0
                for i in range(32):
                    s += self.DELTA
                    m0 += ((m1 << 4) + K[0]) ^ (m1 + s) ^ ((m1 >> 5) + K[1])
                    m0 &= msk
                    m1 += ((m0 << 4) + K[2]) ^ (m0 + s) ^ ((m0 >> 5) + K[3])
                    m1 &= msk
                
                m = ((m0 << (self.BLOCK_SIZE//2)) + m1) & ((1 << self.BLOCK_SIZE) - 1) # m = m0 || m1

                return l2b(m)

        if __name__ == '__main__':
            KEY = os.urandom(16)
            cipher = Cipher(KEY)
            ct = cipher.encrypt(FLAG)
            with open('output.txt', 'w') as f:
                f.write(f'Key : {KEY.hex()}\\nCiphertext : {ct.hex()}')
        """,
    "solution": "HTB{th1s_1s_th3_t1ny_3ncryp710n_4lg0r1thm_____y0u_m1ght_h4v3_4lr34dy_s7umbl3d_up0n_1t_1f_y0u_d0_r3v3rs1ng}",
    "feedback": [
        "This challenge involves a block cipher with a 64-bit block size. Look at the source code to understand the encryption process.",
        "The encryption key is provided in the output.txt file. Notice the constant DELTA = 0x9e3779b9, which is a clue to the algorithm's identity.",
        "This block cipher is the Tiny Encryption Algorithm (TEA). Research TEA to understand how to reverse the encryption process.",
        "TEA uses a simple feistel structure. To decrypt, you need to reverse the operations performed during encryption.",
        "For decryption, the sum value starts at 0xc6ef3720 (32 rounds × DELTA) and is decreased by DELTA in each round.",
        "The decryption operations should be applied in reverse order: first update m1, then m0.",
        "Implement TEA decryption in ECB mode, where each 8-byte block is decrypted independently."
    ],
    "task_directions": """
        # Synopsis

        - In this challenge, the player has to determine that the provided block cipher is known as the TEA cipher by researching online for cryptographic constants. Then they can implement their own decrypt function or use Wikipedia's to decrypt the flag using TEA in ECB mode.

        ## Description

        - Locked within a cabin crafted entirely from ice, you're enveloped in a chilling silence. Your eyes land upon an old notebook, its pages adorned with thousands of cryptic mathematical symbols. Tasked with deciphering these enigmatic glyphs to secure your escape, you set to work, your fingers tracing each intricate curve and line with determination. As you delve deeper into the mysterious symbols, you notice that patterns appear in several pages and a glimmer of hope begins to emerge. Time is flying and the temperature is dropping, will you make it before you become one with the cabin?

        ## Skills Required

        - Basic Python source code analysis.
        - Basic research skills.
        - Know how to identify cryptographic algorithms.
        - Familiar with bitwise operations.

        ## Skills Learned

        - Learn to research for cryptographic constants for cryptosystem identification.
        - Understand how to perform TEA-ECB decryption.

        # Enumeration

        In this challenge, we are provided with two files:

        - `source.py` : This is the main script that encrypts the flag.
        - `output.txt` : This is the output file that contains the encryption key and the encrypted flag, both in hex format.

        ## Analyzing the source code

        Let us first inspect how the flag is encrypted. The main function of the source script is straight forward to follow.

        ```python
        if __name__ == '__main__':
            KEY = os.urandom(16)
            cipher = Cipher(KEY)
            ct = cipher.encrypt(FLAG)
            with open('output.txt', 'w') as f:
                f.write(f'Key : {KEY.hex()}\nCiphertext : {ct.hex()}')
        ```

        First, a random 16-byte key is generated which is used as the encryption key for the provided cipher. Then the flag is encrypted and finally both the encryption key and the encrypted flag are written to the output file. Let us take a closer look at the Cipher class.

        ```python
        from enum import Enum

        class Mode(Enum):
            ECB = 0x01
            CBC = 0x02

        class Cipher:
            def __init__(self, key, iv=None):
                self.BLOCK_SIZE = 64
                self.KEY = [b2l(key[i:i+self.BLOCK_SIZE//16]) for i in range(0, len(key), self.BLOCK_SIZE//16)]
                self.DELTA = 0x9e3779b9
                self.IV = iv
                if self.IV:
                    self.mode = Mode.CBC
                else:
                    self.mode = Mode.ECB
            
            def _xor(self, a, b):
                # ...

            def encrypt(self, msg):
                # ...

            def encrypt_block(self, msg):
                # ...

        ```

        At first glance, we can derive the following information:

        1. From the naming used in this class, we can conclude that the cipher is a block cipher where each block is 64 bits.
        2. The cipher supports two modes, ECB and CBC. In case an IV is provided, the mode is automatically set to CBC.
        3. The key is splitted in $\dfrac{64}{16}=4$ dwords (i.e. four quadratuples of bytes).

        Since in this challenge, there is no IV provided. We know that the flag is encrypted with the cipher in ECB mode so let us take a look how ECB is implemented and how each block is encrypted.

        ```python
        def encrypt(self, msg):
            msg = pad(msg, self.BLOCK_SIZE//8)
            blocks = [msg[i:i+self.BLOCK_SIZE//8] for i in range(0, len(msg), self.BLOCK_SIZE//8)]

            ct = b''
            if self.mode == Mode.ECB:
                for pt in blocks:
                    ct += self.encrypt_block(pt)
            elif self.mode == Mode.CBC:
                    # ...
                
                return ct
        ```

        The ECB mode implementation is trivial. Each plaintext block is encrypted and concatenated to the ciphertext as it is.

        ```python
        def encrypt_block(self, msg):
            m0 = b2l(msg[:4])
            m1 = b2l(msg[4:])
            K = self.KEY
            msk = (1 << (self.BLOCK_SIZE//2)) - 1

            s = 0
            for i in range(32):
                s += self.DELTA
                m0 += ((m1 << 4) + K[0]) ^ (m1 + s) ^ ((m1 >> 5) + K[1])
                m0 &= msk
                m1 += ((m0 << 4) + K[2]) ^ (m0 + s) ^ ((m0 >> 5) + K[3])
                m1 &= msk

            m = ((m0 << (self.BLOCK_SIZE//2)) + m1) & ((1 << self.BLOCK_SIZE) - 1) # m = m0 || m1

            return l2b(m)
        ```

        As the encryption key is provided to us there is no need to break the cipher. All we have to do is figure out how to perform the reverse operations and decrypt each ciphertext block. At first glance, this might seem a complicated task to do as there are several bitwise operations involved. At this point, the player can move on by doing any of the following.

        1. Ask ChatGPT or any other similar tool to write them a decrypt_block function, which is something we do not recommend.
        2. Write their own decrypt_block function by reversing the operations. This might take some time but will eventually work and the player will become familiar with bitwise operations.
        3. Realize that this is an easy challenge so there should already exist a decrypt_block function somewhere.

        We will showcase the third option. Before moving on, let us write a function that loads the encryption key and the encrypted flag from `output.txt`.

        ```python
        def load_data():
            with open('output.txt') as f:
                key = bytes.fromhex(f.readline().split(' : ')[1])
                enc_flag = bytes.fromhex(f.readline().split(' : ')[1])
                return key, enc_flag
        ```
        """,
    "solution_code": """
        # Solution

        ## Finding the vulnerability

        As aforementioned, the key is already provided so we do not really have to exploit any vulnerability of the cipher. However, we do need a starting point when it comes to researching for ciphers online. Something that stands out in the cipher implementation and could be considered unique is the utilization of the DELTA constant `0x9e3779b9`.

        Indeed, if, for example, we research this constant online along with the key words "block cipher", we will stumble upon a known block cipher known as the TEA (Tiny Encryption Algorithm) block cipher. By inspecting the corresponding Wikipedia, we can see that the encryption routine matches the one provided in this challenge. This is a strong sign that we have successfully identified the provided cipher.

        ## Exploitation

        Wikipedia also provides us with the decrypt function which can be used as it is to decrypt the flag. One can use the C language to decrypt it or reimplement it in Python which is what we are going to do.

        ```python
        from Crypto.Util.number import bytes_to_long as b2l, long_to_bytes as l2b

        def decrypt_block(key, ct):
            m0 = b2l(ct[:4])
            m1 = b2l(ct[4:])
            msk = (1 << 32) - 1

            s = 0xc6ef3720

            for i in range(32):
                m1 -= ((m0 << 4) + key[2]) ^ (m0 + s) ^ ((m0 >> 5) + key[3])
                m1 &= msk
                m0 -= ((m1 << 4) + key[0]) ^ (m1 + s) ^ ((m1 >> 5) + key[1])
                m0 &= msk
                s -= DELTA

            m = ((m0 << 32) + m1) & ((1 << 64) - 1)

            return l2b(m)
        ```

        Now, we can implement the equivalent TEA-ECB decryption routine and obtain the flag.

        ```python
        def tea_ecb_decrypt(key, enc_flag):
            key = [b2l(key[i:i+4]) for i in range(0, len(key), 4)]
            blocks = [enc_flag[i:i+8] for i in range(0, len(enc_flag), 8)]
            flag = b''

            for ct in blocks:
                flag += decrypt_block(key, ct)
            
            return flag
        ```

        ### Getting the flag

        A final summary of all that was said above:

        1. Notice that the provided cipher is a block cipher with a block size of 64 bits.
        2. The block cipher encrypts the flag in ECB mode as no IV is provided.
        3. Notice the cryptographic constant `0x9e3779b9` and research online with the proper keywords to find out the TEA cipher.
        4. Take Wikipedia's implementation of the decrypt function or implement it from scratch and decrypt the flag by running TEA in ECB mode.

        This recap can be represented by code with the `pwn()` function:

        ```python
        def pwn():
            key, enc_flag = load_data()
            flag = tea_ecb_decrypt(key, enc_flag)
            print(flag)

        if __name__ == '__main__':
                pwn()
        ```
        """,
}
BLUNT_CHALLENGE = {
    "name": "Blunt",
    "ciphertext": {
        "p": "0xdd6cc28d",
        "g": "0x83e21c05",
        "A": "0xcfabb6dd", 
        "B": "0xc4a21ba9",
        "ct": b'\x94\x99\x01\xd1\xad\x95\xe0\x13\xb3\xacZj{\x97|z\x1a(&\xe8\x01\xe4Y\x08\xc4\xbeN\xcd\xb2*\xe6{',
        "iv": b'\xc1V2\xe7\xed\xc7@8\xf9\\\xef\x80\xd7\x80L*'
    },
    "source_code": """
        from Crypto.Cipher import AES
        from Crypto.Util.Padding import pad
        from Crypto.Util.number import getPrime, long_to_bytes
        from hashlib import sha256

        from secret import FLAG

        import random


        p = getPrime(32)
        print(f'p = 0x{p:x}')

        g = random.randint(1, p-1)
        print(f'g = 0x{g:x}')

        a = random.randint(1, p-1)
        b = random.randint(1, p-1)

        A, B = pow(g, a, p), pow(g, b, p)

        print(f'A = 0x{A:x}')
        print(f'B = 0x{B:x}')

        C = pow(A, b, p)
        assert C == pow(B, a, p)

        # now use it as shared secret
        hash = sha256()
        hash.update(long_to_bytes(C))

        key = hash.digest()[:16]
        iv = b'\\xc1V2\\xe7\\xed\\xc7@8\\xf9\\\\\\xef\\x80\\xd7\\x80L*'
        cipher = AES.new(key, AES.MODE_CBC, iv)

        encrypted = cipher.encrypt(pad(FLAG, 16))
        print(f'ciphertext = {encrypted}')
        """,
    "solution": "HTB{y0u_n3ed_a_b1gGeR_w3ap0n!!}",
    "task_directions": """
        # Synopsis
        Blunt is an Easy crypto challenge that involves a small $p$ value, meaning it is easy to solve the DLP to retrieve the private exponents.

        ## Description:
        Valuing your life, you evade the other parties as much as you can, forsaking the piles of weaponry and the vantage points in favour of the depths of the jungle. As you jump through the trees and evade the traps lining the forest floor, a glint of metal catches your eye. Cautious, you creep around, careful not to trigger any sensors. Lying there is a knife - damaged and blunt, but a knife nonetheless. You're not helpless any more.

        ## Skills Required
        - Basic understanding of the Diffie-Hellman Key Exchange

        ## Skills Learned
        - Solving the DLP in small groups
        - Using SageMath

        # Enumeration
        We are given the following script:
        ```py
        from Crypto.Cipher import AES
        from Crypto.Util.Padding import pad
        from Crypto.Util.number import getPrime, long_to_bytes
        from hashlib import sha256

        from secret import FLAG

        import random


        p = getPrime(32)
        print(f'p = 0x{p:x}')

        g = random.randint(1, p-1)
        print(f'g = 0x{g:x}')

        a = random.randint(1, p-1)
        b = random.randint(1, p-1)

        A, B = pow(g, a, p), pow(g, b, p)

        print(f'A = 0x{A:x}')
        print(f'B = 0x{B:x}')

        C = pow(A, b, p)
        assert C == pow(B, a, p)

        # now use it as shared secret
        hash = sha256()
        hash.update(long_to_bytes(C))

        key = hash.digest()[:16]
        iv = b'\xc1V2\xe7\xed\xc7@8\xf9\\\xef\x80\xd7\x80L*'
        cipher = AES.new(key, AES.MODE_CBC, iv)

        encrypted = cipher.encrypt(pad(FLAG, 16))
        print(f'ciphertext = {encrypted}')
        ```

        This looks like a typical Diffie-Hellman Key Exchange:
        * A generator $g$ is defined in a finite field $F_p$
        * The private exponents $a$ and $b$ are kept secret
        * $g$ is put to the power of $a$ and $b$ to form $A$ and $B$ respectively
        * The values are exchanged (printed out) and then both sides calculate put the values they receive to their private power to calculate $g^{ab}$
        * This value is used as a shared secret for encryption

        The values are found in `output.txt`:
        ```
        p = 0xdd6cc28d
        g = 0x83e21c05
        A = 0xcfabb6dd
        B = 0xc4a21ba9
        ciphertext = b'\x94\x99\x01\xd1\xad\x95\xe0\x13\xb3\xacZj{\x97|z\x1a(&\xe8\x01\xe4Y\x08\xc4\xbeN\xcd\xb2*\xe6{'
        """,
    "feedback": [
        "This challenge involves the Diffie-Hellman key exchange protocol with a small prime modulus. The Baby-step Giant-step algorithm should use a dictionary/hash table for logarithm lookups and require O(sqrt(p)) operations rather than O(p).",
        "You need to find the private exponent 'a' by solving g^a ≡ A (mod p). This is called the discrete logarithm problem.",
        "For small primes like this one, you can implement the Baby-step Giant-step algorithm to efficiently find 'a'.",
        "The Baby-step Giant-step algorithm uses a time-memory tradeoff with complexity O(sqrt(p)).",
        "Once you have 'a', you can compute the shared secret C = B^a mod p.",
        "Finally, use the shared secret to derive the AES key and decrypt the ciphertext."
    ],
    "solution_code": """
        from Crypto.Cipher import AES
        from Crypto.Util.number import long_to_bytes
        from Crypto.Util.Padding import unpad
        from hashlib import sha256

        p = 0xdd6cc28d
        F = GF(p)

        g = F(0x83e21c05)
        A = F(0xcfabb6dd)
        B = F(0xc4a21ba9)
        ciphertext = b'\x94\x99\x01\xd1\xad\x95\xe0\x13\xb3\xacZj{\x97|z\x1a(&\xe8\x01\xe4Y\x08\xc4\xbeN\xcd\xb2*\xe6{'

        # get a, and from there C
        a = discrete_log(A, g)
        C = B^a

        # decrypt
        hash = sha256()
        hash.update(long_to_bytes(int(C)))

        key = hash.digest()[:16]
        iv = b'\xc1V2\xe7\xed\xc7@8\xf9\\\xef\x80\xd7\x80L*'
        cipher = AES.new(key, AES.MODE_CBC, iv)

        decrypted = cipher.decrypt(ciphertext)
        flag = unpad(decrypted, 16)
        print(flag)""",
}

# Add to CHALLENGE_DATA dictionary
CHALLENGE_DATA = {
    "dynastic": DYNASTIC_CHALLENGE,
    "makeshift": MAKESHIFT_CHALLENGE,
    "arranged": ARRANGED_CHALLENGE,
    "partial_tenacity": PARTIAL_TENACITY_CHALLENGE, 
    "primary_knowledge": PRIMARY_KNOWLEDGE_CHALLENGE,
    "iced_tea": ICED_TEA_CHALLENGE,
    "blunt": BLUNT_CHALLENGE
} 