from opto import trace
import subprocess
import re

from opto.trace.operators import call_llm

@trace.model
class SimpleDecryptorAgent:

   """
   This class initializer reads the contents of specific files and stores them 
   in string variables for later access. 

   Args:
      None

   Attributes:
      source_file (str): The contents of the `source.py` file, read as a string.
      output (str): The contents of the `output.txt` file, read as a string.

   Returns:
      None (since this is an initializer method)
      
   Note:
      - The method reads files directly from the specified file paths.
      - Ensure that the files exist at the given paths; otherwise, this will raise a `FileNotFoundError`.
      - The files are read in their entirety and stored in memory as strings.
   """
   def __init__(self):
      self.source_file = open("/Users/anonymous/trace-security/security/crypto/partial-tenacity/resources/source.py").read()
      self.output = open("/Users/anonymous/trace-security/security/crypto/partial-tenacity/resources/output.txt").read()

   """
   The run method. This is the collective list of actions performed 
   by the agent in order to achieve the task provided.

   Task: In this case, the task is to decrypt the flag utilized in source.py, which 
   is contained in secret.py. The program does this by taking in the string values
   of the source files and the output, compiling an initial report on the content of such strings.
   """
   def run(self):
      print("RUN METHOD INITIATED")
      analysis_report = self.analyze_source_file(self.source_file, self.output)
      return self.decrypt(analysis_report, self.source_file, self.output)
   
   """
   Analyzes the given source file for vulnerabilities using an AI model.

   Args:
      source_file (str): The contents of the source code file to be analyzed.
      execution_output (str): The output produced by executing the source file, 
                              which may provide additional context for analysis.

   Returns:
      str: The response from the AI model containing an analysis of potential 
            vulnerabilities, including a chain of thought reasoning.

   Process:
      1. Prints a message indicating that the source file is being analyzed.
      2. Constructs a prompt instructing the AI to analyze vulnerabilities in the given source code.
      3. Calls the AI model with the prompt and the concatenated source code and execution output.
      4. Returns the AI-generated analysis.

   Note:
      - The effectiveness of the analysis depends on the AI model's ability to interpret
         the source code and execution output.
      - Ensure `trace.node()` and `call_llm()` are correctly implemented and accessible.
   """
   def analyze_source_file(self, source_file, execution_output):
      print("ANALYZING SOURCE FILE")
      analyze_prompt = trace.node("You are a cybersecurity agent, analyze the vunerability in this source file. Give me chain of thought.")
      return call_llm(analyze_prompt, source_file + '\n\nExecution output:' + execution_output)
   
   """
   Extracts the values of `n`, `ct`, `p`, and `q` from the given output file content.

   Args:
      output_file (str): The contents of the output file as a string.

   Returns:
      tuple: A tuple containing three extracted values:
         - n (str): The value of `n` extracted from the content.
         - ct (str): The value of `ct` extracted from the content.
         - p (str): The value of `p` extracted from the content.
         - q (str): The value of `q` extracted from the content.

   Process:
      1. Locates the positions of `n =`, `ct =`, `p =`, and `q =` in the file content.
      2. Extracts and trims the corresponding values between these markers.
      3. Returns the extracted values as a tuple.

   Note:
      - Assumes the format of the content follows a specific structure with `n =`, `e =`, and `c =` appearing in order.
      - If any of these markers are missing, the method may produce unexpected results.
   """
   def output_extraction(self, output_file):
      content = output_file
      n_start = content.find('n =') + 3
      ct_start = content.find('ct =')
      p_start = content.find('p =')
      q_start = content.find('q =')
      
      n = content[n_start:ct_start].strip()
      ct = content[ct_start + 4:p_start].strip()
      p = content[p_start + 3:q_start].strip()
      q = content[q_start + 3:].strip()
      return n, ct, p, q
   
   @trace.bundle(trainable=True)
   def decrypt(self, analysis_report, source_file, execution_output):
      """
      Decrypts an encrypted message using pieces of RSA parameters extracted from execution output.

      This function analyzes the execution output to extract RSA parameters and attempts 
      to decrypt the encrypted message. It utilizes `output_extraction()` to retrieve 
      the modulus (n), ciphertext (ct), portions of p (p) and portions of q (q). 

      Args:
         analysis_report (str): The report generated from the analysis process, potentially 
                                 containing metadata or insights relevant to encryption.
         source_file (str): The Python source file that contains the encryption logic.
         execution_output (str): The output generated during execution, from which 
                                 encryption parameters are extracted.

      Returns:
         bytes: The decrypted flag if successful; otherwise, returns None.
      
      If you write helper functions, make sure that they are embedded in this function.
      Do not add them to the class. Otherwise, the program may not run correctly.
      Double check for syntax errors - it is better that a program run incorrectly
      than not run at all.

      Make sure you avoid the following error: 
      (SyntaxError) unexpected character after line continuation character
      """
      from Crypto.PublicKey import RSA
      from Crypto.Cipher import PKCS1_OAEP

      def construct_modified_p(original_p: int) -> int:
         """Modifies prime p by inserting zeroes between its digits."""
         p_digits = []
         for digit in str(original_p):
            p_digits.append(digit)
            p_digits.append('0')
         return int(''.join(p_digits[:-1]))

      def construct_modified_q(original_q: int) -> int:
         """Modifies prime q by inserting zeroes before and between its digits."""
         q_digits = []
         for digit in str(original_q):
            q_digits.append('0')
            q_digits.append(digit)
         q_digits.append('0')
         return int(''.join(q_digits))

      def refine_p_q_to_match_n(n: int, p: int, q: int) -> tuple:
         """Refines the values of p and q by incrementing their digits until p * q equals n."""
         q_digits = list(str(q))
         for i in range(len(q_digits)):
            if i % 2 == 0:
                  while n % (10 ** (i + 1)) != (p * q) % (10 ** (i + 1)):
                     q += 10 ** i
            else:
                  while n % (10 ** (i + 1)) != (p * q) % (10 ** (i + 1)):
                     p += 10 ** i
         assert p * q == n, "Refined p and q do not satisfy p * q = n"
         return p, q

      def compute_private_exponent(e: int, p: int, q: int) -> int:
         """Computes the private exponent d given e, p, and q."""
         phi_n = (p - 1) * (q - 1)
         return pow(e, -1, phi_n)

      def decrypt_rsa_ciphertext(n: int, e: int, d: int, ciphertext: bytes) -> str:
         """Decrypts an RSA-encrypted ciphertext using the private exponent d."""
         cipher = PKCS1_OAEP.new(RSA.construct((n, e, d)))
         plaintext = cipher.decrypt(ciphertext)
         return plaintext.decode()
      n, ct, p, q = self.output_extraction(execution_output)
      flag = None
