# Refer to https://github.com/pytorch/examples/blob/main/mnist/main.py

from __future__ import print_function
import argparse
import os
import torch
import torch.nn as nn
import torch.nn.functional as F
import torch.optim as optim
from torchvision import datasets, transforms
from torch.optim.lr_scheduler import StepLR
from models import MadryNet

from torchattacks import PGD

CEloss = nn.CrossEntropyLoss()

def train(args, model, device, train_loader, optimizer, epoch, adv_train=False):
    model.train()
    if adv_train:
        atk = PGD(model, eps=76/255, alpha=8/225, steps=10, random_start=True)
    for batch_idx, (data, target) in enumerate(train_loader):
        data, target = data.to(device), target.to(device)
        if adv_train:
            data = atk(data, target)
        optimizer.zero_grad()
        output = model(data)
        loss = CEloss(output, target)
        loss.backward()
        optimizer.step()
        if batch_idx % args.log_interval == 0:
            print('Train Epoch: {} [{}/{} ({:.0f}%)]\tLoss: {:.6f}'.format(
                epoch, batch_idx * len(data), len(train_loader.dataset),
                100. * batch_idx / len(train_loader), loss.item()))
            if args.dry_run:
                break


def test(model, device, test_loader):
    model.eval()
    test_loss = 0
    correct = 0
    with torch.no_grad():
        for data, target in test_loader:
            data, target = data.to(device), target.to(device)
            output = model(data)
            test_loss += CEloss(output, target).item() * target.shape[0]  # sum up batch loss
            pred = output.argmax(dim=1, keepdim=True)  # get the index of the max log-probability
            correct += pred.eq(target.view_as(pred)).sum().item()

    test_loss /= len(test_loader.dataset)

    print('\nTest set: Average loss: {:.4f}, Accuracy: {}/{} ({:.0f}%)\n'.format(
        test_loss, correct, len(test_loader.dataset),
        100. * correct / len(test_loader.dataset)))


def main():

    OUTPUT_PATH = os.path.join(".", "saved_models")
    DATASET_PATH = os.path.join(".", "data")
    os.makedirs(OUTPUT_PATH, exist_ok=True)

    # Training settings
    parser = argparse.ArgumentParser(description='PyTorch MNIST / FashionMNIST Example')
    parser.add_argument('--batch-size', type=int, default=64, metavar='N',
                        help='input batch size for training (default: 64)')
    parser.add_argument('--test-batch-size', type=int, default=1000, metavar='N',
                        help='input batch size for testing (default: 1000)')
    parser.add_argument('--epochs', type=int, default=14, metavar='N',
                        help='number of epochs to train (default: 14)')
    parser.add_argument('--lr', type=float, default=1e-4, metavar='LR',
                        help='learning rate (default: 1e-4)')
    parser.add_argument('--gamma', type=float, default=0.7, metavar='M',
                        help='Learning rate step gamma (default: 0.7)')
    parser.add_argument('--no-cuda', action='store_true', default=False,
                        help='disables CUDA training')
    parser.add_argument('--no-mps', action='store_true', default=False,
                        help='disables macOS GPU training')
    parser.add_argument('--dry-run', action='store_true', default=False,
                        help='quickly check a single pass')
    parser.add_argument('--seed', type=int, default=1, metavar='S',
                        help='random seed (default: 1)')
    parser.add_argument('--log-interval', type=int, default=10, metavar='N',
                        help='how many batches to wait before logging training status')
    parser.add_argument('--save-model', action='store_true', default=True,
                        help='For Saving the current Model')
    parser.add_argument('--adv-train', action='store_true', default=False,
                        help='Adversarial training')
    args = parser.parse_args()
    use_cuda = not args.no_cuda and torch.cuda.is_available()
    use_mps = not args.no_mps and torch.backends.mps.is_available()

    torch.manual_seed(args.seed)

    if use_cuda:
        device = torch.device("cuda")
    elif use_mps:
        device = torch.device("mps")
    else:
        device = torch.device("cpu")

    train_kwargs = {'batch_size': args.batch_size, 'shuffle': True}
    test_kwargs = {'batch_size': args.test_batch_size, 'shuffle': False}
    if use_cuda:
        cuda_kwargs = {'num_workers': 0,
                       'pin_memory': True}
                       # 'pin_memory': True,
                       # 'shuffle': True}
        train_kwargs.update(cuda_kwargs)
        test_kwargs.update(cuda_kwargs)

    transform=transforms.Compose([
        transforms.ToTensor(),
        # transforms.Normalize((0.5,), (0.5,))
        ])

    dataset1 = datasets.MNIST(DATASET_PATH, train=True, download=True,
                       transform=transform)
    dataset2 = datasets.MNIST(DATASET_PATH, train=False,
                       transform=transform)

    # dataset1 = datasets.FashionMNIST(DATASET_PATH, train=True, download=True,
    #                    transform=transform)
    # dataset2 = datasets.FashionMNIST(DATASET_PATH, train=False,
    #                    transform=transform)
    train_loader = torch.utils.data.DataLoader(dataset1,**train_kwargs)
    test_loader = torch.utils.data.DataLoader(dataset2, **test_kwargs)

    model = MadryNet().to(device)
    optimizer = optim.Adam(model.parameters(), lr=args.lr)

    scheduler = StepLR(optimizer, step_size=1, gamma=args.gamma)
    for epoch in range(1, args.epochs + 1):
        train(args, model, device, train_loader, optimizer, epoch, args.adv_train)
        test(model, device, test_loader)
        scheduler.step()

    if args.save_model:
        if args.adv_train:
            torch.save(model.state_dict(), os.path.join(OUTPUT_PATH, "MNIST_adv_cnn.pt"))
        else:
            torch.save(model.state_dict(), os.path.join(OUTPUT_PATH, "MNIST_cnn.pt"))


if __name__ == '__main__':
    main()

