# generate random weights for contribution from the latent processes 
gen.coef <- function(idx, p) {   
  #idx is a vector created with c(), contains the index of the bands that contribute the most
  #1 - delta, 2 - theta, 3 - alpha, 4 - beta, 5 - gamma
  #p is the sum of the coefficients corresponding to the bands on idx
  
  coeffs <- numeric(5)
  w1 <- p/length(idx)
  eps1 <- 1e-1
  
  
  for (i in idx) {
    coeffs[i] <- ifelse(i != max(idx), runif(1, w1 - eps1, w1 + eps1), 
                        p - sum(coeffs[idx[-length(idx)]]))
  }
  if (p < 1) {
    idxc <- (1:5)[-idx]
    w2 <- (1-p)/(5 - length(idx))
    eps2 <- 1e-2
    for (i in idxc) {
      coeffs[i] <- ifelse(i != max(idxc), runif(1, w2 - eps2, w2 + eps2), 
                          1 - p - sum(coeffs[idxc]))
    }
  }
  
  coeffs   
}


## filter function for obtaining components at required frequency bands
filter_out <- function(X, fsband,samprate =100, method = "butter",twosided = FALSE){
  P <- ncol(X)
  T <- nrow(X)
  filtered_X <- array(0, dim = c(T, P))
  if(method == "butter"){
    filter_coeffs <- signal::butter(4, fsband/(0.5 * samprate), type="pass")
  } else if(method == "fir"){
    filter_coeffs <- signal::fir1(100, fsband/(0.5 * samprate), type="pass")
  }
if(twosided){
  for (i in 1:P){
  filtered_X[,i] <- as.numeric(signal::filtfilt(filter_coeffs, X[,i]))
  }
} else {
  for (i in 1:P){
  filtered_X[,i] <- as.numeric(signal::filter(filter_coeffs, X[,i]))
  }
}
  return(filtered_X)
}


## calculate the Fourier spectrum of X and Y
lsp_spectrum <- function(filtered_X, filtered_Y, window_size){
  T <- length(filtered_X)
  spectrum <- numeric(T)
  for (i in (window_size+1):(T- window_size)){
    spectrum[i] <- cov(filtered_X[(i-window_size):(i+window_size)],filtered_Y[(i-window_size):(i+window_size)])
  }
  spectrum[1:window_size] <- spectrum[window_size+1]
  spectrum[(T- window_size+1):T] <- spectrum[T-window_size]
  return(spectrum)
}


## calculate the Fourier spectrum of combined X and Y (Z in the context of paper)
lsp_spectrum_z <- function(mv_filtered_X, mv_filtered_Y, window_size){
  T <- nrow(mv_filtered_X)
  Z <- cbind(mv_filtered_X,mv_filtered_Y)
  P <- ncol(Z)
  Z_spectrum <- array(0, dim=c(P,P,T))
  for (i in 1:P){
    for (k in 1:P){
      Z_spectrum[i,k,] <- lsp_spectrum(Z[,i],Z[,k],window_size)
    }
  }
  return(Z_spectrum)
}


## partitioning the spectrum for coherecne calculation 
extract_spectrum_lsp <- function(spectrum,size_XY){
  dim_spec <- dim(spectrum)
  nrows <- dim_spec[1]
  ncols <- dim_spec[2]
  T <- dim_spec[3]
  Px <- size_XY[1]
  Py <- size_XY[2]
  S_xx <- spectrum[1:Px, 1:Px,]
  S_xy <- spectrum[1:Px,(Px+1):ncols,]
  S_yx <- spectrum[(Px+1):nrows,1:Px,]
  S_yy <- spectrum[(Px+1):nrows,(Px+1):ncols,]
  S <- list(Sxx=S_xx,Sxy=S_xy,Syx=S_yx,Syy=S_yy)
  return(S)
}

## Canonical coherence with LSP (STFT-based method)
WaveCan_lsp <- function(Sxx,Sxy,Syx,Syy){
  T <- dim(Sxx)[3]
  Px <- dim(Sxx)[1] # dim of group X
  Py <- dim(Syy)[1] # dim of group Y
  a <- array(0,dim = c(Px,T))
  b <- array(0,dim = c(Py,T))
  rho <- numeric(T)
  
    for (t in 1:T){
      inv_Sxx <- solve(Sxx[,,t])
      inv_Syy <- solve(Syy[,,t])
      A <- inv_Sxx %*% Sxy[,,t] %*% inv_Syy %*% Syx[,,t] 
      B <- inv_Syy %*% Syx[,,t] %*% inv_Sxx %*% Sxy[,,t] 
      eig_A <- eigen(A)
      eig_B <- eigen(B)
      rho[t] <- Re(eig_A$values[1]) # take the real part to avoid numerical issues, the imaginary parts are zero indeed
      a[,t] <- Re(eig_A$vectors[,1])
      b[,t] <- Re(eig_B$vectors[,1])
    }
  waveCan <- list(rho = rho, a = a, b = b)
  return(waveCan)
}


# functions for generating the latent AR(2) process
gen_latent_ar2 <- function(peaklocation, sharpness = 0.05 ,samprate = 100, lenT=T){
  psi <- peaklocation/samprate
  M <- exp(sharpness)
  phi1 <- (2/M)*cos(2*pi*psi)
  phi2 <- -(1/(M^2))
  Z <- arima.sim(lenT, model = list(ar = c(phi1, phi2)))
  Z <- Z - mean(Z) 
  return(Z)
}



gen_ar2_diff_bands <- function(peaklocation_vector, sharpness_vector, samprate=100,lenT=T){
  ar2_group <- array(0, dim = c(lenT , length(peaklocation_vector)))
  for (i in 1:length(peaklocation_vector)){
    ar2_group[,i] <- gen_latent_ar2(peaklocation_vector[i],sharpness_vector[i],samprate, lenT)
  }
  return(ar2_group)
}




#mix_AR2 <- generate_multiple_mix_AR2(50,4,3)

# Function to inject partially shared component
inject_partial_shared_component <- function(Zx, Zy, component_idx,
                                            alpha = 0.7, beta = 0.6,
                                            freq = 37.5, sharpness = 0.075) {
  T <- nrow(Zx)
  half_T <- T %/% 2
  
  # Generate components
  Z_shared <- gen_latent_ar2(freq, sharpness, lenT = T)
  Z_private_X <- gen_latent_ar2(freq, sharpness, lenT = T)
  Z_private_Y <- gen_latent_ar2(freq, sharpness, lenT = T)
  
  # Combine shared + private for first half only
  Zx[1:half_T, component_idx] <- alpha * Z_shared[1:half_T] + (1 - alpha) * Z_private_X[1:half_T]
  Zy[1:half_T, component_idx] <- beta  * Z_shared[1:half_T] + (1 - beta)  * Z_private_Y[1:half_T]
  
  return(list(Zx = Zx, Zy = Zy))
}


#' Generate AR(2) Replicates with Partially Shared Gamma Component
#'
#' @param nrep Number of replicates to generate
#' @param T Length of each time series
#' @param samprate Sampling rate
#' @param peak_X, sharp_X, peak_Y, sharp_Y Frequency and sharpness vectors for X and Y
#' @param alpha, beta, shared_freq, shared_sharp Parameters for shared gamma component
#' @return A list of length nrep, each with $X and $Y matrices (T x Px and T x Py)
generate_replicates_AR <- function(
    nrep,
    T = 1024,
    samprate = 100,
    peak_X = c(2,6,10,17.5,37.5),
    sharp_X = c(0.03,0.03,0.03,0.05,0.075),
    peak_Y = c(2,6,10,17.5,37.5),
    sharp_Y = c(0.03,0.03,0.03,0.05,0.075),
    alpha = 0.7, beta = 0.6,
    shared_freq = 37.5,
    shared_sharp = 0.075
) {
  Px <- 4; Py <- 3
  half_T <- T %/% 2
  
  # Helper: inject partially shared gamma into latent Zx, Zy
  inject_shared <- function(Zx, Zy) {
    Z_shared <- gen_latent_ar2(shared_freq, shared_sharp, samprate, T)
    Z_private_X <- gen_latent_ar2(shared_freq, shared_sharp, samprate, T)
    Z_private_Y <- gen_latent_ar2(shared_freq, shared_sharp, samprate, T)
    # Overwrite gamma (column 5) in first half
    Zx[1:half_T, 5] <- alpha * Z_shared[1:half_T] + (1 - alpha) * Z_private_X[1:half_T]
    Zy[1:half_T, 5] <- beta  * Z_shared[1:half_T] + (1 - beta)  * Z_private_Y[1:half_T]
    list(Zx = scale(Zx), Zy = scale(Zy))
  }
  
  # Helper: build coefficient matrices
  build_coeffs <- function() {
    list(
      X2 = cbind(
        gen.coef(5, 0.95),
        gen.coef(5, 0.90),
        gen.coef(c(1,2), 1),
        gen.coef(c(1,2,3), 1)
      ),
      X1 = cbind(
        gen.coef(c(2,3), 0.95),
        gen.coef(3, 0.80),
        gen.coef(1, 0.9),
        gen.coef(c(3,2), 0.85)
      ),
      Y2 = cbind(
        gen.coef(5, 0.95),
        gen.coef(5, 0.90),
        gen.coef(c(2,3), 1)
      ),
      Y1 = cbind(
        gen.coef(4, 0.9),
        gen.coef(3, 0.95),
        gen.coef(1, 0.9)
      )
    )
  }
  
  replicates <- vector("list", nrep)
  for (i in seq_len(nrep)) {
    # 1. Simulate latent bands
    Zx <- gen_ar2_diff_bands(peak_X, sharp_X, samprate, T)
    Zy <- gen_ar2_diff_bands(peak_Y, sharp_Y, samprate, T)
    
    # 2. Inject shared gamma to first half
    shared <- inject_shared(Zx, Zy)
    Zx <- shared$Zx; Zy <- shared$Zy
    
    # 3. Build coefficient matrices
    coeffs <- build_coeffs()
    
    # 4. Mix to observed X, Y
    X <- rbind(
      Zx[1:half_T, ]     %*% coeffs$X1,  # First half
      Zx[(half_T+1):T, ] %*% coeffs$X2   # Second half
    )
    Y <- rbind(
      Zy[1:half_T, ]     %*% coeffs$Y1,
      Zy[(half_T+1):T, ] %*% coeffs$Y2
    )
    
    replicates[[i]] <- list(X = X, Y = Y)
  }
  return(replicates)
}

