import dgl
import torch
import scipy.sparse as sp
import numpy as np
import copy

import os
import sys
import pickle as pkl
import networkx as nx
import random


def set_seed(seed):
    random.seed(seed)
    np.random.seed(seed)
    torch.manual_seed(seed)
    torch.cuda.manual_seed(seed)
    torch.cuda.manual_seed_all(seed)


def accuracy(output, labels):
    preds = output.max(1)[1].type_as(labels)
    correct = preds.eq(labels).double()
    correct = correct.sum()
    return correct / len(labels)


def parse_index_file(filename):
    index = []
    for line in open(filename):
        index.append(int(line.strip()))
    return index


def sparse_mx_to_torch_sparse_tensor(sparse_mx):
    sparse_mx = sparse_mx.tocoo().astype(np.float32)
    indices = torch.from_numpy(
        np.vstack((sparse_mx.row, sparse_mx.col)).astype(np.int64))
    values = torch.from_numpy(sparse_mx.data)
    shape = torch.Size(sparse_mx.shape)
    return torch.sparse.FloatTensor(indices, values, shape)


def normalize(mx):
    rowsum = np.array(mx.sum(1))
    r_inv = np.power(rowsum, -1).flatten()
    r_inv[np.isinf(r_inv)] = 0.
    r_mat_inv = sp.diags(r_inv)
    mx = r_mat_inv.dot(mx)
    return mx


def consis_loss(logps, tem, lam):
    logps = torch.exp(logps)
    sharp_logps = (torch.pow(logps, 1. / tem) / torch.sum(torch.pow(logps, 1. / tem), dim=1, keepdim=True)).detach()
    loss = torch.mean((logps - sharp_logps).pow(2).sum(1)) * lam

    return loss


def propagate_adj(adj, filter, left_power_number, right_power_number):
    D1 = np.array(adj.sum(axis=1)) ** (left_power_number)
    D2 = np.array(adj.sum(axis=0)) ** (right_power_number)
    D1 = sp.diags(D1[:, 0], format='csr')
    D2 = sp.diags(D2[0, :], format='csr')

    A = adj.dot(D1)
    A = D2.dot(A)

    if filter == 'high':
        A = sp.eye(adj.shape[0]) - A
    if filter == 'full':
        A = sp.eye(adj.shape[0])
    if filter == 'low':
        A = A

    A = sparse_mx_to_torch_sparse_tensor(A)

    return A


def random_walk_adj(adj, filter, power_number):
    D1 = np.array(adj.sum(axis=1), dtype=float) ** (power_number)
    D1 = sp.diags(D1[:, 0], format='csr')

    A = adj.dot(D1)

    if filter == 'high':
        A = sp.eye(adj.shape[0]) - A
    if filter == 'full':
        A = sp.eye(adj.shape[0])
    if filter == 'low':
        A = A

    A = sparse_mx_to_torch_sparse_tensor(A)

    return A


def index_to_mask(index, size):
    mask = torch.zeros(size, dtype=torch.bool)
    mask[index] = 1
    return mask


def random_splits(labels, num_classes, percls_trn, val_lb, seed=42):
    index=[i for i in range(0,labels.shape[0])]
    train_idx=[]
    rnd_state = np.random.RandomState(seed)
    for c in range(num_classes):
        class_idx = np.where(labels.cpu() == c)[0]
        if len(class_idx)<percls_trn:
            train_idx.extend(class_idx)
        else:
            train_idx.extend(rnd_state.choice(class_idx, percls_trn,replace=False))
    rest_index = [i for i in index if i not in train_idx]
    val_idx=rnd_state.choice(rest_index,val_lb,replace=False)
    test_idx=[i for i in rest_index if i not in val_idx]

    train_mask = index_to_mask(train_idx,size=len(labels))
    val_mask = index_to_mask(val_idx,size=len(labels))
    test_mask = index_to_mask(test_idx,size=len(labels))
    return train_mask, val_mask, test_mask


def fixed_splits(labels, num_classes, percls_trn, val_lb, name, seed=42):
    if name in ["Chameleon","Squirrel", "Actor"]:
        seed = 1941488137
    index=[i for i in range(0,labels.shape[0])]
    train_idx=[]
    rnd_state = np.random.RandomState(seed)
    for c in range(num_classes):
        class_idx = np.where(labels.cpu() == c)[0]
        if len(class_idx)<percls_trn:
            train_idx.extend(class_idx)
        else:
            train_idx.extend(rnd_state.choice(class_idx, percls_trn,replace=False))
    rest_index = [i for i in index if i not in train_idx]
    val_idx=rnd_state.choice(rest_index,val_lb,replace=False)
    test_idx=[i for i in rest_index if i not in val_idx]

    train_mask = index_to_mask(train_idx,size=len(labels))
    val_mask = index_to_mask(val_idx,size=len(labels))
    test_mask = index_to_mask(test_idx,size=len(labels))

    return train_mask, val_mask, test_mask


def rand_train_test_idx(label, train_prop=.5, valid_prop=.25, ignore_negative=True):
    """ randomly splits label into train/valid/test splits """
    np.random.seed(42)
    torch.manual_seed(42)

    if ignore_negative:
        labeled_nodes = torch.where(label != -1)[0]
    else:
        labeled_nodes = label

    n = labeled_nodes.shape[0]
    train_num = int(n * train_prop)
    valid_num = int(n * valid_prop)

    perm = torch.as_tensor(np.random.permutation(n))

    train_indices = perm[:train_num]
    val_indices = perm[train_num:train_num + valid_num]
    test_indices = perm[train_num + valid_num:]

    if not ignore_negative:
        return train_indices, val_indices, test_indices

    train_idx = labeled_nodes[train_indices]
    valid_idx = labeled_nodes[val_indices]
    test_idx = labeled_nodes[test_indices]

    return train_idx, valid_idx, test_idx


file_dir_citation = os.getcwd() + '/data'
def load_data_citation(dataset_str='cora'):
    names = ['x', 'y', 'tx', 'ty', 'allx', 'ally', 'graph']
    objects = []
    for i in range(len(names)):
        with open("{}/ind.{}.{}".format(file_dir_citation, dataset_str, names[i]), 'rb') as f:
            if sys.version_info > (3, 0):
                objects.append(pkl.load(f, encoding='latin1'))
            else:
                objects.append(pkl.load(f))
    x, y, tx, ty, allx, ally, graph = tuple(objects)
    test_idx_reorder = parse_index_file("{}/ind.{}.test.index".format(file_dir_citation, dataset_str))
    test_idx_range = np.sort(test_idx_reorder)

    if dataset_str == 'citeseer':
        test_idx_range_full = range(min(test_idx_reorder), max(test_idx_reorder) + 1)
        tx_extended = sp.lil_matrix((len(test_idx_range_full), x.shape[1]))
        tx_extended[test_idx_range - min(test_idx_range), :] = tx
        tx = tx_extended
        ty_extended = np.zeros((len(test_idx_range_full), y.shape[1]))
        ty_extended[test_idx_range - min(test_idx_range), :] = ty
        ty = ty_extended

    features = sp.vstack((allx, tx)).tolil()
    features[test_idx_reorder, :] = features[test_idx_range, :]
    features_norm = normalize(features)

    adj = nx.adjacency_matrix(nx.from_dict_of_lists(graph))
    adj = adj + adj.T.multiply(adj.T > adj) - adj.multiply(adj.T > adj)

    adj = adj + sp.eye(adj.shape[0])

    D1 = np.array(adj.sum(axis=1)) ** (-0.5)
    D2 = np.array(adj.sum(axis=0)) ** (-0.5)
    D1 = sp.diags(D1[:, 0], format='csr')
    D2 = sp.diags(D2[0, :], format='csr')

    A = adj.dot(D1)
    A = D2.dot(A)

    labels = np.vstack((ally, ty))
    labels[test_idx_reorder, :] = labels[test_idx_range, :]  # onehot

    idx_train = range(len(y))
    idx_val = range(len(y), len(y) + 500)
    idx_test = test_idx_range.tolist()

    # features = torch.FloatTensor(np.array(features.todense()))
    features_norm = torch.FloatTensor(np.array(features_norm.todense()))
    labels = torch.LongTensor(np.argmax(labels, -1))
    A = sparse_mx_to_torch_sparse_tensor(A)
    idx_train = torch.LongTensor(idx_train)
    idx_val = torch.LongTensor(idx_val)
    idx_test = torch.LongTensor(idx_test)

    return A, features_norm, labels, idx_train, idx_val, idx_test, adj


from load_geom import load_geom
from dgl.data import CoauthorCSDataset, CoraFullDataset, CoauthorPhysicsDataset
SEEDS = [1941488137, 4198936517, 983997847, 4023022221, 4019585660, 2108550661, 1648766618, 629014539,
             3212139042, 2424918363]
def load_data(dataset, full, random_split, train_rate, val_rate, i):
    if dataset in {'cora', 'citeseer', 'pubmed'}:
        _, features, labels, train_mask, val_mask, test_mask, adj = load_data_citation(dataset)
        percls_trn = int(round(train_rate * len(labels) / int(labels.max() + 1)))
        val_lb = int(round(val_rate * len(labels)))
        if full and random_split:
            train_mask, val_mask, test_mask = random_splits(labels, int(labels.max()) + 1, percls_trn, val_lb,
                                                            seed=SEEDS[i])
        # else:
        #     train_mask, val_mask, test_mask = fixed_splits(labels, int(labels.max()) + 1, percls_trn, val_lb,
        #                                                    dataset)
    if dataset in {'texas', 'cornell', "chameleon", 'squirrel', 'wisconsin'}:
        path = os.getcwd()
        dataset_split = path + '/splits/' + f'{dataset}_split_0.6_0.2_{i}.npz'
        g, features, labels, train_mask, val_mask, test_mask = load_geom(dataset, dataset_split,
                                                                         train_percentage=None, val_percentage=None,
                                                                         embedding_mode=None, embedding_method=None,
                                                                         embedding_method_graph=None,
                                                                         embedding_method_space=None)
        percls_trn = int(round(train_rate * len(labels) / int(labels.max() + 1)))
        val_lb = int(round(val_rate * len(labels)))
        g = dgl.add_self_loop(g)
        adj = g.adj(scipy_fmt='csr')
        if full and random_split:
            train_mask, val_mask, test_mask = random_splits(labels, int(labels.max()) + 1, percls_trn, val_lb,
                                                            seed=SEEDS[i])
        if not full and random_split:
            train_mask, val_mask, test_mask = random_splits(labels, int(labels.max()) + 1, percls_trn, val_lb,
                                                            seed=SEEDS[i])
        if not full and not random_split:
            train_mask, val_mask, test_mask = fixed_splits(labels, int(labels.max()) + 1, percls_trn, val_lb, dataset)
    if dataset in {'cora-full', 'cs', 'physics'}:
        if dataset == 'cora-full':
            data = CoraFullDataset()
        if dataset == 'cs':
            data = CoauthorCSDataset()
        if dataset == 'physics':
            data = CoauthorPhysicsDataset()

        g = data[0]  # .to(device)
        g = dgl.add_self_loop(g)
        adj = g.adj(scipy_fmt='csr')
        features = g.ndata['feat']
        labels = g.ndata['label']

        percls_trn = int(round(train_rate * len(labels) / int(labels.max() + 1)))
        val_lb = int(round(val_rate * len(labels)))
        if full and random_split:
            train_mask, val_mask, test_mask = random_splits(labels, int(labels.max()) + 1, percls_trn, val_lb,
                                                            seed=SEEDS[i])
        if not full and random_split:
            train_mask, val_mask, test_mask = random_splits(labels, int(labels.max()) + 1, percls_trn, val_lb,
                                                            seed=SEEDS[i])
        if not full and not random_split:
            train_mask, val_mask, test_mask = fixed_splits(labels, int(labels.max()) + 1, percls_trn, val_lb, dataset)

    return adj, features, labels, train_mask, val_mask, test_mask

