import pandas as pd
import numpy as np
import archetypes as arch


def load_archetype_matrix(filepath):
    """
    Load the archetype matrix from a CSV file.

    Args:
        filepath (str): Path to the CSV file containing the archetype matrix.

    Returns:
        np.ndarray: Archetype matrix as a NumPy array.
    """
    archetypes = pd.read_csv(filepath)
    return archetypes.to_numpy()


def normalize_data(values, data_min, data_max, archetype_min, archetype_max):
    """
    Normalize the data to the range of archetypes.

    Args:
        values (np.ndarray): Original data values.
        data_min (float): Minimum value in the data.
        data_max (float): Maximum value in the data.
        archetype_min (float): Minimum value in the archetypes.
        archetype_max (float): Maximum value in the archetypes.

    Returns:
        np.ndarray: Normalized values.
    """
    normalized = (values - data_min) / (data_max - data_min)
    scaled = normalized * (archetype_max - archetype_min) + archetype_min
    return scaled


def decompose_hvf_data(aa, hvf_data):
    """
    Perform decomposition using the archetype analysis model.

    Args:
        aa (arch.AA): Trained archetype analysis model.
        hvf_data (np.ndarray): HVF data to be decomposed.

    Returns:
        np.ndarray: Decomposition coefficients.
    """
    return aa.transform(hvf_data)


def reconstruct_from_coefficients(archetype_matrix, coefficients):
    """
    Reconstruct data using archetype coefficients.

    Args:
        archetype_matrix (np.ndarray): Matrix of archetypes.
        coefficients (np.ndarray): Decomposition coefficients.

    Returns:
        np.ndarray: Reconstructed data.
    """
    return coefficients @ archetype_matrix