# coding=utf-8
# Copyright 2022 EleutherAI and the HuggingFace Inc. team. All rights reserved.
#
# This code is based on EleutherAI's GPT-NeoX library and the GPT-NeoX
# and OPT implementations in this library. It has been modified from its
# original forms to accommodate minor architectural differences compared
# to GPT-NeoX and OPT used by the Meta AI team that trained the model.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
""" PyTorch LLaMA model."""
import math
from typing import List, Optional, Tuple, Union
import string

# from nltk.corpus import stopwords

import torch
import torch.nn.functional as F
import torch.utils.checkpoint
from torch import nn
from torch.nn import BCEWithLogitsLoss, CrossEntropyLoss, MSELoss
import torch.distributed as dist

from transformers.activations import ACT2FN
from transformers.modeling_outputs import (
    BaseModelOutputWithPast,
    CausalLMOutputWithPast,
    SequenceClassifierOutputWithPast,
)
from transformers.modeling_utils import PreTrainedModel
from transformers.utils import (
    add_start_docstrings,
    add_start_docstrings_to_model_forward,
    logging,
    replace_return_docstrings,
)
from transformers import LlamaConfig
from transformers.models.llama.modeling_llama import LlamaDecoderLayer

from src.model.concrete_gate import ConcreteGate
from src.utils.train_utils import SoftThresholdingCenter


logger = logging.get_logger(__name__)

_CONFIG_FOR_DOC = "LlamaConfig"


# Copied from transformers.models.bart.modeling_bart._make_causal_mask
def _make_causal_mask(
    input_ids_shape: torch.Size,
    dtype: torch.dtype,
    device: torch.device,
    past_key_values_length: int = 0,
):
    """
    Make causal mask used for bi-directional self-attention.
    """
    bsz, tgt_len = input_ids_shape
    mask = torch.full((tgt_len, tgt_len), torch.finfo(dtype).min, device=device)
    mask_cond = torch.arange(mask.size(-1), device=device)
    mask.masked_fill_(mask_cond < (mask_cond + 1).view(mask.size(-1), 1), 0)
    mask = mask.to(dtype)

    if past_key_values_length > 0:
        mask = torch.cat(
            [
                torch.zeros(
                    tgt_len, past_key_values_length, dtype=dtype, device=device
                ),
                mask,
            ],
            dim=-1,
        )
    return mask[None, None, :, :].expand(
        bsz, 1, tgt_len, tgt_len + past_key_values_length
    )


def nanstd(o, mean, dim):
    return torch.sqrt(
        torch.nanmean(
            torch.pow(torch.abs(o - mean.unsqueeze(dim)), 2),
            dim=dim,
        )
    )


# Copied from transformers.models.bart.modeling_bart._expand_mask
def _expand_mask(mask: torch.Tensor, dtype: torch.dtype, tgt_len: Optional[int] = None):
    """
    Expands attention_mask from `[bsz, seq_len]` to `[bsz, 1, tgt_seq_len, src_seq_len]`.
    """
    bsz, src_len = mask.size()
    tgt_len = tgt_len if tgt_len is not None else src_len

    expanded_mask = mask[:, None, None, :].expand(bsz, 1, tgt_len, src_len).to(dtype)

    inverted_mask = 1.0 - expanded_mask

    return inverted_mask.masked_fill(
        inverted_mask.to(torch.bool), torch.finfo(dtype).min
    )


class LlamaRMSNorm(nn.Module):
    def __init__(self, hidden_size, eps=1e-6):
        """
        LlamaRMSNorm is equivalent to T5LayerNorm
        """
        super().__init__()
        self.weight = nn.Parameter(torch.ones(hidden_size))
        self.variance_epsilon = eps

    def forward(self, hidden_states):
        input_dtype = hidden_states.dtype
        hidden_states = hidden_states.to(torch.float32)
        variance = hidden_states.pow(2).mean(-1, keepdim=True)
        hidden_states = hidden_states * torch.rsqrt(variance + self.variance_epsilon)
        return self.weight * hidden_states.to(input_dtype)


def _get_fill_value(config, amplify_factor):
    if config.amplify_k_scope == "question":
        return 0
    amplify_factor = (
        amplify_factor
        if config.amplify_output_factor is None
        else config.amplify_output_factor
    )
    if config.amplify_decay is not None:
        return amplify_factor * config.amplify_decay
    return amplify_factor


class LlamaRotaryEmbedding(torch.nn.Module):
    def __init__(self, dim, max_position_embeddings=2048, base=10000, device=None):
        super().__init__()

        self.dim = dim
        self.max_position_embeddings = max_position_embeddings
        self.base = base
        inv_freq = 1.0 / (
            self.base ** (torch.arange(0, self.dim, 2).float().to(device) / self.dim)
        )
        self.register_buffer("inv_freq", inv_freq)

        # Build here to make `torch.jit.trace` work.
        self._set_cos_sin_cache(
            seq_len=max_position_embeddings,
            device=self.inv_freq.device,
            dtype=torch.get_default_dtype(),
        )

    def _set_cos_sin_cache(self, seq_len, device, dtype):
        self.max_seq_len_cached = seq_len
        t = torch.arange(
            self.max_seq_len_cached, device=device, dtype=self.inv_freq.dtype
        )

        freqs = torch.einsum("i,j->ij", t, self.inv_freq)
        # Different from paper, but it uses a different permutation in order to obtain the same calculation
        emb = torch.cat((freqs, freqs), dim=-1)
        self.register_buffer(
            "cos_cached", emb.cos()[None, None, :, :].to(dtype), persistent=False
        )
        self.register_buffer(
            "sin_cached", emb.sin()[None, None, :, :].to(dtype), persistent=False
        )

    def forward(self, x, seq_len=None):
        # x: [bs, num_attention_heads, seq_len, head_size]
        if seq_len > self.max_seq_len_cached:
            self._set_cos_sin_cache(seq_len=seq_len, device=x.device, dtype=x.dtype)

        return (
            self.cos_cached[:, :, :seq_len, ...].to(dtype=x.dtype),
            self.sin_cached[:, :, :seq_len, ...].to(dtype=x.dtype),
        )


class LlamaLinearScalingRotaryEmbedding(LlamaRotaryEmbedding):
    """LlamaRotaryEmbedding extended with linear scaling. Credits to the Reddit user /u/kaiokendev"""

    def __init__(
        self,
        dim,
        max_position_embeddings=2048,
        base=10000,
        device=None,
        scaling_factor=1.0,
    ):
        self.scaling_factor = scaling_factor
        super().__init__(dim, max_position_embeddings, base, device)

    def _set_cos_sin_cache(self, seq_len, device, dtype):
        self.max_seq_len_cached = seq_len
        t = torch.arange(
            self.max_seq_len_cached, device=device, dtype=self.inv_freq.dtype
        )
        t = t / self.scaling_factor

        freqs = torch.einsum("i,j->ij", t, self.inv_freq)
        # Different from paper, but it uses a different permutation in order to obtain the same calculation
        emb = torch.cat((freqs, freqs), dim=-1)
        self.register_buffer(
            "cos_cached", emb.cos()[None, None, :, :].to(dtype), persistent=False
        )
        self.register_buffer(
            "sin_cached", emb.sin()[None, None, :, :].to(dtype), persistent=False
        )


class LlamaDynamicNTKScalingRotaryEmbedding(LlamaRotaryEmbedding):
    """LlamaRotaryEmbedding extended with Dynamic NTK scaling. Credits to the Reddit users /u/bloc97 and /u/emozilla"""

    def __init__(
        self,
        dim,
        max_position_embeddings=2048,
        base=10000,
        device=None,
        scaling_factor=1.0,
    ):
        self.scaling_factor = scaling_factor
        super().__init__(dim, max_position_embeddings, base, device)

    def _set_cos_sin_cache(self, seq_len, device, dtype):
        self.max_seq_len_cached = seq_len

        if seq_len > self.max_position_embeddings:
            base = self.base * (
                (self.scaling_factor * seq_len / self.max_position_embeddings)
                - (self.scaling_factor - 1)
            ) ** (self.dim / (self.dim - 2))
            inv_freq = 1.0 / (
                base ** (torch.arange(0, self.dim, 2).float().to(device) / self.dim)
            )
            self.register_buffer("inv_freq", inv_freq)

        t = torch.arange(
            self.max_seq_len_cached, device=device, dtype=self.inv_freq.dtype
        )

        freqs = torch.einsum("i,j->ij", t, self.inv_freq)
        # Different from paper, but it uses a different permutation in order to obtain the same calculation
        emb = torch.cat((freqs, freqs), dim=-1)
        self.register_buffer(
            "cos_cached", emb.cos()[None, None, :, :].to(dtype), persistent=False
        )
        self.register_buffer(
            "sin_cached", emb.sin()[None, None, :, :].to(dtype), persistent=False
        )


def rotate_half(x):
    """Rotates half the hidden dims of the input."""
    x1 = x[..., : x.shape[-1] // 2]
    x2 = x[..., x.shape[-1] // 2 :]
    return torch.cat((-x2, x1), dim=-1)


def apply_rotary_pos_emb(q, k, cos, sin, position_ids):
    # The first two dimensions of cos and sin are always 1, so we can `squeeze` them.
    cos = cos.squeeze(1).squeeze(0)  # [seq_len, dim]
    sin = sin.squeeze(1).squeeze(0)  # [seq_len, dim]
    cos = cos[position_ids].unsqueeze(1)  # [bs, 1, seq_len, dim]
    sin = sin[position_ids].unsqueeze(1)  # [bs, 1, seq_len, dim]
    q_embed = (q * cos) + (rotate_half(q) * sin)
    k_embed = (k * cos) + (rotate_half(k) * sin)
    return q_embed, k_embed


class LlamaMLP(nn.Module):
    def __init__(self, config):
        super().__init__()
        self.pretraining_tp = config.pretraining_tp
        self.hidden_size = config.hidden_size
        self.intermediate_size = config.intermediate_size
        self.gate_proj = nn.Linear(self.hidden_size, self.intermediate_size, bias=False)
        self.up_proj = nn.Linear(self.hidden_size, self.intermediate_size, bias=False)
        self.down_proj = nn.Linear(self.intermediate_size, self.hidden_size, bias=False)
        self.act_fn = ACT2FN[config.hidden_act]

    def forward(self, x):
        if self.pretraining_tp > 1:
            slice = self.intermediate_size // self.pretraining_tp
            gate_proj_slices = self.gate_proj.weight.split(slice, dim=0)
            up_proj_slices = self.up_proj.weight.split(slice, dim=0)
            down_proj_slices = self.down_proj.weight.split(slice, dim=1)

            gate_proj = torch.cat(
                [F.linear(x, gate_proj_slices[i]) for i in range(self.pretraining_tp)],
                dim=-1,
            )
            up_proj = torch.cat(
                [F.linear(x, up_proj_slices[i]) for i in range(self.pretraining_tp)],
                dim=-1,
            )

            intermediate_states = (self.act_fn(gate_proj) * up_proj).split(slice, dim=2)
            down_proj = [
                F.linear(intermediate_states[i], down_proj_slices[i])
                for i in range(self.pretraining_tp)
            ]
            down_proj = sum(down_proj)
        else:
            down_proj = self.down_proj(self.act_fn(self.gate_proj(x)) * self.up_proj(x))

        return down_proj


def repeat_kv(hidden_states: torch.Tensor, n_rep: int) -> torch.Tensor:
    """
    This is the equivalent of torch.repeat_interleave(x, dim=1, repeats=n_rep). The hidden states go from (batch,
    num_key_value_heads, seqlen, head_dim) to (batch, num_attention_heads, seqlen, head_dim)
    """
    batch, num_key_value_heads, slen, head_dim = hidden_states.shape
    if n_rep == 1:
        return hidden_states
    hidden_states = hidden_states[:, :, None, :, :].expand(
        batch, num_key_value_heads, n_rep, slen, head_dim
    )
    return hidden_states.reshape(batch, num_key_value_heads * n_rep, slen, head_dim)


class LlamaAttentionAmplify(nn.Module):
    """Multi-headed attention from 'Attention Is All You Need' paper"""

    def __init__(self, config: LlamaConfig, layer_heads=None):
        super().__init__()
        self.config = config
        self.hidden_size = config.hidden_size
        self.num_heads = config.num_attention_heads
        self.head_dim = self.hidden_size // self.num_heads
        self.num_key_value_heads = config.num_key_value_heads
        self.num_key_value_groups = self.num_heads // self.num_key_value_heads
        self.pretraining_tp = config.pretraining_tp
        self.max_position_embeddings = config.max_position_embeddings
        self.layer_heads = layer_heads
        self.params = None
        if (
            self.config.rg_grad_head_score is not None
            and "concrete" in self.config.rg_grad_head_score
        ):
            self.params = ConcreteGate([1, self.num_heads, 1, 1])
        elif (
            self.config.rg_grad_head_score is not None
            and "discrete" in self.config.rg_grad_head_score
        ):
            self.soft_gate = nn.Parameter(
                torch.zeros([1, self.num_heads, 1, 1], requires_grad=True)
            )

        if (self.head_dim * self.num_heads) != self.hidden_size:
            raise ValueError(
                f"hidden_size must be divisible by num_heads (got `hidden_size`: {self.hidden_size}"
                f" and `num_heads`: {self.num_heads})."
            )
        self.q_proj = nn.Linear(
            self.hidden_size, self.num_heads * self.head_dim, bias=False
        )
        self.k_proj = nn.Linear(
            self.hidden_size, self.num_key_value_heads * self.head_dim, bias=False
        )
        self.v_proj = nn.Linear(
            self.hidden_size, self.num_key_value_heads * self.head_dim, bias=False
        )
        self.o_proj = nn.Linear(
            self.num_heads * self.head_dim, self.hidden_size, bias=False
        )
        if self.config.amplify_soft_mask is not None:
            center = (
                self.config.amplify_adapt_threshold_factor
                + self.config.amplify_adapt_threshold_factor_upper
            ) / 2
            half_width = (
                self.config.amplify_adapt_threshold_factor_upper
                - self.config.amplify_adapt_threshold_factor
            ) / 2
            self.soft_threshold = SoftThresholdingCenter(
                center, half_width, k=self.config.amplify_soft_mask
            )
        self._init_rope()

    def _init_rope(self):
        if self.config.rope_scaling is None:
            self.rotary_emb = LlamaRotaryEmbedding(
                self.head_dim, max_position_embeddings=self.max_position_embeddings
            )
        else:
            scaling_type = self.config.rope_scaling["type"]
            scaling_factor = self.config.rope_scaling["factor"]
            if scaling_type == "linear":
                self.rotary_emb = LlamaLinearScalingRotaryEmbedding(
                    self.head_dim,
                    max_position_embeddings=self.max_position_embeddings,
                    scaling_factor=scaling_factor,
                )
            elif scaling_type == "dynamic":
                self.rotary_emb = LlamaDynamicNTKScalingRotaryEmbedding(
                    self.head_dim,
                    max_position_embeddings=self.max_position_embeddings,
                    scaling_factor=scaling_factor,
                )
            else:
                raise ValueError(f"Unknown RoPE scaling type {scaling_type}")

    def _shape(self, tensor: torch.Tensor, seq_len: int, bsz: int):
        return (
            tensor.view(bsz, seq_len, self.num_heads, self.head_dim)
            .transpose(1, 2)
            .contiguous()
        )

    def _prepare_attention_mask(
        self, attention_mask, k_token_mask, q_token_mask, bsz, q_len, kv_seq_len
    ):
        amplify_factor = self.config.amplify_factor
        is_cache = kv_seq_len != k_token_mask.shape[1]
        if is_cache:
            if self.config.amplify_q_scope == "question":
                return None
            k_token_mask = torch.cat(
                [
                    k_token_mask,
                    torch.full(
                        (k_token_mask.shape[0], kv_seq_len - k_token_mask.shape[-1]),
                        _get_fill_value(self.config, amplify_factor),
                        device=k_token_mask.device,
                        dtype=k_token_mask.dtype,
                    ),
                ],
                dim=1,
            )
            q_token_mask = None
        expanded_token_mask = (
            k_token_mask[:, None, None, :]
            .expand(bsz, self.num_heads, q_len, kv_seq_len)
            .to(attention_mask.dtype)
        )
        # assign ln(0.01)
        scale_constant = torch.tensor(
            [0.01], dtype=expanded_token_mask.dtype, device=expanded_token_mask.device
        ).log()
        expanded_token_mask[
            expanded_token_mask == 0
        ] = scale_constant.item()  # use .item() to extract the scalar value
        expanded_token_mask[expanded_token_mask == amplify_factor] = 0
        if (
            q_token_mask is not None
            and not is_cache
            and self.config.amplify_q_scope != "all"
        ):
            expanded_token_mask *= q_token_mask[:, None, :, None]
        if self.config.amplify_head_id is not None:
            new_expanded_token_mask = torch.zeros_like(expanded_token_mask)
            new_expanded_token_mask[:, self.config.amplify_head_id, :, :] = 1
            expanded_token_mask = expanded_token_mask * new_expanded_token_mask
        if self.layer_heads is not None and len(self.layer_heads) > 0:
            indices_tensor = torch.tensor(
                self.layer_heads, device=attention_mask.device, dtype=torch.long
            )
            broadcasted_indices = indices_tensor.view(1, -1, 1, 1).expand(
                bsz, -1, q_len, kv_seq_len
            )
            new_expanded_token_mask = torch.zeros_like(
                expanded_token_mask, dtype=torch.int
            )
            new_expanded_token_mask.scatter_(1, broadcasted_indices, 1)
            expanded_token_mask = expanded_token_mask * new_expanded_token_mask
        return expanded_token_mask + attention_mask

    def get_attn_weight_entropy(self, attn_weights):
        entropy = -(attn_weights * (attn_weights + 1e-6).log()).sum(dim=-1).mean()
        return entropy

    def _prepare_expanded_token_mask_soft(
        self,
        k_token_mask,  # dimension: (bsz, seq_len)
        attn_weights,  # dimension: (bsz, num_heads, q_len, kv_seq_len)
        q_token_mask,  # dimension: (bsz, seq_len)
        bsz,  # dimension: (bsz)
        q_len,  # dimension: (q_len)
        kv_seq_len,  # dimension: (kv_seq_len)
        tgt_index,
    ):
        amplify_factor = self.config.amplify_factor
        is_cache = kv_seq_len != k_token_mask.shape[-1]
        k_token_mask_len = k_token_mask.shape[-1]
        if len(k_token_mask.shape) == 2:
            assert amplify_factor != 0
            if is_cache:
                if self.config.amplify_q_scope == "question":
                    return None
                k_token_mask = torch.cat(
                    [
                        k_token_mask,
                        torch.full(
                            (
                                k_token_mask.shape[0],
                                kv_seq_len - k_token_mask.shape[-1],
                            ),
                            _get_fill_value(self.config, amplify_factor),
                            device=k_token_mask.device,
                            dtype=k_token_mask.dtype,
                        ),
                    ],
                    dim=1,
                )
                q_token_mask = None
            expanded_token_mask = (
                k_token_mask[:, None, None, :]
                .expand(bsz, 1, q_len, kv_seq_len)
                .to(attn_weights.dtype)
            )
        elif len(k_token_mask.shape) == 3:
            if amplify_factor <= 0:
                return None
            expanded_token_mask = k_token_mask.unsqueeze(1)
        if (
            q_token_mask is not None
            and not is_cache
            and self.config.amplify_q_scope != "all"
        ):
            expanded_token_mask = (
                expanded_token_mask.clone()
            )  # Clone the tensor before in-place operation
            expanded_token_mask *= q_token_mask[:, None, :, None]
        # if self.config.amplify_topk and (
        #     self.config.amplify_total_threshold_output is not None
        #     or self.config.amplify_total_threshold_upper is not None
        # ):
        #     range_tensor = (
        #         torch.arange(kv_seq_len)
        #         .expand(bsz, kv_seq_len)
        #         .to(device=tgt_index.device)
        #     )
        #     bool_mask = range_tensor >= tgt_index.unsqueeze(
        #         1
        #     )  # shape (bsz, kv_seq_len)

        if self.config.amplify_topk is not None:
            if self.config.amplify_total_threshold is not None:
                binary_old_mask = (expanded_token_mask != 0).int()
                if self.config.amplify_layer_threshold:
                    masked_attn_weights = (
                        attn_weights.mean(dim=1, keepdim=True) * binary_old_mask
                    )
                else:
                    masked_attn_weights = attn_weights * binary_old_mask

                if (
                    self.config.amplify_adapt_threshold is not None
                    or self.config.amplify_norm_threshold
                ):
                    masked_attn_weights_nan = masked_attn_weights.masked_fill(
                        ~binary_old_mask.bool(), torch.nan
                    )
                    weight_mean = torch.nanmean(masked_attn_weights_nan, dim=-1)
                    if (
                        self.config.amplify_adapt_threshold == "mean_std"
                        or self.config.amplify_norm_threshold
                    ):
                        weight_std = nanstd(
                            masked_attn_weights_nan, weight_mean, dim=-1
                        )
                # TODO consider constant threshold?
                # TODO remove question upper threshold?
                soft_mask = self.soft_threshold(
                    masked_attn_weights, weight_mean, weight_std
                )
                # print("soft_mask", soft_mask)
                # temp_mask = mask * upper_mask
                # temp_mask.masked_fill_(~(bool_mask[:, None, None, :]), 1)
                # mask = mask * temp_mask
                expanded_token_mask = expanded_token_mask.clone() * soft_mask
        if self.config.amplify_head_id is not None:
            new_expanded_token_mask = torch.zeros_like(attn_weights)
            new_expanded_token_mask[:, self.config.amplify_head_id, :, :] = 1
            expanded_token_mask = (
                expanded_token_mask.expand(-1, self.num_heads, -1, -1)
                * new_expanded_token_mask
            )
        if self.layer_heads is not None and len(self.layer_heads) > 0:
            indices_tensor = torch.tensor(
                self.layer_heads, device=attn_weights.device, dtype=torch.long
            )
            broadcasted_indices = indices_tensor.view(1, -1, 1, 1).expand(
                bsz, -1, q_len, kv_seq_len
            )
            new_expanded_token_mask = torch.ones_like(attn_weights, dtype=torch.bool)
            new_expanded_token_mask.scatter_(1, broadcasted_indices, False)
            expanded_token_mask = expanded_token_mask.expand(
                -1, self.num_heads, -1, -1
            ).masked_fill_(new_expanded_token_mask, 0)
        return expanded_token_mask

    def _prepare_expanded_token_mask(
        self,
        k_token_mask,  # dimension: (bsz, seq_len)
        attn_weights,  # dimension: (bsz, num_heads, q_len, kv_seq_len)
        q_token_mask,  # dimension: (bsz, seq_len)
        bsz,  # dimension: (bsz)
        q_len,  # dimension: (q_len)
        kv_seq_len,  # dimension: (kv_seq_len)
        tgt_index,
    ):
        amplify_factor = self.config.amplify_factor
        is_cache = kv_seq_len != k_token_mask.shape[-1]
        k_token_mask_len = k_token_mask.shape[-1]
        if len(k_token_mask.shape) == 2:
            assert amplify_factor != 0
            if is_cache:
                if self.config.amplify_q_scope == "question":
                    return None
                k_token_mask = torch.cat(
                    [
                        k_token_mask,
                        torch.full(
                            (
                                k_token_mask.shape[0],
                                kv_seq_len - k_token_mask.shape[-1],
                            ),
                            _get_fill_value(self.config, amplify_factor),
                            device=k_token_mask.device,
                            dtype=k_token_mask.dtype,
                        ),
                    ],
                    dim=1,
                )
                q_token_mask = None
            expanded_token_mask = (
                k_token_mask[:, None, None, :]
                .expand(bsz, 1, q_len, kv_seq_len)
                .to(attn_weights.dtype)
            )
        elif len(k_token_mask.shape) == 3:
            if amplify_factor <= 0:
                return None
            expanded_token_mask = k_token_mask.unsqueeze(1)
        if (
            q_token_mask is not None
            and not is_cache
            and self.config.amplify_q_scope != "all"
        ):
            expanded_token_mask = (
                expanded_token_mask.clone()
            )  # Clone the tensor before in-place operation
            expanded_token_mask *= q_token_mask[:, None, :, None]

        if self.config.amplify_topk and (
            self.config.amplify_total_threshold_output is not None
            or self.config.amplify_total_threshold_upper is not None
        ):
            range_tensor = (
                torch.arange(kv_seq_len)
                .expand(bsz, kv_seq_len)
                .to(device=tgt_index.device)
            )
            bool_mask = range_tensor >= tgt_index.unsqueeze(
                1
            )  # shape (bsz, kv_seq_len)
            if self.config.amplify_total_threshold_output is not None:
                threshold = torch.full_like(
                    bool_mask,
                    self.config.amplify_total_threshold,
                    dtype=k_token_mask.dtype,
                )
                threshold.masked_fill_(
                    bool_mask, self.config.amplify_total_threshold_output
                )
        if self.config.amplify_topk is not None:
            if self.config.amplify_total_threshold is not None:
                binary_old_mask = (expanded_token_mask != 0).int()
                if self.config.amplify_layer_threshold:
                    masked_attn_weights = (
                        attn_weights.mean(dim=1, keepdim=True) * binary_old_mask
                    )
                else:
                    masked_attn_weights = attn_weights * binary_old_mask

                if (
                    self.config.amplify_adapt_threshold is not None
                    or self.config.amplify_norm_threshold
                ):
                    masked_attn_weights_nan = masked_attn_weights.masked_fill(
                        ~binary_old_mask.bool(), torch.nan
                    )
                    weight_mean = torch.nanmean(masked_attn_weights_nan, dim=-1)
                    if (
                        self.config.amplify_adapt_threshold == "mean_std"
                        or self.config.amplify_norm_threshold
                    ):
                        weight_std = nanstd(
                            masked_attn_weights_nan, weight_mean, dim=-1
                        )

                if self.config.amplify_norm_threshold:
                    masked_attn_weights = (
                        masked_attn_weights - weight_mean.unsqueeze(-1)
                    ) / weight_std.unsqueeze(-1)

                if self.config.amplify_adapt_threshold is not None:
                    if self.config.amplify_adapt_threshold == "mean_std":
                        adapt_threshold = (
                            weight_mean
                            + weight_std * self.config.amplify_adapt_threshold_factor
                        ).unsqueeze(-1)
                    else:
                        adapt_threshold = (
                            weight_mean.unsqueeze(-1)
                            * self.config.amplify_adapt_threshold_factor
                        )
                    # if dist.is_initialized() and dist.get_rank() == 0:
                    #     print("adapt", adapt_threshold[0, 0, 0])
                    #     print("mean", weight_mean[0, 0, 0])
                    #     print((masked_attn_weights.sum(dim=-1) / binary_old_mask.sum(dim=-1))[0, 0, 0])
                else:
                    if self.config.amplify_total_threshold_output is not None:
                        adapt_threshold = threshold[:, None, None, :]
                    else:
                        adapt_threshold = self.config.amplify_total_threshold
                mask = (masked_attn_weights >= adapt_threshold).to(
                    dtype=attn_weights.dtype
                )

                if self.config.amplify_total_threshold_upper is not None:
                    if self.config.amplify_adapt_threshold_factor_upper is not None:
                        if self.config.amplify_adapt_threshold == "mean_std":
                            adapt_upper_threshold = (
                                weight_mean
                                + weight_std
                                * self.config.amplify_adapt_threshold_factor_upper
                            ).unsqueeze(-1)
                        else:
                            adapt_upper_threshold = (
                                weight_mean.unsqueeze(-1)
                                * self.config.amplify_adapt_threshold_factor_upper
                            )
                    else:
                        adapt_upper_threshold = (
                            self.config.amplify_total_threshold_upper
                        )

                    upper_mask = (masked_attn_weights <= adapt_upper_threshold).to(
                        dtype=attn_weights.dtype
                    )
                    if self.config.amplify_upper_position is not None:
                        upper_mask = torch.ones_like(masked_attn_weights)
                        last_k = min(
                            self.config.amplify_upper_position,
                            kv_seq_len - k_token_mask_len,
                        )
                        upper_mask[:, :, :, -last_k:] = 0
                    # just for debug
                    # upper_mask = torch.ones_like(masked_attn_weights)
                    # upper_mask.masked_fill_(
                    #     masked_attn_weights > self.config.amplify_total_threshold_upper, 2
                    # )
                    if self.config.amplify_exclude_cal is not None:
                        is_cals_tensor = torch.tensor(
                            self.is_cals, dtype=torch.bool, device=mask.device
                        )
                        if self.config.amplify_exclude_cal == "all":
                            upper_mask[is_cals_tensor] = 1
                        elif self.config.amplify_exclude_cal == "reverse":
                            upper_mask[is_cals_tensor] = 1 - upper_mask[is_cals_tensor]
                    temp_mask = mask * upper_mask
                    if not self.config.amplify_upper_question:
                        temp_mask.masked_fill_(~(bool_mask[:, None, None, :]), 1)
                    mask = mask * temp_mask
                expanded_token_mask = expanded_token_mask.clone() * mask
            else:
                masked_attn_weights = attn_weights * expanded_token_mask
                _, top_indexes = torch.topk(
                    masked_attn_weights,
                    k=min(self.config.amplify_topk, masked_attn_weights.shape[-1]),
                    dim=-1,
                    sorted=True,
                )  # dimension: (bsz, num_heads, q_len, amplify_topk)
                ones = torch.full_like(
                    top_indexes, amplify_factor, dtype=attn_weights.dtype
                )
                new_expanded_token_mask = torch.zeros_like(attn_weights)
                new_expanded_token_mask.scatter_(-1, top_indexes, ones)
                binary_old_mask = (expanded_token_mask != 0).int()
                expanded_token_mask = binary_old_mask * new_expanded_token_mask
        if self.config.amplify_topk_head is not None:
            binary_old_mask = (expanded_token_mask != 0).int()
            masked_attn_weights = attn_weights * binary_old_mask
            total_attn_weights = masked_attn_weights.sum(dim=-1).sum(dim=-1)
            _, top_indexes = torch.topk(
                total_attn_weights,
                k=min(self.config.amplify_topk_head, masked_attn_weights.shape[1]),
                dim=1,
                sorted=True,
            )
            ones = torch.full_like(
                top_indexes, amplify_factor, dtype=attn_weights.dtype
            )
            new_token_mask = torch.zeros(
                total_attn_weights.shape,
                device=total_attn_weights.device,
                dtype=attn_weights.dtype,
            )
            new_token_mask.scatter_(-1, top_indexes, ones)
            new_expanded_token_mask = (
                new_token_mask.unsqueeze(-1).unsqueeze(-1).expand_as(attn_weights)
            )
            expanded_token_mask = binary_old_mask * new_expanded_token_mask
        if self.config.amplify_head_id is not None:
            new_expanded_token_mask = torch.zeros_like(attn_weights)
            new_expanded_token_mask[:, self.config.amplify_head_id, :, :] = 1
            expanded_token_mask = (
                expanded_token_mask.expand(-1, self.num_heads, -1, -1)
                * new_expanded_token_mask
            )
        if self.layer_heads is not None and len(self.layer_heads) > 0:
            indices_tensor = torch.tensor(
                self.layer_heads, device=attn_weights.device, dtype=torch.long
            )
            broadcasted_indices = indices_tensor.view(1, -1, 1, 1).expand(
                bsz, -1, q_len, kv_seq_len
            )
            new_expanded_token_mask = torch.ones_like(attn_weights, dtype=torch.bool)
            new_expanded_token_mask.scatter_(1, broadcasted_indices, False)
            expanded_token_mask = expanded_token_mask.expand(
                -1, self.num_heads, -1, -1
            ).masked_fill_(new_expanded_token_mask, 0)
        if self.config.amplify_exclude_self:
            if dist.is_initialized() and dist.get_rank() == 0:
                print(
                    "self exclude mask",
                    expanded_token_mask[0, 0, torch.arange(q_len), torch.arange(q_len)],
                )
        return expanded_token_mask

    def forward(
        self,
        hidden_states: torch.Tensor,
        attention_mask: Optional[torch.Tensor] = None,
        position_ids: Optional[torch.LongTensor] = None,
        past_key_value: Optional[Tuple[torch.Tensor]] = None,
        output_attentions: bool = False,
        use_cache: bool = False,
        k_token_mask: Optional[torch.Tensor] = None,
        q_token_mask: Optional[torch.Tensor] = None,
        tgt_index: Optional[torch.Tensor] = None,
    ) -> Tuple[torch.Tensor, Optional[torch.Tensor], Optional[Tuple[torch.Tensor]]]:
        bsz, q_len, _ = hidden_states.size()

        if self.pretraining_tp > 1:
            key_value_slicing = (
                self.num_key_value_heads * self.head_dim
            ) // self.pretraining_tp
            query_slices = self.q_proj.weight.split(
                (self.num_heads * self.head_dim) // self.pretraining_tp, dim=0
            )
            key_slices = self.k_proj.weight.split(key_value_slicing, dim=0)
            value_slices = self.v_proj.weight.split(key_value_slicing, dim=0)

            query_states = [
                F.linear(hidden_states, query_slices[i])
                for i in range(self.pretraining_tp)
            ]
            query_states = torch.cat(query_states, dim=-1)

            key_states = [
                F.linear(hidden_states, key_slices[i])
                for i in range(self.pretraining_tp)
            ]
            key_states = torch.cat(key_states, dim=-1)

            value_states = [
                F.linear(hidden_states, value_slices[i])
                for i in range(self.pretraining_tp)
            ]
            value_states = torch.cat(value_states, dim=-1)

        else:
            query_states = self.q_proj(hidden_states)
            key_states = self.k_proj(hidden_states)
            value_states = self.v_proj(hidden_states)

        query_states = query_states.view(
            bsz, q_len, self.num_heads, self.head_dim
        ).transpose(1, 2)
        key_states = key_states.view(
            bsz, q_len, self.num_key_value_heads, self.head_dim
        ).transpose(1, 2)
        value_states = value_states.view(
            bsz, q_len, self.num_key_value_heads, self.head_dim
        ).transpose(1, 2)

        kv_seq_len = key_states.shape[-2]
        if past_key_value is not None:
            kv_seq_len += past_key_value[0].shape[-2]
        cos, sin = self.rotary_emb(value_states, seq_len=kv_seq_len)
        query_states, key_states = apply_rotary_pos_emb(
            query_states, key_states, cos, sin, position_ids
        )

        if past_key_value is not None:
            # reuse k, v, self_attention
            key_states = torch.cat([past_key_value[0], key_states], dim=2)
            value_states = torch.cat([past_key_value[1], value_states], dim=2)

        past_key_value = (key_states, value_states) if use_cache else None

        # repeat k/v heads if n_kv_heads < n_heads
        key_states = repeat_kv(key_states, self.num_key_value_groups)
        value_states = repeat_kv(value_states, self.num_key_value_groups)

        attn_weights = torch.matmul(
            query_states, key_states.transpose(2, 3)
        ) / math.sqrt(
            self.head_dim
        )  # dimension: (bsz, num_heads, q_len, kv_seq_len)

        if attn_weights.size() != (bsz, self.num_heads, q_len, kv_seq_len):
            raise ValueError(
                f"Attention weights should be of size {(bsz, self.num_heads, q_len, kv_seq_len)}, but is"
                f" {attn_weights.size()}"
            )

        if attention_mask is not None:
            if attention_mask.size() != (bsz, 1, q_len, kv_seq_len):
                raise ValueError(
                    f"Attention mask should be of size {(bsz, 1, q_len, kv_seq_len)}, but is {attention_mask.size()}"
                )
            if (
                self.config.amplify_reverse
                and k_token_mask is not None
                and (self.layer_heads is None or len(self.layer_heads) != 0)
            ):
                attention_mask = self._prepare_attention_mask(
                    attention_mask,
                    k_token_mask,
                    q_token_mask,
                    bsz,
                    q_len,
                    kv_seq_len,
                )
            if (
                k_token_mask is not None
                and self.config.amplify_skip_stopwords == "temp"
                and q_len == 1
            ):
                non_zero_mask = (
                    self.k_token_mask[:, None, None, :]
                    & self.is_input_stopwords_tensor[:, None, None, None]
                )
                # if dist.is_initialized() and dist.get_rank() == 0:
                #     if not any(self.is_input_stopwords_tensor):
                #         print(
                #             "Before Softmax",
                #             nn.functional.softmax(
                #                 attn_weights + attention_mask, dim=-1, dtype=torch.float32
                #             )[self.is_input_stopwords_tensor][0, 0, 0, :-50],
                #         )
                attn_weights[
                    non_zero_mask.expand(-1, self.num_heads, -1, -1)
                ] /= self.config.amplify_skip_penalty
                # if dist.is_initialized() and dist.get_rank() == 0:
                #     if not any(self.is_input_stopwords_tensor):
                #         print(
                #             "After Softmax",
                #             nn.functional.softmax(
                #                 attn_weights + attention_mask, dim=-1, dtype=torch.float32
                #             )[self.is_input_stopwords_tensor][0, 0, 0, :-50],
                #         )
            attn_weights = attn_weights + attention_mask

        # upcast attention to fp32
        attn_weights = nn.functional.softmax(attn_weights, dim=-1, dtype=torch.float32)
        # .to(
        #     query_states.dtype
        # )  # dimension: (bsz, num_heads, q_len, kv_seq_len)
        if (
            not self.config.amplify_reverse
            and k_token_mask is not None
            and (self.layer_heads is None or len(self.layer_heads) != 0)
        ):
            expanded_token_mask = (
                self._prepare_expanded_token_mask(
                    k_token_mask,
                    attn_weights,
                    q_token_mask,
                    bsz,
                    q_len,
                    kv_seq_len,
                    tgt_index,
                )
                if self.config.amplify_soft_mask is None
                else self._prepare_expanded_token_mask_soft(
                    k_token_mask,
                    attn_weights,
                    q_token_mask,
                    bsz,
                    q_len,
                    kv_seq_len,
                    tgt_index,
                )
            )
            if expanded_token_mask is not None:
                if self.config.rg_grad_head_score == "concrete":
                    expanded_token_mask = expanded_token_mask.expand(
                        -1, self.num_heads, -1, -1
                    )
                    expanded_token_mask = self.params(expanded_token_mask)
                elif self.config.rg_grad_head_score == "discrete":
                    expanded_token_mask = expanded_token_mask.expand(
                        -1, self.num_heads, -1, -1
                    )
                    expanded_token_mask = expanded_token_mask * self.soft_gate.exp()
                increment = attn_weights * expanded_token_mask
                if self.config.rg_grad_head_score == "concrete_post":
                    increment = self.params(increment)
                elif self.config.rg_grad_head_score == "discrete_post":
                    increment = increment * self.soft_gate.exp()

                # Get the first non-zero position in attention_mask
                if self.config.amplify_use_sink:
                    mask_idx = torch.arange(
                        q_token_mask.shape[-1], 0, -1, device=q_token_mask.device
                    )
                    masked_q_token_mask = mask_idx * q_token_mask
                    first_non_zero_positions = torch.argmax(masked_q_token_mask, dim=-1)
                    increment_sum = increment.sum(dim=-1, keepdim=True)
                    attn_weights = attn_weights.scatter_add(
                        -1,
                        first_non_zero_positions[:, None, None, None],
                        -increment_sum,
                    )
                    attn_weights = attn_weights.clamp(min=0)
                # if dist.is_initialized() and dist.get_rank() == 0:
                #     print("expanded_token_mask", expanded_token_mask[0, 0, -1, :])
                #     print("attn_weights", attn_weights[0, 0, -1, :])
                #     mask = (k_token_mask != 0).int()
                #     mask = torch.cat(
                #         [
                #             mask,
                #             torch.ones(
                #                 (bsz, kv_seq_len - mask.shape[-1]),
                #                 device=mask.device,
                #                 dtype=mask.dtype,
                #             ),
                #         ],
                #         dim=-1,
                #     )
                #     print(
                #         "avg_attn",
                #         (attn_weights[0, 0, -1, :] * mask[0]).sum() / mask[0].sum(),
                #     )
                #     print("Before entropy", self.get_attn_weight_entropy(attn_weights))
                attn_weights = attn_weights + increment
                # if dist.is_initialized() and dist.get_rank() == 0:
                #     print("After entropy", self.get_attn_weight_entropy(attn_weights))
                #     non_zero_mask = (attn_weights != 0) & self.is_input_stopwords_tensor[:, None, None, None]
                #     print("Temp Before", attn_weights[self.is_input_stopwords_tensor])
                #     attn_weights[non_zero_mask] += self.config.amplify_skip_penalty
                #     print("Temp After", attn_weights[self.is_input_stopwords_tensor])
                attn_weights = attn_weights / attn_weights.sum(dim=-1, keepdim=True)
        if self.config.amplify_smooth_window is not None and q_len != 1:
            pooled_attn_weights = F.avg_pool1d(
                attn_weights.view(-1, 1, attn_weights.shape[-1]),
                kernel_size=self.config.amplify_smooth_window,
                stride=1,
                padding=self.config.amplify_smooth_window // 2,
            ).view(bsz, self.num_heads, q_len, kv_seq_len)
            # TODO consider left padding
            # if q_len != 1:
            #     assert q_len == kv_seq_len
            #     tril_mask = ~torch.tril(
            #         torch.ones(
            #             q_len, q_len, device=attn_weights.device, dtype=torch.bool
            #         ),
            #         diagonal=-self.config.amplify_smooth_window // 2,
            #     )[None, None, :, :]
            #     pooled_attn_weights[tril_mask] = attn_weights[tril_mask]
            attn_weights = pooled_attn_weights
            attn_weights = attn_weights / attn_weights.sum(dim=-1, keepdim=True)

        attn_weights = attn_weights.to(query_states.dtype)
        if self.config.rg_grad_head_score == "qk":
            if self.params is None:
                self.params = torch.ones(
                    (bsz, self.num_heads),
                    requires_grad=True,
                    dtype=attn_weights.dtype,
                    device=attn_weights.device,
                )
            attn_weights = attn_weights * self.params.view(bsz, self.num_heads, 1, 1)
        attn_output = torch.matmul(attn_weights, value_states)
        if attn_output.size() != (bsz, self.num_heads, q_len, self.head_dim):
            raise ValueError(
                f"`attn_output` should be of size {(bsz, self.num_heads, q_len, self.head_dim)}, but is"
                f" {attn_output.size()}"
            )
        if self.config.rg_grad_head_score == "v":
            if self.params is None:
                self.params = torch.ones(
                    (bsz, self.num_heads),
                    requires_grad=True,
                    dtype=attn_output.dtype,
                    device=attn_output.device,
                )
            attn_output = attn_output * self.params.view(bsz, self.num_heads, 1, 1)

        attn_output = attn_output.transpose(1, 2).contiguous()
        attn_output = attn_output.reshape(bsz, q_len, self.hidden_size)

        if self.pretraining_tp > 1:
            attn_output = attn_output.split(
                self.hidden_size // self.pretraining_tp, dim=2
            )
            o_proj_slices = self.o_proj.weight.split(
                self.hidden_size // self.pretraining_tp, dim=1
            )
            attn_output = sum(
                [
                    F.linear(attn_output[i], o_proj_slices[i])
                    for i in range(self.pretraining_tp)
                ]
            )
        else:
            attn_output = self.o_proj(attn_output)

        if not output_attentions:
            attn_weights = None

        return attn_output, attn_weights, past_key_value


class LlamaDecoderLayerAttn(nn.Module):
    def __init__(self, config: LlamaConfig, layer_heads=None):
        super().__init__()
        self.hidden_size = config.hidden_size
        self.self_attn = LlamaAttentionAmplify(config=config, layer_heads=layer_heads)
        self.mlp = LlamaMLP(config)
        self.input_layernorm = LlamaRMSNorm(config.hidden_size, eps=config.rms_norm_eps)
        self.post_attention_layernorm = LlamaRMSNorm(
            config.hidden_size, eps=config.rms_norm_eps
        )

    def forward(
        self,
        hidden_states: torch.Tensor,
        attention_mask: Optional[torch.Tensor] = None,
        position_ids: Optional[torch.LongTensor] = None,
        past_key_value: Optional[Tuple[torch.Tensor]] = None,
        output_attentions: Optional[bool] = False,
        use_cache: Optional[bool] = False,
        k_token_mask: Optional[torch.Tensor] = None,
        q_token_mask: Optional[torch.Tensor] = None,
        tgt_index: Optional[torch.Tensor] = None,
    ) -> Tuple[
        torch.FloatTensor, Optional[Tuple[torch.FloatTensor, torch.FloatTensor]]
    ]:
        """
        Args:
            hidden_states (`torch.FloatTensor`): input to the layer of shape `(batch, seq_len, embed_dim)`
            attention_mask (`torch.FloatTensor`, *optional*): attention mask of size
                `(batch, 1, tgt_len, src_len)` where padding elements are indicated by very large negative values.
            output_attentions (`bool`, *optional*):
                Whether or not to return the attentions tensors of all attention layers. See `attentions` under
                returned tensors for more detail.
            use_cache (`bool`, *optional*):
                If set to `True`, `past_key_values` key value states are returned and can be used to speed up decoding
                (see `past_key_values`).
            past_key_value (`Tuple(torch.FloatTensor)`, *optional*): cached past key and value projection states
        """

        residual = hidden_states

        hidden_states = self.input_layernorm(hidden_states)

        # Self Attention
        hidden_states, self_attn_weights, present_key_value = self.self_attn(
            hidden_states=hidden_states,
            attention_mask=attention_mask,
            position_ids=position_ids,
            past_key_value=past_key_value,
            output_attentions=output_attentions,
            use_cache=use_cache,
            k_token_mask=k_token_mask,
            q_token_mask=q_token_mask,
            tgt_index=tgt_index,
        )
        hidden_states = residual + hidden_states

        # Fully Connected
        residual = hidden_states
        hidden_states = self.post_attention_layernorm(hidden_states)
        hidden_states = self.mlp(hidden_states)
        hidden_states = residual + hidden_states

        outputs = (hidden_states,)

        if output_attentions:
            outputs += (self_attn_weights,)

        if use_cache:
            outputs += (present_key_value,)

        return outputs


LLAMA_START_DOCSTRING = r"""
    This model inherits from [`PreTrainedModel`]. Check the superclass documentation for the generic methods the
    library implements for all its model (such as downloading or saving, resizing the input embeddings, pruning heads
    etc.)

    This model is also a PyTorch [torch.nn.Module](https://pytorch.org/docs/stable/nn.html#torch.nn.Module) subclass.
    Use it as a regular PyTorch Module and refer to the PyTorch documentation for all matter related to general usage
    and behavior.

    Parameters:
        config ([`LlamaConfig`]):
            Model configuration class with all the parameters of the model. Initializing with a config file does not
            load the weights associated with the model, only the configuration. Check out the
            [`~PreTrainedModel.from_pretrained`] method to load the model weights.
"""


@add_start_docstrings(
    "The bare LLaMA Model outputting raw hidden-states without any specific head on top.",
    LLAMA_START_DOCSTRING,
)
class LlamaPreTrainedModel(PreTrainedModel):
    config_class = LlamaConfig
    base_model_prefix = "model"
    supports_gradient_checkpointing = True
    _no_split_modules = ["LlamaDecoderLayer"]
    _skip_keys_device_placement = "past_key_values"

    def _init_weights(self, module):
        std = self.config.initializer_range
        if isinstance(module, nn.Linear):
            module.weight.data.normal_(mean=0.0, std=std)
            if module.bias is not None:
                module.bias.data.zero_()
        elif isinstance(module, nn.Embedding):
            module.weight.data.normal_(mean=0.0, std=std)
            if module.padding_idx is not None:
                module.weight.data[module.padding_idx].zero_()

    def _set_gradient_checkpointing(self, module, value=False):
        if isinstance(module, LlamaModel):
            module.gradient_checkpointing = value


LLAMA_INPUTS_DOCSTRING = r"""
    Args:
        input_ids (`torch.LongTensor` of shape `(batch_size, sequence_length)`):
            Indices of input sequence tokens in the vocabulary. Padding will be ignored by default should you provide
            it.

            Indices can be obtained using [`AutoTokenizer`]. See [`PreTrainedTokenizer.encode`] and
            [`PreTrainedTokenizer.__call__`] for details.

            [What are input IDs?](../glossary#input-ids)
        attention_mask (`torch.Tensor` of shape `(batch_size, sequence_length)`, *optional*):
            Mask to avoid performing attention on padding token indices. Mask values selected in `[0, 1]`:

            - 1 for tokens that are **not masked**,
            - 0 for tokens that are **masked**.

            [What are attention masks?](../glossary#attention-mask)

            Indices can be obtained using [`AutoTokenizer`]. See [`PreTrainedTokenizer.encode`] and
            [`PreTrainedTokenizer.__call__`] for details.

            If `past_key_values` is used, optionally only the last `decoder_input_ids` have to be input (see
            `past_key_values`).

            If you want to change padding behavior, you should read [`modeling_opt._prepare_decoder_attention_mask`]
            and modify to your needs. See diagram 1 in [the paper](https://arxiv.org/abs/1910.13461) for more
            information on the default strategy.

            - 1 indicates the head is **not masked**,
            - 0 indicates the head is **masked**.
        position_ids (`torch.LongTensor` of shape `(batch_size, sequence_length)`, *optional*):
            Indices of positions of each input sequence tokens in the position embeddings. Selected in the range `[0,
            config.n_positions - 1]`.

            [What are position IDs?](../glossary#position-ids)
        past_key_values (`tuple(tuple(torch.FloatTensor))`, *optional*, returned when `use_cache=True` is passed or when `config.use_cache=True`):
            Tuple of `tuple(torch.FloatTensor)` of length `config.n_layers`, with each tuple having 2 tensors of shape
            `(batch_size, num_heads, sequence_length, embed_size_per_head)`) and 2 additional tensors of shape
            `(batch_size, num_heads, encoder_sequence_length, embed_size_per_head)`.

            Contains pre-computed hidden-states (key and values in the self-attention blocks and in the cross-attention
            blocks) that can be used (see `past_key_values` input) to speed up sequential decoding.

            If `past_key_values` are used, the user can optionally input only the last `decoder_input_ids` (those that
            don't have their past key value states given to this model) of shape `(batch_size, 1)` instead of all
            `decoder_input_ids` of shape `(batch_size, sequence_length)`.
        inputs_embeds (`torch.FloatTensor` of shape `(batch_size, sequence_length, hidden_size)`, *optional*):
            Optionally, instead of passing `input_ids` you can choose to directly pass an embedded representation. This
            is useful if you want more control over how to convert `input_ids` indices into associated vectors than the
            model's internal embedding lookup matrix.
        use_cache (`bool`, *optional*):
            If set to `True`, `past_key_values` key value states are returned and can be used to speed up decoding (see
            `past_key_values`).
        output_attentions (`bool`, *optional*):
            Whether or not to return the attentions tensors of all attention layers. See `attentions` under returned
            tensors for more detail.
        output_hidden_states (`bool`, *optional*):
            Whether or not to return the hidden states of all layers. See `hidden_states` under returned tensors for
            more detail.
        return_dict (`bool`, *optional*):
            Whether or not to return a [`~utils.ModelOutput`] instead of a plain tuple.
"""


@add_start_docstrings(
    "The bare LLaMA Model outputting raw hidden-states without any specific head on top.",
    LLAMA_START_DOCSTRING,
)
class LlamaModel(LlamaPreTrainedModel):
    """
    Transformer decoder consisting of *config.num_hidden_layers* layers. Each layer is a [`LlamaDecoderLayer`]

    Args:
        config: LlamaConfig
    """

    def __init__(self, config: LlamaConfig):
        super().__init__(config)
        self.padding_idx = config.pad_token_id
        self.vocab_size = config.vocab_size

        self.embed_tokens = nn.Embedding(
            config.vocab_size, config.hidden_size, self.padding_idx
        )
        print("Layer Head Config", config.amplify_layer_head)
        self.layers = nn.ModuleList(
            [
                LlamaDecoderLayerAttn(
                    config,
                    layer_heads=None
                    if config.amplify_layer_head is None
                    else config.amplify_layer_head.get(str(i), []),
                )
                for i in range(config.num_hidden_layers)
            ]
        )
        self.norm = LlamaRMSNorm(config.hidden_size, eps=config.rms_norm_eps)

        self.gradient_checkpointing = False
        # Initialize weights and apply final processing
        self.post_init()

    def get_input_embeddings(self):
        return self.embed_tokens

    def set_input_embeddings(self, value):
        self.embed_tokens = value

    # Copied from transformers.models.bart.modeling_bart.BartDecoder._prepare_decoder_attention_mask
    def _prepare_decoder_attention_mask(
        self, attention_mask, input_shape, inputs_embeds, past_key_values_length
    ):
        # create causal mask
        # [bsz, seq_len] -> [bsz, 1, tgt_seq_len, src_seq_len]
        combined_attention_mask = None
        if input_shape[-1] > 1:
            combined_attention_mask = _make_causal_mask(
                input_shape,
                inputs_embeds.dtype,
                device=inputs_embeds.device,
                past_key_values_length=past_key_values_length,
            )

        if attention_mask is not None:
            # [bsz, seq_len] -> [bsz, 1, tgt_seq_len, src_seq_len]
            expanded_attn_mask = _expand_mask(
                attention_mask, inputs_embeds.dtype, tgt_len=input_shape[-1]
            ).to(inputs_embeds.device)
            combined_attention_mask = (
                expanded_attn_mask
                if combined_attention_mask is None
                else expanded_attn_mask + combined_attention_mask
            )

        return combined_attention_mask

    @add_start_docstrings_to_model_forward(LLAMA_INPUTS_DOCSTRING)
    def forward(
        self,
        input_ids: torch.LongTensor = None,
        attention_mask: Optional[torch.Tensor] = None,
        position_ids: Optional[torch.LongTensor] = None,
        past_key_values: Optional[List[torch.FloatTensor]] = None,
        inputs_embeds: Optional[torch.FloatTensor] = None,
        use_cache: Optional[bool] = None,
        output_attentions: Optional[bool] = None,
        output_hidden_states: Optional[bool] = None,
        k_token_mask: Optional[torch.Tensor] = None,
        q_token_mask: Optional[torch.Tensor] = None,
        tgt_index: Optional[torch.Tensor] = None,
        return_dict: Optional[bool] = None,
    ) -> Union[Tuple, BaseModelOutputWithPast]:
        output_attentions = (
            output_attentions
            if output_attentions is not None
            else self.config.output_attentions
        )
        output_hidden_states = (
            output_hidden_states
            if output_hidden_states is not None
            else self.config.output_hidden_states
        )
        use_cache = use_cache if use_cache is not None else self.config.use_cache

        return_dict = (
            return_dict if return_dict is not None else self.config.use_return_dict
        )

        # retrieve input_ids and inputs_embeds
        if input_ids is not None and inputs_embeds is not None:
            raise ValueError(
                "You cannot specify both decoder_input_ids and decoder_inputs_embeds at the same time"
            )
        elif input_ids is not None:
            batch_size, seq_length = input_ids.shape
        elif inputs_embeds is not None:
            batch_size, seq_length, _ = inputs_embeds.shape
        else:
            raise ValueError(
                "You have to specify either decoder_input_ids or decoder_inputs_embeds"
            )

        seq_length_with_past = seq_length
        past_key_values_length = 0

        if past_key_values is not None:
            past_key_values_length = past_key_values[0][0].shape[2]
            seq_length_with_past = seq_length_with_past + past_key_values_length

        if position_ids is None:
            device = input_ids.device if input_ids is not None else inputs_embeds.device
            position_ids = torch.arange(
                past_key_values_length,
                seq_length + past_key_values_length,
                dtype=torch.long,
                device=device,
            )
            position_ids = position_ids.unsqueeze(0).view(-1, seq_length)
        else:
            position_ids = position_ids.view(-1, seq_length).long()

        if inputs_embeds is None:
            inputs_embeds = self.embed_tokens(input_ids)
        # embed positions
        if attention_mask is None:
            attention_mask = torch.ones(
                (batch_size, seq_length_with_past),
                dtype=torch.bool,
                device=inputs_embeds.device,
            )
        attention_mask = self._prepare_decoder_attention_mask(
            attention_mask,
            (batch_size, seq_length),
            inputs_embeds,
            past_key_values_length,
        )

        hidden_states = inputs_embeds

        if self.gradient_checkpointing and self.training:
            if use_cache:
                logger.warning_once(
                    "`use_cache=True` is incompatible with gradient checkpointing. Setting `use_cache=False`..."
                )
                use_cache = False

        # decoder layers
        all_hidden_states = () if output_hidden_states else None
        all_self_attns = () if output_attentions else None
        next_decoder_cache = () if use_cache else None

        for idx, decoder_layer in enumerate(self.layers):
            if output_hidden_states:
                all_hidden_states += (hidden_states,)

            past_key_value = (
                past_key_values[idx] if past_key_values is not None else None
            )

            use_amplify_attn = (
                idx in range(self.config.start_layer, self.config.end_layer + 1)
                if self.config.amplify_layer is None
                else idx in self.config.amplify_layer
            )
            # if use_amplify_attn:
            #     print("Layers", idx)

            if self.gradient_checkpointing and self.training:
                def create_custom_forward(module):
                    def custom_forward(*inputs):
                        *actual_inputs, k_token_mask, q_token_mask, tgt_index = inputs
                        # None for past_key_value
                        return module(
                            *actual_inputs,
                            output_attentions,
                            None,
                            k_token_mask,
                            q_token_mask,
                            tgt_index,
                        )

                    return custom_forward

                layer_outputs = torch.utils.checkpoint.checkpoint(
                    create_custom_forward(decoder_layer),
                    hidden_states,
                    attention_mask,
                    position_ids,
                    None,
                    k_token_mask if use_amplify_attn else None,
                    q_token_mask if use_amplify_attn else None,
                    tgt_index if use_amplify_attn else None,
                )
            else:
                layer_outputs = decoder_layer(
                    hidden_states,
                    attention_mask=attention_mask,
                    position_ids=position_ids,
                    past_key_value=past_key_value,
                    output_attentions=output_attentions,
                    use_cache=use_cache,
                    k_token_mask=k_token_mask if use_amplify_attn else None,
                    q_token_mask=q_token_mask if use_amplify_attn else None,
                    tgt_index=tgt_index if use_amplify_attn else None,
                )

            hidden_states = layer_outputs[0]

            if use_cache:
                next_decoder_cache += (layer_outputs[2 if output_attentions else 1],)

            if output_attentions:
                all_self_attns += (layer_outputs[1],)

        hidden_states = self.norm(hidden_states)

        # add hidden states from the last decoder layer
        if output_hidden_states:
            all_hidden_states += (hidden_states,)

        next_cache = next_decoder_cache if use_cache else None
        if not return_dict:
            return tuple(
                v
                for v in [hidden_states, next_cache, all_hidden_states, all_self_attns]
                if v is not None
            )
        return BaseModelOutputWithPast(
            last_hidden_state=hidden_states,
            past_key_values=next_cache,
            hidden_states=all_hidden_states,
            attentions=all_self_attns,
        )


class LlamaForCausalLMAttn(LlamaPreTrainedModel):
    _tied_weights_keys = ["lm_head.weight"]

    def __init__(self, config, **kwargs):
        super().__init__(config)
        self.model = LlamaModel(config)
        self.pretraining_tp = config.pretraining_tp
        self.vocab_size = config.vocab_size
        self.lm_head = nn.Linear(config.hidden_size, config.vocab_size, bias=False)

        # Initialize weights and apply final processing
        self.post_init()

    def get_input_embeddings(self):
        return self.model.embed_tokens

    def set_input_embeddings(self, value):
        self.model.embed_tokens = value

    def get_output_embeddings(self):
        return self.lm_head

    def set_output_embeddings(self, new_embeddings):
        self.lm_head = new_embeddings

    def set_decoder(self, decoder):
        self.model = decoder

    def get_decoder(self):
        return self.model

    def judge_stopwords(self, input_ids):
        punctuation = r"""!"#$%&'(),.:;?@[\]^_`{|}~"""

        def is_stopword(t):
            # Judge whether a token is a stopword
            if t in ["▁", "<0x0A>"]:
                return True
            # Remove punctuations
            t = t.replace("▁", "").lower()
            # if t in string.punctuation:
            if t in punctuation:
                return True
            # english_stopwords = set(stopwords.words('english'))
            # if t.replace("▁", "").lower() in english_stopwords:
            #     return True
            return False

        tokens = self.tokenizer.convert_ids_to_tokens(input_ids.squeeze(dim=-1))
        return [is_stopword(t) for t in tokens]

    def judge_calculation(self, input_ids):
        def is_arith_sym(t):
            if (
                t.isnumeric()
                or t in ["▁+", "▁-", "▁*", "▁/", "▁^", "▁(", "▁)"]
                or t in ["+", "-", "*", "/", "^", "(", ")"]
                or t == "▁x"
            ):
                return True
            if self.config.amplify_exclude_cal == "joint":
                return False
            if t == "▁" or t == ".":  # TODO verify
                return True
            return False

        tokens = self.tokenizer.convert_ids_to_tokens(input_ids.squeeze(dim=-1))

        for i, t in enumerate(tokens):
            if self.config.amplify_exclude_cal == "joint":
                self.is_cals[i] = is_arith_sym(t)
            elif "=" in t:
                if (
                    dist.is_initialized()
                    and dist.get_rank() == 0
                    and not self.is_cals[i]
                ):
                    print(f"Find {t} in {tokens}. Enable calculation.")
                self.is_cals[i] = True
            elif not is_arith_sym(t):
                if dist.is_initialized() and dist.get_rank() == 0 and self.is_cals[i]:
                    print(f"Find {t} in {tokens}. Disable calculation.")
                self.is_cals[i] = False

    def _init_prepare_expanded_token_mask(
        self,
        k_token_mask,
        bsz,  # dimension: (bsz)
        q_len,  # dimension: (q_len)
        kv_seq_len,  # dimension: (kv_seq_len)
    ):
        amplify_factor = self.config.amplify_factor
        assert amplify_factor != 0
        is_cache = kv_seq_len != k_token_mask.shape[1]
        if is_cache:
            if self.config.amplify_q_scope == "question":
                return None
            k_token_mask = torch.cat(
                [
                    k_token_mask,
                    torch.full(
                        size=(
                            k_token_mask.shape[0],
                            kv_seq_len - k_token_mask.shape[-1],
                        ),
                        fill_value=_get_fill_value(self.config, amplify_factor),
                        device=k_token_mask.device,
                        dtype=k_token_mask.dtype,
                    ),
                ],
                dim=1,
            )

        expanded_k_token_mask = (
            k_token_mask[:, None, :]
            .expand(bsz, q_len, kv_seq_len)
            .to(k_token_mask.dtype)
        )

        if (
            self.config.amplify_exclude_cal == "normal"
            or self.config.amplify_exclude_cal == "remain"
        ):
            is_cals_tensor = torch.tensor(
                self.is_cals, dtype=torch.bool, device=k_token_mask.device
            )
            expanded_k_token_mask[is_cals_tensor] = 0

        if self.config.amplify_skip_stopwords is not None and q_len == 1:
            is_input_stopwords_tensor = torch.tensor(
                self.is_input_stopwords, dtype=torch.bool, device=k_token_mask.device
            )
            if (
                self.config.amplify_skip_stopwords == "normal"
                or self.config.amplify_skip_stopwords == "temp"
            ):
                expanded_k_token_mask[is_input_stopwords_tensor] = 0
            elif self.config.amplify_skip_stopwords == "reverse":
                expanded_k_token_mask[~is_input_stopwords_tensor] = 0
            elif self.config.amplify_skip_stopwords == "amplify":
                expanded_k_token_mask[
                    ~is_input_stopwords_tensor
                ] *= self.config.amplify_skip_penalty
            if self.config.amplify_skip_stopwords == "temp":
                # Set is_input_stopwords_tensor to all attention
                for layer in self.model.layers:
                    layer.self_attn.is_input_stopwords_tensor = (
                        is_input_stopwords_tensor
                    )
                    layer.self_attn.k_token_mask = k_token_mask != 0
                print("temptemp")
        return expanded_k_token_mask

    def _prepare_expanded_token_mask(
        self,
        k_token_mask,  # dimension: (bsz, seq_len)
        attn_weights,  # dimension: (bsz, num_heads, q_len, kv_seq_len)
        bsz,  # dimension: (bsz)
        q_len,  # dimension: (q_len)
        kv_seq_len,  # dimension: (kv_seq_len)
        tgt_index,
        hidden_states,
    ):
        amplify_factor = self.config.amplify_factor
        assert amplify_factor != 0
        is_cache = kv_seq_len != k_token_mask.shape[1]
        if is_cache:
            if self.config.amplify_q_scope == "question":
                return None
            k_token_mask = torch.cat(
                [
                    k_token_mask,
                    torch.full(
                        size=(
                            k_token_mask.shape[0],
                            kv_seq_len - k_token_mask.shape[-1],
                        ),
                        fill_value=_get_fill_value(self.config, amplify_factor),
                        device=k_token_mask.device,
                        dtype=k_token_mask.dtype,
                    ),
                ],
                dim=1,
            )
        if (
            self.config.amplify_total_threshold_output is not None
            or self.config.amplify_exclude_self
            or self.config.amplify_total_threshold_upper is not None
        ):
            range_tensor = (
                torch.arange(kv_seq_len)
                .expand(bsz, kv_seq_len)
                .to(device=tgt_index.device)
            )
            bool_mask = range_tensor >= tgt_index.unsqueeze(
                1
            )  # shape (bsz, kv_seq_len)
            if self.config.amplify_total_threshold_output is not None:
                threshold = torch.full_like(
                    k_token_mask, self.config.amplify_total_threshold
                )
                threshold.masked_fill_(
                    bool_mask, self.config.amplify_total_threshold_output
                )

        expanded_k_token_mask = (
            k_token_mask[:, None, :]
            .expand(bsz, q_len, kv_seq_len)
            .to(attn_weights.dtype)
        )
        if (
            self.config.amplify_exclude_cal == "normal"
            or self.config.amplify_exclude_cal == "remain"
        ):
            expanded_k_token_mask[is_cals_tensor] = 0

        if self.config.amplify_skip_stopwords is not None and q_len == 1:
            is_input_stopwords_tensor = torch.tensor(
                self.is_input_stopwords, dtype=torch.bool, device=k_token_mask.device
            )
            if (
                self.config.amplify_skip_stopwords == "normal"
                or self.config.amplify_skip_stopwords == "temp"
            ):
                expanded_k_token_mask[is_input_stopwords_tensor] = 0
            elif self.config.amplify_skip_stopwords == "reverse":
                expanded_k_token_mask[~is_input_stopwords_tensor] = 0
            elif self.config.amplify_skip_stopwords == "amplify":
                # expanded_k_token_mask[is_input_stopwords_tensor] *= 1.5
                expanded_k_token_mask[
                    ~is_input_stopwords_tensor
                ] *= self.config.amplify_skip_penalty
            if self.config.amplify_skip_stopwords == "temp":
                # Set is_input_stopwords_tensor to all attention
                for layer in self.model.layers:
                    layer.self_attn.is_input_stopwords_tensor = (
                        is_input_stopwords_tensor
                    )
                    layer.self_attn.k_token_mask = k_token_mask != 0
        if self.config.amplify_uncert_threshold is not None:
            logits = self.lm_head(hidden_states)
            logits = logits.float()
            output_ids = logits.argmax(dim=-1)[:, -1]
            self.is_stops = self.judge_stopwords(output_ids)
            is_stopwords_tensor = torch.tensor(
                self.is_stops, dtype=torch.bool, device=k_token_mask.device
            )
            if self.config.amplify_uncert_score_type == "neg_logp":
                uncert_scores = -(
                    torch.log_softmax(logits[:, -1], dim=-1).max(dim=-1).values
                )  # dimension: (bsz)
            else:
                log_probs = torch.log_softmax(logits[:, -1], dim=-1)
                top2_probs, _ = torch.topk(log_probs, k=2, dim=-1)
                uncert_scores = top2_probs[:, 1] - top2_probs[:, 0]  # negative
            is_uncert = uncert_scores >= self.config.amplify_uncert_threshold
            is_uncert &= ~is_stopwords_tensor
            self.is_uncert = is_uncert.cpu().numpy()
        # if self.config.amplify_input_uncert_percentage is not None:
        #     if not self.is_cache:
        #         logits = self.lm_head(hidden_states)
        #         logits = logits.float()
        #         # get logits of input_ids

                

        #     # select top k tokens
        #     topk_logits, topk_indexes = torch.topk(
        #         logits[:, -1], k=self.config.amplify_input_uncert_topk, dim=-1
        #     )
        if self.config.amplify_exclude_cal is not None:
            is_cals_tensor = torch.tensor(
                self.is_cals, dtype=torch.bool, device=k_token_mask.device
            )

        if self.config.amplify_total_threshold is not None:
            binary_old_mask = (expanded_k_token_mask != 0).int()
            masked_attn_weights = attn_weights * binary_old_mask

            if self.config.amplify_total_threshold_output is not None:
                mask = (masked_attn_weights >= threshold.unsqueeze(1)).to(
                    dtype=attn_weights.dtype
                )
            else:
                mask = (masked_attn_weights >= self.config.amplify_total_threshold).to(
                    dtype=attn_weights.dtype
                )

            if self.config.amplify_total_threshold_upper is not None:
                upper_mask = (
                    masked_attn_weights <= self.config.amplify_total_threshold_upper
                ).to(dtype=attn_weights.dtype)

                if self.config.amplify_exclude_cal is not None:
                    if self.config.amplify_exclude_cal == "joint":
                        upper_mask[is_uncert & is_cals_tensor] *= (
                            1 - upper_mask[is_uncert & is_cals_tensor]
                        )
                        # expanded_k_token_mask[is_uncert & is_cals_tensor] = 0
                        # TODO
                        # bool_mask[is_uncert & is_cals_tensor] = 1
                    elif self.config.amplify_exclude_cal == "all":
                        upper_mask[is_cals_tensor] = 1
                    elif self.config.amplify_exclude_cal == "reverse":
                        upper_mask[is_cals_tensor] = 1 - upper_mask[is_cals_tensor]

                temp_mask = mask * upper_mask
                temp_mask.masked_fill_(~(bool_mask[:, None, :]), 1)
                mask = mask * temp_mask

            expanded_k_token_mask = expanded_k_token_mask * mask
        else:
            masked_attn_weights = attn_weights * expanded_k_token_mask
            _, top_indexes = torch.topk(
                masked_attn_weights,
                k=min(self.config.amplify_total_topk, masked_attn_weights.shape[-1]),
                dim=-1,
                sorted=True,
            )  # dimension: (bsz, q_len, amplify_topk)
            ones = torch.full_like(
                top_indexes, amplify_factor, dtype=attn_weights.dtype
            )
            new_expanded_token_mask = torch.zeros_like(attn_weights)
            new_expanded_token_mask.scatter_(-1, top_indexes, ones)
            binary_old_mask = (expanded_k_token_mask != 0).int()
            expanded_k_token_mask = binary_old_mask * new_expanded_token_mask

        if self.config.amplify_uncert_threshold is not None:
            if self.config.amplify_uncert_type == "constant":
                expanded_k_token_mask[is_uncert] *= self.config.amplify_uncert_constant
            elif self.config.amplify_uncert_type == "neg_logp":
                binary_old_mask = (expanded_k_token_mask != 0).int()
                expanded_k_token_mask = (
                    uncert_scores[:, None, None]
                    * binary_old_mask
                    * self.config.amplify_uncert_constant
                ).clamp(max=self.config.amplify_uncert_upper)
            elif self.config.amplify_uncert_type == "joint":
                expanded_k_token_mask[
                    is_uncert & ~is_cals_tensor
                ] *= self.config.amplify_uncert_constant
                if dist.is_initialized() and dist.get_rank() == 0:
                    print("is_cals", is_cals_tensor)
                    print("is_uncert", is_uncert)
                    print("is_stopwords", is_stopwords_tensor)
            if dist.is_initialized() and dist.get_rank() == 0:
                print(uncert_scores)
                if hasattr(self, "tokenizer"):
                    self.decode_tokens = self.tokenizer.convert_ids_to_tokens(
                        torch.log_softmax(logits[:, -1], dim=-1).argmax(dim=-1)
                    )
                    print("Before input tokens", self.decode_tokens)

        return expanded_k_token_mask

    def _prepare_cal_stopwords(self, input_ids):
        if self.config.amplify_exclude_cal is not None:
            if input_ids.shape[-1] != 1:
                self.is_cals = [False] * input_ids.shape[0]
            else:
                self.judge_calculation(input_ids)
        if self.config.amplify_skip_stopwords is not None:
            if input_ids.shape[-1] != 1:
                self.is_input_stopwords = [False] * input_ids.shape[0]
            else:
                self.is_input_stopwords = self.judge_stopwords(input_ids)
            if dist.is_initialized() and dist.get_rank() == 0:
                print(self.is_input_stopwords)
                print(
                    self.tokenizer.convert_ids_to_tokens(
                        input_ids[:, -1].squeeze(dim=-1)
                    )
                )

    def grad_process(self, grad, use_abs="after", grad_type="layer"):
        if use_abs is not None:
            grad = abs(grad)
        return grad

    def get_attn_grad(self, use_abs="after", grad_type="layer"):
        grads = []
        for layer in self.model.layers:
            grad = self.grad_process(
                layer.self_attn.params.grad.detach(),
                use_abs=use_abs,
                grad_type=grad_type,
            )
            grads.append(grad)
        return grads

    def clear_attn_grad(self):
        for layer in self.model.layers:
            if layer.self_attn.params is not None:
                # layer.self_attn.params.grad = None
                layer.self_attn.params = None

    @add_start_docstrings_to_model_forward(LLAMA_INPUTS_DOCSTRING)
    @replace_return_docstrings(
        output_type=CausalLMOutputWithPast, config_class=_CONFIG_FOR_DOC
    )
    def forward(
        self,
        input_ids: torch.LongTensor = None,
        attention_mask: Optional[torch.Tensor] = None,
        position_ids: Optional[torch.LongTensor] = None,
        past_key_values: Optional[List[torch.FloatTensor]] = None,
        inputs_embeds: Optional[torch.FloatTensor] = None,
        labels: Optional[torch.LongTensor] = None,
        use_cache: Optional[bool] = None,
        output_attentions: Optional[bool] = None,
        output_hidden_states: Optional[bool] = None,
        k_token_mask: Optional[torch.Tensor] = None,
        q_token_mask: Optional[torch.LongTensor] = None,
        tgt_index: Optional[torch.LongTensor] = None,
        return_dict: Optional[bool] = None,
    ) -> Union[Tuple, CausalLMOutputWithPast]:
        r"""
        Args:
            labels (`torch.LongTensor` of shape `(batch_size, sequence_length)`, *optional*):
                Labels for computing the masked language modeling loss. Indices should either be in `[0, ...,
                config.vocab_size]` or -100 (see `input_ids` docstring). Tokens with indices set to `-100` are ignored
                (masked), the loss is only computed for the tokens with labels in `[0, ..., config.vocab_size]`.

        Returns:

        Example:

        ```python
        >>> from transformers import AutoTokenizer, LlamaForCausalLM

        >>> model = LlamaForCausalLM.from_pretrained(PATH_TO_CONVERTED_WEIGHTS)
        >>> tokenizer = AutoTokenizer.from_pretrained(PATH_TO_CONVERTED_TOKENIZER)

        >>> prompt = "Hey, are you conscious? Can you talk to me?"
        >>> inputs = tokenizer(prompt, return_tensors="pt")

        >>> # Generate
        >>> generate_ids = model.generate(inputs.input_ids, max_length=30)
        >>> tokenizer.batch_decode(generate_ids, skip_special_tokens=True, clean_up_tokenization_spaces=False)[0]
        "Hey, are you conscious? Can you talk to me?\nI'm not conscious, but I can talk to you."
        ```"""

        output_attentions = (
            output_attentions
            if output_attentions is not None
            else self.config.output_attentions
        )
        output_hidden_states = (
            output_hidden_states
            if output_hidden_states is not None
            else self.config.output_hidden_states
        )
        return_dict = (
            return_dict if return_dict is not None else self.config.use_return_dict
        )

        do_two_forward = (
            (
                self.config.amplify_total_topk is not None
                or self.config.amplify_uncert_threshold is not None
            )
            and k_token_mask is not None
            and not self.config.amplify_layer_threshold
        )
        # TODO
        # self._prepare_cal_stopwords(input_ids)

        if (
            not do_two_forward
            and k_token_mask is not None
            and (
                self.config.amplify_exclude_cal is not None
                or self.config.amplify_skip_stopwords is not None
            )
        ):
            expanded_k_token_mask = self._init_prepare_expanded_token_mask(
                k_token_mask,
                input_ids.shape[0],
                input_ids.shape[-1],
                past_key_values[0][0].shape[2] + input_ids.shape[-1]
                if past_key_values is not None
                else input_ids.shape[-1],
            )
            input_k_token_mask = expanded_k_token_mask
        else:
            input_k_token_mask = k_token_mask

        # decoder outputs consists of (dec_features, layer_state, dec_hidden, dec_attn)
        outputs = self.model(
            input_ids=input_ids,
            attention_mask=attention_mask,
            position_ids=position_ids,
            past_key_values=past_key_values,
            inputs_embeds=inputs_embeds,
            use_cache=use_cache,
            output_attentions=output_attentions or do_two_forward,
            output_hidden_states=output_hidden_states,
            k_token_mask=input_k_token_mask if not do_two_forward else None,
            q_token_mask=q_token_mask if not do_two_forward else None,
            tgt_index=tgt_index if not do_two_forward else None,
            return_dict=return_dict,
        )
        ret_past_key_values = outputs.past_key_values
        ret_hidden_states = outputs.hidden_states
        ret_attentions = outputs.attentions

        if do_two_forward:
            attentions = outputs.attentions
            total_attn_weights = (sum(attentions) / len(attentions)).mean(dim=1)
            expanded_k_token_mask = self._prepare_expanded_token_mask(
                k_token_mask,
                total_attn_weights,
                input_ids.shape[0],
                total_attn_weights.shape[-2],
                total_attn_weights.shape[-1],
                tgt_index,
                outputs[0],
            )
            if dist.is_initialized() and dist.get_rank() == 0:
                print("input_ids", input_ids[0])
                print("expanded_k_token_mask", expanded_k_token_mask[0])

            # decoder outputs consists of (dec_features, layer_state, dec_hidden, dec_attn)
            outputs = self.model(
                input_ids=input_ids,
                attention_mask=attention_mask,
                position_ids=position_ids,
                past_key_values=past_key_values,
                inputs_embeds=inputs_embeds,
                use_cache=use_cache,
                output_attentions=output_attentions,
                output_hidden_states=output_hidden_states,
                k_token_mask=expanded_k_token_mask,
                q_token_mask=q_token_mask,
                tgt_index=tgt_index,
                return_dict=return_dict,
            )
            if self.config.amplify_exclude_cal != "remain":
                ret_past_key_values = outputs.past_key_values
                ret_hidden_states = outputs.hidden_states
                ret_attentions = outputs.attentions

        hidden_states = outputs[0]
        logits = self.lm_head(hidden_states)
        logits = logits.float()
        if dist.is_initialized() and dist.get_rank() == 0:
            if hasattr(self, "tokenizer") and hasattr(self, "decode_tokens"):
                decode_tokens = self.tokenizer.convert_ids_to_tokens(
                    torch.log_softmax(logits[:, -1], dim=-1).argmax(dim=-1)
                )
                print("After input tokens", decode_tokens)
                if decode_tokens != self.decode_tokens:
                    print("Different!")
                    print("Before diff tokens", self.decode_tokens)
                    print("After diff tokens", decode_tokens)
                if self.config.amplify_uncert_threshold is not None:
                    print("Uncert")
                    for dt, sdt, is_uncert in zip(
                        decode_tokens, self.decode_tokens, self.is_uncert
                    ):
                        if not is_uncert:
                            continue
                        if dt != sdt:
                            print("Real different token!", dt, sdt)

        loss = None
        if labels is not None:
            # Shift so that tokens < n predict n
            shift_logits = logits[..., :-1, :].contiguous()
            shift_labels = labels[..., 1:].contiguous()
            # Flatten the tokens
            loss_fct = CrossEntropyLoss()
            shift_logits = shift_logits.view(-1, self.config.vocab_size)
            shift_labels = shift_labels.view(-1)
            # Enable model parallelism
            shift_labels = shift_labels.to(shift_logits.device)
            loss = loss_fct(shift_logits, shift_labels)

        if not return_dict:
            output = (logits,) + outputs[1:]
            return (loss,) + output if loss is not None else output

        return CausalLMOutputWithPast(
            loss=loss,
            logits=logits,
            past_key_values=ret_past_key_values,
            hidden_states=ret_hidden_states,
            attentions=ret_attentions,
        )

    def prepare_inputs_for_generation(
        self,
        input_ids,
        past_key_values=None,
        attention_mask=None,
        inputs_embeds=None,
        **kwargs,
    ):
        if past_key_values:
            input_ids = input_ids[:, -1:]

        position_ids = kwargs.get("position_ids", None)
        k_token_mask = kwargs.get("k_token_mask", None)
        q_token_mask = kwargs.get("q_token_mask", None)
        tgt_index = kwargs.get("tgt_index", None)
        if attention_mask is not None and position_ids is None:
            # create position_ids on the fly for batch generation
            position_ids = attention_mask.long().cumsum(-1) - 1
            position_ids.masked_fill_(attention_mask == 0, 1)
            if past_key_values:
                position_ids = position_ids[:, -1].unsqueeze(-1)

        # if `inputs_embeds` are passed, we only want to use them in the 1st generation step
        if inputs_embeds is not None and past_key_values is None:
            model_inputs = {"inputs_embeds": inputs_embeds}
        else:
            model_inputs = {"input_ids": input_ids}

        model_inputs.update(
            {
                "position_ids": position_ids,
                "past_key_values": past_key_values,
                "use_cache": kwargs.get("use_cache"),
                "attention_mask": attention_mask,
                "k_token_mask": k_token_mask,
                "q_token_mask": q_token_mask,
                "tgt_index": tgt_index,
            }
        )
        return model_inputs

    @staticmethod
    def _reorder_cache(past_key_values, beam_idx):
        reordered_past = ()
        for layer_past in past_key_values:
            reordered_past += (
                tuple(
                    past_state.index_select(0, beam_idx.to(past_state.device))
                    for past_state in layer_past
                ),
            )
        return reordered_past
