# DIP Sampling Batch for Water-Prob-V2

import torch
import ujson
import os
import numpy as np
from torch.nn import functional as F
from typing import Union
from transformers import AutoTokenizer
import sys
from itertools import product
import hashlib
import random
import pickle

# Logits & Prompts
json_file_path_1 = "../../data/results/prob2/dip-p1"
json_file_path_2 = "../../data/results/prob2/dip-p2"
prompt_file_path_1 = "../../data/prompts/fixkey-p1.txt"
prompt_file_path_2 = "../../data/prompts/fixkey-p2.txt"

json_file_paths = [json_file_path_1, json_file_path_2]

with open(prompt_file_path_1, "r") as f:
    prompt1 = f.readlines()
    prompt1 = "".join(prompt1)

with open(prompt_file_path_2, "r") as f:
    prompt2 = f.readlines()
    prompt2 = "".join(prompt2)

prompts = [prompt1, prompt2]

# Constants(Fill parts)
letters = [f" {chr(i)}" for i in range(65, 91)]
numbers_en = [
    " zero",
    " one",
    " two",
    " three",
    " four",
    " five",
    " six",
    " seven",
    " eight",
    " nine",
]
animal_choice = [" cat", " dog", " tiger", " lion"]
combinations_main = [
    "".join(comb) for comb in product(letters, numbers_en, animal_choice)
]


def from_random(
    rng: Union[torch.Generator, list[torch.Generator]], vocab_size: int
) -> torch.LongTensor:
    """Generate a permutation from the random number generator."""
    if isinstance(rng, list):
        batch_size = len(rng)
        shuffle = torch.stack(
            [
                torch.randperm(vocab_size, generator=rng[i], device=rng[i].device)
                for i in range(batch_size)
            ]
        )
    else:
        shuffle = torch.randperm(vocab_size, generator=rng, device=rng.device)
    return shuffle


def reweight_logits(
    shuffle: torch.LongTensor, p_logits: torch.FloatTensor, alpha: float
) -> torch.FloatTensor:
    """Reweight the logits using the shuffle and alpha."""
    unshuffle = torch.argsort(shuffle, dim=-1)

    s_p_logits = torch.gather(p_logits, -1, shuffle)
    s_log_cumsum = torch.logcumsumexp(s_p_logits, dim=-1)

    # normalize the log_cumsum to force the last element to be 0
    s_log_cumsum = s_log_cumsum - s_log_cumsum[..., -1:]
    s_cumsum = torch.exp(s_log_cumsum)
    s_p = F.softmax(s_p_logits, dim=-1)

    boundary_1 = torch.argmax((s_cumsum > alpha).to(torch.int), dim=-1, keepdim=True)
    p_boundary_1 = torch.gather(s_p, -1, boundary_1)
    portion_in_right_1 = (torch.gather(s_cumsum, -1, boundary_1) - alpha) / p_boundary_1
    portion_in_right_1 = torch.clamp(portion_in_right_1, 0, 1)
    s_all_portion_in_right_1 = (s_cumsum > alpha).type_as(p_logits)
    s_all_portion_in_right_1.scatter_(-1, boundary_1, portion_in_right_1)

    boundary_2 = torch.argmax(
        (s_cumsum > (1 - alpha)).to(torch.int), dim=-1, keepdim=True
    )
    p_boundary_2 = torch.gather(s_p, -1, boundary_2)
    portion_in_right_2 = (
        torch.gather(s_cumsum, -1, boundary_2) - (1 - alpha)
    ) / p_boundary_2
    portion_in_right_2 = torch.clamp(portion_in_right_2, 0, 1)
    s_all_portion_in_right_2 = (s_cumsum > (1 - alpha)).type_as(p_logits)
    s_all_portion_in_right_2.scatter_(-1, boundary_2, portion_in_right_2)

    s_all_portion_in_right = s_all_portion_in_right_2 / 2 + s_all_portion_in_right_1 / 2
    s_shift_logits = torch.log(s_all_portion_in_right)
    shift_logits = torch.gather(s_shift_logits, -1, unshuffle)

    return p_logits + shift_logits


def _get_rng_seed(context_code: any) -> int:
    """Get the random seed from the given context code and private key."""
    m = hashlib.sha256()
    m.update(context_code)
    m.update(hash_key)

    full_hash = m.digest()
    seed = int.from_bytes(full_hash, "big") % (2**32 - 1)
    return seed


def _extract_context_code(context: torch.Tensor, prefix_length: int) -> torch.Tensor:
    """Extract context code from the given context tensor."""
    if prefix_length == 0:
        return context
    else:
        return context[:, -prefix_length:]

def get_seed_for_cipher(input_ids: torch.Tensor, prefix_length: int):
    """Get the seeds for the cipher using vectorized tensor operations."""
    # Extract the context codes using tensor slicing
    context_codes = _extract_context_code(input_ids, prefix_length)
    
    # Concatenate the tensor slices for hashing
    batch_size = context_codes.size(0)
    seeds = []
    
    # Iterate over batch and compute seeds in parallel
    for i in range(batch_size):
        context_code = context_codes[i].detach().cpu().numpy().tobytes()  # Convert to bytes for hashlib
        seed = _get_rng_seed(context_code)
        seeds.append(seed)
        
    # assert all seed in seeds are equal
    # assert all(seed == seeds[0] for seed in seeds)
    
    return seeds


def _apply_watermark(input_ids: torch.LongTensor, scores: torch.FloatTensor, alpha:float, prefix_length: int) -> torch.FloatTensor:
    """Apply watermark to the scores."""
    seeds = get_seed_for_cipher(input_ids, prefix_length=prefix_length)

    rng = [torch.Generator(device=scores.device).manual_seed(seed) for seed in seeds]
    # mask = torch.tensor(mask, device=scores.device)
    shuffle = from_random(rng, scores.size(1))

    reweighted_scores = reweight_logits(shuffle, scores, alpha)

    return reweighted_scores


def _sampling(logits, top_k=None, top_p=None, temperature=1.0):
    """Sampling function"""
    assert temperature > 0, "temperature must be positive"

    _logits = logits / temperature

    # Apply top-k sampling
    if top_k > 0:
        top_k = min(
            top_k, _logits.size(-1)
        )  # Ensure top_k is not greater than the vocabulary size
        indices_to_remove = _logits < torch.topk(_logits, top_k)[0][..., -1, None]
        _logits[indices_to_remove] = float("-inf")

    # Apply top-p sampling
    if top_p > 0 and top_p < 1:
        sorted_logits, sorted_indices = torch.sort(_logits, descending=True)
        cumulative_probs = torch.cumsum(F.softmax(sorted_logits, dim=-1), dim=-1)
        sorted_indices_to_remove = cumulative_probs > top_p
        if sorted_indices_to_remove[..., 1:].any():
            sorted_indices_to_remove[..., 1:] = sorted_indices_to_remove[
                ..., :-1
            ].clone()
            sorted_indices_to_remove[..., 0] = 0

        # scatter sorted tensors to original indexing
        indices_to_remove = sorted_indices_to_remove.scatter(
            1, sorted_indices, sorted_indices_to_remove
        )
        _logits[indices_to_remove] = float("-inf")

    # Get probability distribution
    probs = F.softmax(_logits, dim=-1)
    print("shape of probs: ", probs.shape)
    sampled_indices = torch.multinomial(probs, num_samples=1)
    return sampled_indices


def dip_sampling(
    input_ids,
    scores,
    alpha,
    prefix_length,
    temperature,
    top_k,
    top_p,
):
    if input_ids.shape[-1] < prefix_length:
        return scores

    reweighted_scores = _apply_watermark(input_ids, scores, alpha, prefix_length)

    # print("Sampling...")
    sampled_indices = _sampling(
        reweighted_scores, top_k=top_k, top_p=top_p, temperature=temperature
    )

    return sampled_indices


def get_logits(ctx: str, logits: dict, tokenizer: AutoTokenizer):
    cur_logits = logits
    pre_str = "Example12:"
    pre_tokens = tokenizer.encode(pre_str, add_special_tokens=False)
    pre_ctx_tokens = tokenizer.encode(pre_str + ctx, add_special_tokens=False)
    ctx_token = pre_ctx_tokens[len(pre_tokens) :]

    for id in ctx_token:
        cur_logits = cur_logits[str(id)]

    assert len(cur_logits.keys()) == 1
    return torch.tensor(cur_logits["logits"], device=device)


def sample_batch_wm(
    logits: torch.Tensor,
    batch_size,
    input_ids,
    vocab_size,
    temperature,
    top_k,
    top_p,
    alpha,
    prefix_length,
):
    vocab_size = logits["logits"].shape[-1]
    cur_logits_batch = [logits] * batch_size
    active = torch.ones(batch_size, dtype=torch.bool, device=device)
    token_ids = torch.full((batch_size,), -1, dtype=torch.long, device=device)
    context_ids = [None for _ in range(batch_size)]
    
    while active.any():
        active_indices = torch.nonzero(active).squeeze(1)
        
        logits_batch = torch.stack(
            [
                (cur_logits_batch[i]["logits"]).squeeze(0).to(device)
                for i in active_indices
            ]
        )
        tokens = dip_sampling(
            input_ids=input_ids[active_indices],
            scores=logits_batch,
            alpha=alpha,
            prefix_length=prefix_length,
            temperature=temperature,
            top_k=top_k,
            top_p=top_p,
        ).squeeze(1)
        
        # print(tokens)
        # Update context_ids and cur_logits_batch
        token_idx = 0
        for i in range(batch_size):
            if not active[i]:
                continue

            token_id = tokens[token_idx].item()
            token_idx += 1

            if context_ids[i] is None:
                context_ids[i] = [token_id]
            else:
                context_ids[i].append(token_id)
                
            # Dynamic update input_ids, append new sampled token, and remove the first token
            input_ids[i] = torch.cat(
                [input_ids[i][1:], torch.tensor([token_id], device=input_ids.device)]
            )

            # token_id_str = str(token_id)
            if token_id in cur_logits_batch[i]:
                cur_logits_batch[i] = cur_logits_batch[i][token_id]
            else:
                if len(cur_logits_batch[i]) == 1 and "logits" in cur_logits_batch[i]:
                    # Current only has 'logits' item, which means this is the last token of a legal prefix
                    token_ids[i] = token_id
                    active[i] = False
                else:
                    # Not in legal sampling list, mark as completed and illegal
                    token_ids[i] = -1
                    active[i] = False
        # print("active_indices: ", torch.nonzero(active).squeeze(1))
        
    return token_ids.cpu().numpy(), context_ids


def run(
    combinations,
    model_name,
    samples,
    alpha,
    batch_size,
    prefix_length,
    sample_iter,
    device,
):
    num_iters = samples

    tokenizer = AutoTokenizer.from_pretrained(model_path)
    if model_name in ["opt27b", "opt13b"]: # ATTENTION: vocab size for opt-13b and opt-27b is different
        vocab_size = 50272
    else:
        vocab_size = tokenizer.vocab_size

    print("Loading remote logits...")
    with open(f"../../data/logits/fixkey-p1-logits-{model_name}.pickle", "rb") as f:
        remote_logits_1 = pickle.load(f)

    with open(f"../../data/logits/fixkey-p2-logits-{model_name}.pickle", "rb") as f:
        remote_logits_2 = pickle.load(f)

    print("Converting remote logits to tensors...")
    remote_logits = [remote_logits_1, remote_logits_2]

    def convert_logits_to_tensor(d):
        for key, value in d.items():
            if isinstance(value, dict):
                convert_logits_to_tensor(value)
            elif key == "logits":
                d[key].to(device)

    convert_logits_to_tensor(remote_logits[0])
    convert_logits_to_tensor(remote_logits[1])

    print("Convert done. Starting sampling...")

    with torch.no_grad():
        for idx in range(2):
            print(f"Processing prompt {idx}...")

            for combination in combinations:
                temperature = combination["temperature"]
                top_p = combination["topp"]
                top_k = combination["topk"]

                print(
                    f"Running combination: temperature={temperature}, topp={top_p}, topk={top_k}"
                )

                json_file_name = f"{json_file_paths[idx]}-{model_name}-temp-{temperature}-topp-{top_p}-topk-{top_k}-alpha-{alpha}-prefixlen-{prefix_length}-{samples}-prob2-iter-{sample_iter}.json"
                # if already exists, skip
                if os.path.exists(json_file_name):
                    print(f"File already exists: {json_file_name}")
                    continue

                mapping_S_wm = {}
                mapping_S_uw = {}

                input_ids = tokenizer.encode(prompts[idx], return_tensors="pt").to(device)
                input_ids = input_ids.repeat(batch_size, 1)
                for iter in range(num_iters // batch_size):
                    print(f"Iter: {iter + 1}")

                    wm_tokens, wm_contexts = sample_batch_wm(
                        logits=remote_logits[idx],
                        batch_size=batch_size,
                        input_ids=input_ids,
                        vocab_size=vocab_size,
                        temperature=temperature,
                        top_k=top_k,
                        top_p=top_p,
                        alpha=alpha,
                        prefix_length=prefix_length,
                    )

                    wm_valid_indices = np.where(wm_tokens != -1)[0]
                    wm_valid_contexts = [wm_contexts[i] for i in wm_valid_indices]
                    valid_wm_tokens = wm_tokens[wm_valid_indices]

                    for i, ctx in enumerate(wm_valid_contexts):
                        context_str = f' {tokenizer.decode(ctx).rsplit("|")[0].strip()}'
                        token = valid_wm_tokens[i]

                        if context_str not in mapping_S_wm:
                            mapping_S_wm[context_str] = {}
                            mapping_S_wm[context_str]["S_wm"] = [0] * vocab_size
                        mapping_S_wm[context_str]["S_wm"][token] += 1

                results = {
                    "watermarked": {str(k): v for k, v in mapping_S_wm.items()},
                    "unwatermarked": {str(k): v for k, v in mapping_S_uw.items()},
                }

                with open(
                    json_file_name,
                    "w",
                ) as json_file:
                    ujson.dump(results, json_file, separators=(",", ":"))

                # Clear CUDA cache to free memory after each combination
                torch.cuda.empty_cache()
                print("Cleared CUDA cache after combination.")


if __name__ == "__main__":
    import argparse

    parser = argparse.ArgumentParser(description="DIP Sampling Batch for Water-Prob-V2")
    parser.add_argument("--model_name", type=str, required=True, help="model_name parameter")
    parser.add_argument("--samples", type=int, required=True, help="samples parameter")
    parser.add_argument("--alpha", type=float, required=True, help="alpha parameter")
    parser.add_argument(
        "--prefix_length", type=int, required=True, help="prefix_length parameter"
    )
    parser.add_argument("--batch_size", type=int, required=True, help="batch_size parameter")
    parser.add_argument("--device", type=int, required=True, help="device parameter")
    parser.add_argument("--option", default="experiment", type=str, required=False, help="option parameter")
    parser.add_argument("--model_path", type=str, required=True, help="model_path parameter")
    parser.add_argument(
        "--sample_iter", type=int, required=False, help="sample_iter parameter"
    )

    args = parser.parse_args()

    model_path = args.model_path

    if args.option == "all":
        combinations = [
            {"temperature": 1.0, "topp": 1.0, "topk": 0},
            {"temperature": 0.8, "topp": 1.0, "topk": 0},
            {"temperature": 0.7, "topp": 1.0, "topk": 0},
            {"temperature": 0.6, "topp": 1.0, "topk": 0},
            {"temperature": 1.2, "topp": 1.0, "topk": 0},
            {"temperature": 1.4, "topp": 1.0, "topk": 0},
            {"temperature": 1.6, "topp": 1.0, "topk": 0},
            {"temperature": 1.0, "topp": 0.7, "topk": 0},
            {"temperature": 1.0, "topp": 0.8, "topk": 0},
            {"temperature": 1.0, "topp": 0.9, "topk": 0},
            {"temperature": 1.0, "topp": 1.0, "topk": 100},
            {"temperature": 1.0, "topp": 1.0, "topk": 200},
            {"temperature": 1.0, "topp": 1.0, "topk": 500},
            {"temperature": 0.8, "topp": 1.0, "topk": 50},
            {"temperature": 0.7, "topp": 1.0, "topk": 50},
            {"temperature": 0.6, "topp": 1.0, "topk": 50},
            {"temperature": 0.8, "topp": 0.7, "topk": 0},
            {"temperature": 0.7, "topp": 0.7, "topk": 0},
            {"temperature": 0.6, "topp": 0.7, "topk": 0},
            {"temperature": 0.6, "topp": 0.7, "topk": 50},
            {"temperature": 1.2, "topp": 0.7, "topk": 50},
            {"temperature": 0.8, "topp": 0.7, "topk": 50},
        ]
    elif args.option == "temp":
        combinations = [
            {"temperature": 1.5, "topp": 1.0, "topk": 0},
            {"temperature": 1.4, "topp": 1.0, "topk": 0},
            {"temperature": 1.3, "topp": 1.0, "topk": 0},
            {"temperature": 1.2, "topp": 1.0, "topk": 0},
            {"temperature": 1.1, "topp": 1.0, "topk": 0},
            {"temperature": 1.0, "topp": 1.0, "topk": 0},
            {"temperature": 0.9, "topp": 1.0, "topk": 0},
            {"temperature": 0.8, "topp": 1.0, "topk": 0},
            {"temperature": 0.7, "topp": 1.0, "topk": 0},
            {"temperature": 0.6, "topp": 1.0, "topk": 0},
            {"temperature": 0.5, "topp": 1.0, "topk": 0},
            {"temperature": 0.4, "topp": 1.0, "topk": 0},
            {"temperature": 0.3, "topp": 1.0, "topk": 0},
            {"temperature": 0.2, "topp": 1.0, "topk": 0},
            {"temperature": 0.1, "topp": 1.0, "topk": 0},
        ]
    elif args.option == "joint":
        combinations = [
            {"temperature": 0.8, "topp": 1.0, "topk": 50},
            {"temperature": 0.7, "topp": 1.0, "topk": 50},
            {"temperature": 0.6, "topp": 1.0, "topk": 50},
            {"temperature": 0.8, "topp": 0.7, "topk": 0},
            {"temperature": 0.7, "topp": 0.7, "topk": 0},
            {"temperature": 0.6, "topp": 0.7, "topk": 0},
            {"temperature": 0.6, "topp": 0.7, "topk": 50},
            {"temperature": 1.2, "topp": 0.7, "topk": 50},
            {"temperature": 0.8, "topp": 0.7, "topk": 50},
        ]
    elif args.option == "temp-most":
        combinations = [
            {"temperature": 1.2, "topp": 1.0, "topk": 0},
            {"temperature": 1.1, "topp": 1.0, "topk": 0},
            {"temperature": 1.0, "topp": 1.0, "topk": 0},
            {"temperature": 0.9, "topp": 1.0, "topk": 0},
            {"temperature": 0.8, "topp": 1.0, "topk": 0},
        ]
    elif args.option == "experiment":
        combinations = [
            {"temperature": 1.0, "topp": 1.0, "topk": 0},
        ]
    print("Device: ", args.device)
    os.environ["CUDA_VISIBLE_DEVICES"] = f"{args.device}"
    device = torch.device("cuda" if torch.cuda.is_available() else "cpu")

    # DIP Config
    hash_seed = 42
    random.seed(hash_seed)
    hash_key = random.getrandbits(1024).to_bytes(128, "big")

    run(
        combinations=combinations,
        model_name=args.model_name,
        samples=args.samples,
        alpha=args.alpha,
        batch_size=args.batch_size,
        prefix_length=args.prefix_length,
        sample_iter=args.sample_iter,
        device=device,
    )
    sys.exit(0)
