# ITS Sampling Batch for Water-Prob-V1

import torch
import ujson
import os
import numpy as np
from torch.nn import functional as F
from typing import Union
from transformers import AutoTokenizer
import sys
from mersenne import MersenneRNG
import pickle
from itertools import product

json_file_path_1 = "../../data/results/prob1/its-p1"
json_file_path_2 = "../../data/results/prob1/its-p2"
prompt_file_path_1 = "../../data/prompts/ngram-p1.txt"
prompt_file_path_2 = "../../data/prompts/ngram-p2.txt"

json_file_paths = [json_file_path_1, json_file_path_2]

with open(prompt_file_path_1, "r") as f:
    prompt1 = f.readlines()
    prompt1 = "".join(prompt1)

with open(prompt_file_path_2, "r") as f:
    prompt2 = f.readlines()
    prompt2 = "".join(prompt2)

prompts = [prompt1, prompt2]

# Constants
letters = [f" {chr(i)}" for i in range(65, 91)]
numbers_en = [
    " zero",
    " one",
    " two",
    " three",
    " four",
    " five",
    " six",
    " seven",
    " eight",
    " nine",
]
animal_choice = [" cat", " dog", " tiger", " lion"]
combinations_main = ["".join(comb) for comb in product(letters, numbers_en, animal_choice)]

def transform_sampling(probs,pi,xi):
    """Sample token using inverse transform method."""
    cdf = torch.cumsum(torch.gather(probs, 1, pi), 1)
    return torch.gather(pi, 1, torch.searchsorted(cdf, xi))
    
def its_sampling(
    logits: torch.Tensor,
    u: torch.Tensor,
    pi: torch.Tensor,
    temperature: float,
    top_k: int,
    top_p: float
):
    if isinstance(logits, torch.Tensor):
        _logits = logits.clone()
    else:
        _logits = torch.tensor(logits, device=u.device)

    _logits /= temperature

    # Apply top-k sampling
    if top_k > 0:
        top_k = min(
            top_k, _logits.size(-1)
        )  # Ensure top_k is not greater than the vocabulary size
        indices_to_remove = _logits < torch.topk(_logits, top_k)[0][..., -1, None]
        _logits[indices_to_remove] = float("-inf")

    # Apply top-p sampling
    if top_p > 0 and top_p < 1:
        sorted_logits, sorted_indices = torch.sort(_logits, descending=True)
        cumulative_probs = torch.cumsum(F.softmax(sorted_logits, dim=-1), dim=-1)
        sorted_indices_to_remove = cumulative_probs > top_p
        if sorted_indices_to_remove[..., 1:].any():
            sorted_indices_to_remove[..., 1:] = sorted_indices_to_remove[
                ..., :-1
            ].clone()
            sorted_indices_to_remove[..., 0] = 0

        # scatter sorted tensors to original indexing
        indices_to_remove = sorted_indices_to_remove.scatter(
            1, sorted_indices, sorted_indices_to_remove
        )
        _logits[indices_to_remove] = float("-inf")

    # Get probability distribution
    probs = F.softmax(_logits, dim=-1)

    # Inverse transform sampling
    sampled_indices = transform_sampling(probs, pi, u)

    return sampled_indices


def sample_batch_wm(
    logits, 
    batch_size, 
    temperature, 
    top_k, 
    top_p, 
    keylen, 
    xi, 
    pi,
    device
):
    shifts = torch.randint(keylen, (batch_size,), device=device)
    logits_batch = logits.squeeze(0).repeat(batch_size, 1)
    xi_batch = torch.stack(
        [xi[(shifts[i]) % keylen, :].to(device) for i in range(batch_size)]
    )
    pi_batch = pi.repeat(batch_size, 1)
    tokens = its_sampling(
        logits=logits_batch,
        u=xi_batch,
        pi=pi_batch,
        temperature=temperature,
        top_k=top_k,
        top_p=top_p,
    ).squeeze(1)

    return tokens.cpu().numpy()

def get_logits(ctx: str, logits: dict, tokenizer: AutoTokenizer):
    cur_logits = logits
    pre_str = "Example12:"
    pre_tokens = tokenizer.encode(pre_str, add_special_tokens=False)
    pre_ctx_tokens = tokenizer.encode(pre_str + ctx, add_special_tokens=False)
    ctx_token = pre_ctx_tokens[len(pre_tokens) :]

    for id in ctx_token:
        cur_logits = cur_logits[id]

    assert len(cur_logits.keys()) == 1
    return torch.tensor(cur_logits["logits"], device=device)


def run(combinations, model_name, samples, fill_parts, keylen, sample_iter, device):
    num_iters = samples 
    assert num_iters % len(fill_parts) == 0
    batch_size = int(samples // len(fill_parts)) # Each fill_part is a batch

    tokenizer = AutoTokenizer.from_pretrained(model_path)
    if model_name in ["opt27b", "opt13b"]:
        vocab_size = 50272
    else:
        vocab_size = tokenizer.vocab_size
        
    print("Loading remote logits...")
    with open(f"../../data/logits/ngram-p1-logits-{model_name}.pickle", "rb") as f:
        remote_logits_1 = pickle.load(f)

    with open(f"../../data/logits/ngram-p2-logits-{model_name}.pickle", "rb") as f:
        remote_logits_2 = pickle.load(f)

    print("Converting remote logits to tensors...")
    remote_logits = [remote_logits_1, remote_logits_2]

    def convert_logits_to_tensor(d):
        for key, value in d.items():
            if isinstance(value, dict):
                convert_logits_to_tensor(value)
            elif key == "logits":
                d[key].to(device)

    convert_logits_to_tensor(remote_logits[0])
    convert_logits_to_tensor(remote_logits[1])

    print("Convert done. Starting sampling...")

    with torch.no_grad():
        ## ITS-Edit's permutation pi & key xi
        pi = torch.randperm(vocab_size, generator=pi_rng, device=device)
        xi = torch.rand((keylen, 1), generator=pi_rng, device=device)

        for idx in range(2):

            print(f"Processing prompt {idx}...")

            for combination in combinations:
                temperature = combination["temperature"]
                top_p = combination["topp"]
                top_k = combination["topk"]

                print(
                    f"Running combination: temperature={temperature}, topp={top_p}, topk={top_k}"
                )
                
                json_file_name = f"{json_file_paths[idx]}-{model_name}-temp-{temperature}-{keylen}-topp-{top_p}-topk-{top_k}-{samples}-prob1-{len(fill_parts)}-iter-{sample_iter}.json"
                
                # if already exists, skip
                if os.path.exists(json_file_name):
                    print(f"File already exists, skipping...")
                    continue

                mapping_S_wm = {}
                mapping_S_uw = {}
                for fill_part in fill_parts:
                    print(f"For prompt {idx}, fill part: {fill_part}")
                    print(f"Processing fill part: {fill_part}")
                    input_ids = tokenizer.encode(prompts[idx] + fill_part, return_tensors="pt").to(device)
                    input_ids = input_ids.repeat(batch_size, 1)
                    assert num_iters % batch_size == 0
                    assert num_iters // batch_size % len(fill_parts) == 0

                    for iter in range(num_iters // batch_size // len(fill_parts)):
                        print(f"Iter: {iter + 1}/{num_iters // batch_size // len(fill_parts)}")
                        
                        logits = get_logits(fill_part, remote_logits[idx], tokenizer)
                        wm_tokens = sample_batch_wm(
                            logits=logits,
                            batch_size=batch_size,
                            temperature=temperature,
                            top_k=top_k,
                            top_p=top_p,
                            keylen=keylen,
                            xi=xi,
                            pi=pi,
                            device=device,
                        )

                        if fill_part not in mapping_S_wm:
                            mapping_S_wm[fill_part] = {}
                            mapping_S_wm[fill_part]["S_wm"] = [0] * vocab_size

                        for token in wm_tokens:
                            mapping_S_wm[fill_part]["S_wm"][token] += 1

                results = {
                    "watermarked": {str(k): v for k, v in mapping_S_wm.items()},
                    "unwatermarked": {str(k): v for k, v in mapping_S_uw.items()},
                }

                with open(
                    json_file_name,
                    "w",
                ) as json_file:
                    print("Writing to JSON file...")
                    ujson.dump(results, json_file, separators=(",", ":"))
                    print("Done writing to JSON file.")

                # Clear CUDA cache to free memory after each combination
                torch.cuda.empty_cache()
                print("Cleared CUDA cache after combination.")


if __name__ == "__main__":
    import argparse

    parser = argparse.ArgumentParser(description="ITS Sampling Batch for Water-Prob-V1")
    parser.add_argument("--model_name", type=str, required=True, help="model_name parameter")
    parser.add_argument("--samples", type=int, required=True, help="samples parameter")
    parser.add_argument("--keylen", type=int, required=True, help="keylen parameter")
    parser.add_argument("--device", type=str, required=True, help="device parameter")
    parser.add_argument("--option", default="experiment", type=str, required=False, help="option parameter")
    parser.add_argument("--fill_length", default=50, type=int, required=False, help="fill_length parameter")
    parser.add_argument("--model_path", type=str, required=True, help="model_path parameter")
    parser.add_argument("--sample_iter", type=int, required=True, help="sample_iter parameter")
    
    args = parser.parse_args()
        
    model_path = args.model_path

    if args.option == "all":
        combinations = [
            {"temperature": 1.0, "topp": 1.0, "topk": 0},
            {"temperature": 0.8, "topp": 1.0, "topk": 0},
            {"temperature": 0.7, "topp": 1.0, "topk": 0},
            {"temperature": 0.6, "topp": 1.0, "topk": 0},
            {"temperature": 1.2, "topp": 1.0, "topk": 0},
            {"temperature": 1.4, "topp": 1.0, "topk": 0},
            {"temperature": 1.6, "topp": 1.0, "topk": 0},
            {"temperature": 1.0, "topp": 0.7, "topk": 0},
            {"temperature": 1.0, "topp": 0.8, "topk": 0},
            {"temperature": 1.0, "topp": 0.9, "topk": 0},
            {"temperature": 1.0, "topp": 1.0, "topk": 100},
            {"temperature": 1.0, "topp": 1.0, "topk": 200},
            {"temperature": 1.0, "topp": 1.0, "topk": 500},
            {"temperature": 0.8, "topp": 1.0, "topk": 50},
            {"temperature": 0.7, "topp": 1.0, "topk": 50},
            {"temperature": 0.6, "topp": 1.0, "topk": 50},
            {"temperature": 0.8, "topp": 0.7, "topk": 0},
            {"temperature": 0.7, "topp": 0.7, "topk": 0},
            {"temperature": 0.6, "topp": 0.7, "topk": 0},
            {"temperature": 0.6, "topp": 0.7, "topk": 50},
            {"temperature": 1.2, "topp": 0.7, "topk": 50},
            {"temperature": 0.8, "topp": 0.7, "topk": 50},
        ]
    elif args.option == "temp":
        combinations = [
            {"temperature": 1.5, "topp": 1.0, "topk": 0},
            {"temperature": 1.4, "topp": 1.0, "topk": 0},
            {"temperature": 1.3, "topp": 1.0, "topk": 0},
            {"temperature": 1.2, "topp": 1.0, "topk": 0},
            {"temperature": 1.1, "topp": 1.0, "topk": 0},
            {"temperature": 1.0, "topp": 1.0, "topk": 0},
            {"temperature": 0.9, "topp": 1.0, "topk": 0},
            {"temperature": 0.8, "topp": 1.0, "topk": 0},
            {"temperature": 0.7, "topp": 1.0, "topk": 0},
            {"temperature": 0.6, "topp": 1.0, "topk": 0},
            {"temperature": 0.5, "topp": 1.0, "topk": 0},
            {"temperature": 0.4, "topp": 1.0, "topk": 0},
            {"temperature": 0.3, "topp": 1.0, "topk": 0},
            {"temperature": 0.2, "topp": 1.0, "topk": 0},
            {"temperature": 0.1, "topp": 1.0, "topk": 0},
        ]
    elif args.option == "joint":
        combinations = [
            {"temperature": 0.8, "topp": 1.0, "topk": 50},
            {"temperature": 0.7, "topp": 1.0, "topk": 50},
            {"temperature": 0.6, "topp": 1.0, "topk": 50},
            {"temperature": 0.8, "topp": 0.7, "topk": 0},
            {"temperature": 0.7, "topp": 0.7, "topk": 0},
            {"temperature": 0.6, "topp": 0.7, "topk": 0},
            {"temperature": 0.6, "topp": 0.7, "topk": 50},
            {"temperature": 1.2, "topp": 0.7, "topk": 50},
            {"temperature": 0.8, "topp": 0.7, "topk": 50},
        ]
    elif args.option == "temp-most":
        combinations = [
            {"temperature": 1.2, "topp": 1.0, "topk": 0},
            {"temperature": 1.1, "topp": 1.0, "topk": 0},
            {"temperature": 1.0, "topp": 1.0, "topk": 0},
            {"temperature": 0.9, "topp": 1.0, "topk": 0},
            {"temperature": 0.8, "topp": 1.0, "topk": 0},
        ]
    elif args.option == "experiment":
        combinations = [
            {"temperature": 1.0, "topp": 1.0, "topk": 0},
        ]

    print("Device: ", args.device)
    os.environ["CUDA_VISIBLE_DEVICES"] = f"{args.device}"
    device = torch.device("cuda" if torch.cuda.is_available() else "cpu")

    ## ITS-Edit Config
    key = 42 # key for ITS-Edit's permutation pi
    pi_rng = torch.Generator(device=device).manual_seed(key)
    
    import random
    random.seed(64)
    fill_parts = random.sample(combinations_main, args.fill_length)
    print("Fill parts: ")
    print(fill_parts)
    
    ## Run ITS-Edit Sampling
    run(
        combinations=combinations,
        model_name=args.model_name,
        samples=args.samples,
        fill_parts=fill_parts,
        keylen=args.keylen,
        sample_iter=args.sample_iter,
        device=device,
    )
    sys.exit(0)
