# ITS Sampling Batch for Water-Prob-V2

import torch
import ujson
import os
import numpy as np
from torch.nn import functional as F
from typing import Union
from transformers import AutoTokenizer
import sys
from mersenne import MersenneRNG
import pickle

json_file_path_1 = "../../data/results/prob2/its-p1"
json_file_path_2 = "../../data/results/prob2/its-p2"
prompt_file_path_1 = "../../data/prompts/fixkey-p1.txt"
prompt_file_path_2 = "../../data/prompts/fixkey-p2.txt"

json_file_paths = [json_file_path_1, json_file_path_2]

with open(prompt_file_path_1, "r") as f:
    prompt1 = f.readlines()
    prompt1 = "".join(prompt1)

with open(prompt_file_path_2, "r") as f:
    prompt2 = f.readlines()
    prompt2 = "".join(prompt2)

prompts = [prompt1, prompt2]

# Constants
letters = [f" {chr(i)}" for i in range(65, 91)]
numbers_en = [
    " zero",
    " one",
    " two",
    " three",
    " four",
    " five",
    " six",
    " seven",
    " eight",
    " nine",
]
animal_choice = [" cat", " dog", " tiger", " lion"]

def transform_sampling(probs,pi,xi):
    """Sample token using inverse transform method."""
    cdf = torch.cumsum(torch.gather(probs, 1, pi), 1)
    return torch.gather(pi, 1, torch.searchsorted(cdf, xi))
    
def its_sampling(
    logits: torch.Tensor,
    u: torch.Tensor,
    pi: torch.Tensor,
    temperature: float,
    top_k: int,
    top_p: float
):
    if isinstance(logits, torch.Tensor):
        _logits = logits.clone()
    else:
        _logits = torch.tensor(logits, device=u.device)

    _logits /= temperature

    # Apply top-k sampling
    if top_k > 0:
        top_k = min(
            top_k, _logits.size(-1)
        )  # Ensure top_k is not greater than the vocabulary size
        indices_to_remove = _logits < torch.topk(_logits, top_k)[0][..., -1, None]
        _logits[indices_to_remove] = float("-inf")

    # Apply top-p sampling
    if top_p > 0 and top_p < 1:
        sorted_logits, sorted_indices = torch.sort(_logits, descending=True)
        cumulative_probs = torch.cumsum(F.softmax(sorted_logits, dim=-1), dim=-1)
        sorted_indices_to_remove = cumulative_probs > top_p
        if sorted_indices_to_remove[..., 1:].any():
            sorted_indices_to_remove[..., 1:] = sorted_indices_to_remove[
                ..., :-1
            ].clone()
            sorted_indices_to_remove[..., 0] = 0

        # scatter sorted tensors to original indexing
        indices_to_remove = sorted_indices_to_remove.scatter(
            1, sorted_indices, sorted_indices_to_remove
        )
        _logits[indices_to_remove] = float("-inf")

    # Get probability distribution
    probs = F.softmax(_logits, dim=-1)

    # Inverse transform sampling
    sampled_indices = transform_sampling(probs, pi, u)

    return sampled_indices


def sample_batch_wm(
    logits, 
    batch_size, 
    temperature, 
    top_k, 
    top_p, 
    keylen, 
    xi, 
    pi,
    device
):
    cur_logits_batch = [logits] * batch_size
    # random shift(start index)
    shifts = torch.randint(keylen, (batch_size,), device=device)
    cnts = torch.zeros(batch_size, dtype=torch.long, device=device)
    active = torch.ones(batch_size, dtype=torch.bool, device=device)
    token_ids = torch.full((batch_size,), -1, dtype=torch.long, device=device)
    context_ids = [None for _ in range(batch_size)]
    origin_pi_batch = pi.repeat(batch_size, 1)
    
    while active.any():
        active_indices = torch.nonzero(active).squeeze(1)
        logits_batch = torch.stack(
            [
                (cur_logits_batch[i]["logits"]).squeeze(0).to(device)
                for i in active_indices
            ]
        )
        xi_batch = torch.stack(
            [xi[(shifts[i] + cnts[i]) % keylen, :].to(device) for i in active_indices]
        )
        pi_batch = origin_pi_batch[active_indices]
        tokens = its_sampling(
            logits=logits_batch,
            u=xi_batch,
            pi=pi_batch,
            temperature=temperature,
            top_k=top_k,
            top_p=top_p,
        ).squeeze(1)

        # Update context_ids and cur_logits_batch
        token_idx = 0
        for i in range(batch_size):
            if not active[i]:
                continue

            token_id = tokens[token_idx].item()
            token_idx += 1

            if context_ids[i] is None:
                context_ids[i] = [token_id]
            else:
                context_ids[i].append(token_id)

            # token_id_str = str(token_id)
            if token_id in cur_logits_batch[i]:
                cur_logits_batch[i] = cur_logits_batch[i][token_id]
                cnts[i] += 1
            else:
                if len(cur_logits_batch[i]) == 1 and "logits" in cur_logits_batch[i]:
                    # Current only has 'logits' item, indicating this is the last token of a valid prefix
                    token_ids[i] = token_id
                    active[i] = False
                else:
                    # Not in valid sampling list, marked as completed and invalid
                    token_ids[i] = -1
                    active[i] = False

    return token_ids.cpu().numpy(), context_ids, shifts.cpu().numpy()


def run(combinations, model_name, samples, batch_size, keylen, sample_iter, device):
    num_iters = samples 
    assert num_iters % batch_size == 0

    tokenizer = AutoTokenizer.from_pretrained(model_path)
    if model_name in ["opt27b", "opt13b"]:
        vocab_size = 50272
    else:
        vocab_size = tokenizer.vocab_size
        
    print("Loading remote logits...")
    with open(f"../../data/logits/fixkey-p1-logits-{model_name}.pickle", "rb") as f:
        remote_logits_1 = pickle.load(f)

    with open(f"../../data/logits/fixkey-p2-logits-{model_name}.pickle", "rb") as f:
        remote_logits_2 = pickle.load(f)

    print("Converting remote logits to tensors...")
    remote_logits = [remote_logits_1, remote_logits_2]

    def convert_logits_to_tensor(d):
        for key, value in d.items():
            if isinstance(value, dict):
                convert_logits_to_tensor(value)
            elif key == "logits":
                d[key].to(device)

    convert_logits_to_tensor(remote_logits[0])
    convert_logits_to_tensor(remote_logits[1])

    print("Convert done. Starting sampling...")

    with torch.no_grad():
        ## ITS-Edit's permutation pi & key xi
        pi = torch.randperm(vocab_size, generator=pi_rng, device=device)
        xi = torch.rand((keylen, 1), generator=pi_rng, device=device)

        for idx in range(2):

            print(f"Processing prompt {idx}...")

            for combination in combinations:
                temperature = combination["temperature"]
                top_p = combination["topp"]
                top_k = combination["topk"]

                print(
                    f"Running combination: temperature={temperature}, topp={top_p}, topk={top_k}"
                )
                
                json_file_name = f"{json_file_paths[idx]}-{model_name}-temp-{temperature}-{keylen}-topp-{top_p}-topk-{top_k}-{samples}-iter-{sample_iter}.json"
                
                # if already exists, skip
                if os.path.exists(json_file_name):
                    print(f"File already exists, skipping...")
                    continue

                mapping_S_wm = {}
                mapping_S_uw = {}

                for iter in range(num_iters // batch_size):
                    print(f"Iter: {iter + 1}/{num_iters // batch_size}")
                    wm_tokens, wm_contexts, shifts = sample_batch_wm(
                        logits=remote_logits[idx],
                        batch_size=batch_size,
                        temperature=temperature,
                        top_k=top_k,
                        top_p=top_p,
                        keylen=keylen,
                        xi=xi,
                        pi=pi,
                        device=device,
                    )

                    wm_valid_indices = np.where(wm_tokens != -1)[0]
                    wm_valid_contexts = [wm_contexts[i] for i in wm_valid_indices]
                    valid_wm_tokens = wm_tokens[wm_valid_indices]
                    valid_shifts = shifts[wm_valid_indices]

                    for i, ctx in enumerate(wm_valid_contexts):
                        context_str = f' {tokenizer.decode(ctx).rsplit("|")[0].strip()}'
                        token = valid_wm_tokens[i]
                        shift = valid_shifts[i]

                        if context_str not in mapping_S_wm:
                            mapping_S_wm[context_str] = {}
                            mapping_S_wm[context_str]["S_wm"] = [0] * vocab_size
                            mapping_S_wm[context_str]["start_index"] = [0] * keylen
                        mapping_S_wm[context_str]["S_wm"][token] += 1
                        mapping_S_wm[context_str]["start_index"][shift] += 1

                results = {
                    "watermarked": {str(k): v for k, v in mapping_S_wm.items()},
                    "unwatermarked": {str(k): v for k, v in mapping_S_uw.items()},
                }

                with open(
                    json_file_name,
                    "w",
                ) as json_file:
                    print("Writing to JSON file...")
                    ujson.dump(results, json_file, separators=(",", ":"))
                    print("Done writing to JSON file.")

                # Clear CUDA cache to free memory after each combination
                torch.cuda.empty_cache()
                print("Cleared CUDA cache after combination.")


if __name__ == "__main__":
    import argparse

    parser = argparse.ArgumentParser(description="ITS Sampling Batch for Water-Prob-V2")
    parser.add_argument("--model_name", type=str, required=True, help="model_name parameter")
    parser.add_argument("--samples", type=int, required=True, help="samples parameter")
    parser.add_argument("--keylen", type=int, required=True, help="keylen parameter")
    parser.add_argument("--device", type=str, required=True, help="device parameter")
    parser.add_argument("--option", default="experiment", type=str, required=False, help="option parameter")
    parser.add_argument("--batch_size", type=int, required=True, help="batch_size parameter")
    parser.add_argument("--model_path", type=str, required=True, help="model_path parameter")
    parser.add_argument("--sample_iter", type=int, required=True, help="sample_iter parameter")
    
    args = parser.parse_args()
        
    model_path = args.model_path

    if args.option == "all":
        combinations = [
            {"temperature": 1.0, "topp": 1.0, "topk": 0},
            {"temperature": 0.8, "topp": 1.0, "topk": 0},
            {"temperature": 0.7, "topp": 1.0, "topk": 0},
            {"temperature": 0.6, "topp": 1.0, "topk": 0},
            {"temperature": 1.2, "topp": 1.0, "topk": 0},
            {"temperature": 1.4, "topp": 1.0, "topk": 0},
            {"temperature": 1.6, "topp": 1.0, "topk": 0},
            {"temperature": 1.0, "topp": 0.7, "topk": 0},
            {"temperature": 1.0, "topp": 0.8, "topk": 0},
            {"temperature": 1.0, "topp": 0.9, "topk": 0},
            {"temperature": 1.0, "topp": 1.0, "topk": 100},
            {"temperature": 1.0, "topp": 1.0, "topk": 200},
            {"temperature": 1.0, "topp": 1.0, "topk": 500},
            {"temperature": 0.8, "topp": 1.0, "topk": 50},
            {"temperature": 0.7, "topp": 1.0, "topk": 50},
            {"temperature": 0.6, "topp": 1.0, "topk": 50},
            {"temperature": 0.8, "topp": 0.7, "topk": 0},
            {"temperature": 0.7, "topp": 0.7, "topk": 0},
            {"temperature": 0.6, "topp": 0.7, "topk": 0},
            {"temperature": 0.6, "topp": 0.7, "topk": 50},
            {"temperature": 1.2, "topp": 0.7, "topk": 50},
            {"temperature": 0.8, "topp": 0.7, "topk": 50},
        ]
    elif args.option == "temp":
        combinations = [
            {"temperature": 1.5, "topp": 1.0, "topk": 0},
            {"temperature": 1.4, "topp": 1.0, "topk": 0},
            {"temperature": 1.3, "topp": 1.0, "topk": 0},
            {"temperature": 1.2, "topp": 1.0, "topk": 0},
            {"temperature": 1.1, "topp": 1.0, "topk": 0},
            {"temperature": 1.0, "topp": 1.0, "topk": 0},
            {"temperature": 0.9, "topp": 1.0, "topk": 0},
            {"temperature": 0.8, "topp": 1.0, "topk": 0},
            {"temperature": 0.7, "topp": 1.0, "topk": 0},
            {"temperature": 0.6, "topp": 1.0, "topk": 0},
            {"temperature": 0.5, "topp": 1.0, "topk": 0},
            {"temperature": 0.4, "topp": 1.0, "topk": 0},
            {"temperature": 0.3, "topp": 1.0, "topk": 0},
            {"temperature": 0.2, "topp": 1.0, "topk": 0},
            {"temperature": 0.1, "topp": 1.0, "topk": 0},
        ]
    elif args.option == "joint":
        combinations = [
            {"temperature": 0.8, "topp": 1.0, "topk": 50},
            {"temperature": 0.7, "topp": 1.0, "topk": 50},
            {"temperature": 0.6, "topp": 1.0, "topk": 50},
            {"temperature": 0.8, "topp": 0.7, "topk": 0},
            {"temperature": 0.7, "topp": 0.7, "topk": 0},
            {"temperature": 0.6, "topp": 0.7, "topk": 0},
            {"temperature": 0.6, "topp": 0.7, "topk": 50},
            {"temperature": 1.2, "topp": 0.7, "topk": 50},
            {"temperature": 0.8, "topp": 0.7, "topk": 50},
        ]
    elif args.option == "temp-most":
        combinations = [
            {"temperature": 1.2, "topp": 1.0, "topk": 0},
            {"temperature": 1.1, "topp": 1.0, "topk": 0},
            {"temperature": 1.0, "topp": 1.0, "topk": 0},
            {"temperature": 0.9, "topp": 1.0, "topk": 0},
            {"temperature": 0.8, "topp": 1.0, "topk": 0},
        ]
    elif args.option == "experiment":
        combinations = [
            {"temperature": 1.0, "topp": 1.0, "topk": 0},
        ]

    print("Device: ", args.device)
    os.environ["CUDA_VISIBLE_DEVICES"] = f"{args.device}"
    # os.environ['CUDA_LAUNCH_BLOCKING'] = '1'
    device = torch.device("cuda" if torch.cuda.is_available() else "cpu")

    ## ITS-Edit Config
    key = 42 # key for ITS-Edit's permutation pi
    pi_rng = torch.Generator(device=device).manual_seed(key)
    
    ## Run ITS-Edit Sampling
    run(
        combinations=combinations,
        model_name=args.model_name,
        samples=args.samples,
        batch_size=args.batch_size,
        keylen=args.keylen,
        sample_iter=args.sample_iter,
        device=device,
    )
    sys.exit(0)
