# KGW Sampling Batch for Water-Prob-V1

import torch
import ujson
import os
import numpy as np
from torch.nn import functional as F
from typing import Union
from transformers import AutoTokenizer
import sys
from itertools import product
import pickle

# Logits & Prompts
json_file_path_1 = "../../data/results/prob1/kgw-p1"
json_file_path_2 = "../../data/results/prob1/kgw-p2"
prompt_file_path_1 = "../../data/prompts/ngram-p1.txt"
prompt_file_path_2 = "../../data/prompts/ngram-p2.txt"

json_file_paths = [json_file_path_1, json_file_path_2]

with open(prompt_file_path_1, "r") as f:
    prompt1 = f.readlines()
    prompt1 = "".join(prompt1)

with open(prompt_file_path_2, "r") as f:
    prompt2 = f.readlines()
    prompt2 = "".join(prompt2)

prompts = [prompt1, prompt2]

# Constants(Fill parts)
letters = [f" {chr(i)}" for i in range(65, 91)]
numbers_en = [
    " zero",
    " one",
    " two",
    " three",
    " four",
    " five",
    " six",
    " seven",
    " eight",
    " nine",
]
animal_choice = [" cat", " dog", " tiger", " lion"]
combinations_main = [
    "".join(comb) for comb in product(letters, numbers_en, animal_choice)
]


# Hash Strategy
def _f_additive(
    input_ids: torch.LongTensor, prefix_length: int, prf: torch.Tensor, vocab_size
) -> int:
    """Get the previous token additive."""
    batch_size, sequence_length = input_ids.shape

    assert (
        sequence_length >= prefix_length
    ), "Sequence length must be at least as long as prefix_length"
    additive_result = torch.ones(batch_size, device=input_ids.device)

    for i in range(0, prefix_length):
        additive_result += input_ids[:, -1 - i].float()

    return prf[additive_result.long() % vocab_size]


def _f_skip(input_ids, prefix_length, prf, vocab_size):
    """Get the previous token skip."""
    batch_size, seq_length = input_ids.shape

    # Ensure that we can access the token `prefix_length` before the last one
    assert (
        seq_length >= prefix_length
    ), "Sequence length must be at least as long as prefix_length"

    # Extract the token that is `prefix_length` before the last token for each batch
    skip_tokens = input_ids[:, -prefix_length]  # Shape: (batch_size,)

    # Use these tokens to index into the `prf` tensor
    skip_values = prf[skip_tokens]  # Shape: (batch_size,)

    return skip_values


def _f_min(input_ids, prefix_length, prf, vocab_size):
    """Get the previous token min."""
    # Initialize a list to store the min values for each batch
    batch_size, seq_length = input_ids.shape

    # Ensure that we can take prefix_length tokens from the sequence
    assert (
        seq_length >= prefix_length
    ), "Sequence length must be at least as long as prefix_length"

    # Gather the last `prefix_length` tokens for each batch
    # We take the last `prefix_length` tokens from each row
    # input_ids[:, -prefix_length:] will be of shape (batch_size, prefix_length)
    last_tokens = input_ids[:, -prefix_length:]  # (batch_size, prefix_length)

    # Look up the PRF values for these tokens (using the input_ids as indices into `prf`)
    prf_values = prf[last_tokens]  # Shape: (batch_size, prefix_length)

    # Compute the minimum along the second dimension (i.e., across the prefix tokens for each batch)
    min_values = prf_values.min(dim=1).values  # Shape: (batch_size,)

    return min_values


def _f_time(input_ids, prefix_length, prf, vocab_size):
    batch_size, sequence_length = input_ids.shape
    time_result = torch.ones(batch_size, device=input_ids.device)

    for i in range(prefix_length):
        time_result *= input_ids[:, -1 - i].float()

    return prf[time_result.long() % vocab_size]


def _get_greenlist_ids_left(input_ids, gamma, prf, vocab_size, prefix_length):
    """Get greenlist ids for the previous context."""
    hash_results = _f(
        input_ids, prefix_length=prefix_length, prf=prf, vocab_size=vocab_size
    )
    seeds = ((hash_key * hash_results) % vocab_size).to(device)

    greenlist_size = int(vocab_size * gamma)
    rng_cuda = torch.Generator(device=device)

    vocab_permutations = torch.stack(
        [
            torch.randperm(
                vocab_size, device=device, generator=rng_cuda.manual_seed(seed.item())
            )
            for seed in seeds
        ],
        dim=0,
    )

    greenlist_ids = vocab_permutations[:, :greenlist_size]
    return greenlist_ids


def _calc_greenlist_mask(scores, greenlist_token_ids):
    """Calculate the greenlist mask for the given scores and greenlist token ids."""
    batch_size, vocab_size = scores.shape
    green_tokens_mask = torch.zeros(
        batch_size, vocab_size, device=scores.device, dtype=torch.bool
    )
    green_tokens_mask.scatter_(1, greenlist_token_ids, True)
    return green_tokens_mask


def _bias_greenlist_logits(scores, greenlist_mask, greenlist_bias):
    """Bias the greenlist logits."""
    _scores = scores.clone()
    _scores[greenlist_mask] = scores[greenlist_mask] + greenlist_bias
    return _scores


def _sampling(logits, top_k=None, top_p=None, temperature=1.0):
    """Sampling function"""
    assert temperature > 0, "temperature must be positive"

    _logits = logits / temperature

    # Apply top-k sampling
    if top_k > 0:
        top_k = min(
            top_k, _logits.size(-1)
        )  # Ensure top_k is not greater than the vocabulary size
        indices_to_remove = _logits < torch.topk(_logits, top_k)[0][..., -1, None]
        _logits[indices_to_remove] = float("-inf")

    # Apply top-p sampling
    if top_p > 0 and top_p < 1:
        sorted_logits, sorted_indices = torch.sort(_logits, descending=True)
        cumulative_probs = torch.cumsum(F.softmax(sorted_logits, dim=-1), dim=-1)
        sorted_indices_to_remove = cumulative_probs > top_p
        if sorted_indices_to_remove[..., 1:].any():
            sorted_indices_to_remove[..., 1:] = sorted_indices_to_remove[
                ..., :-1
            ].clone()
            sorted_indices_to_remove[..., 0] = 0

        # scatter sorted tensors to original indexing
        indices_to_remove = sorted_indices_to_remove.scatter(
            1, sorted_indices, sorted_indices_to_remove
        )
        _logits[indices_to_remove] = float("-inf")

    # Get probability distribution
    probs = F.softmax(_logits, dim=-1)
    sampled_indices = torch.multinomial(probs, num_samples=1)
    return sampled_indices


def kgw_sampling(
    input_ids,
    scores,
    gamma,
    delta,
    prefix_length,
    vocab_size,
    temperature,
    top_k,
    top_p,
    prf,
):
    if input_ids.shape[-1] < prefix_length:
        return scores

    # print("Getting greenlist ids...")
    batched_greenlist_ids = _get_greenlist_ids_left(
        input_ids, gamma, prf, vocab_size, prefix_length
    )

    # print("Calculating greenlist mask...")
    green_tokens_mask = _calc_greenlist_mask(
        scores, greenlist_token_ids=batched_greenlist_ids
    )

    # print("Biasing greenlist logits...")
    scores = _bias_greenlist_logits(
        scores, greenlist_mask=green_tokens_mask, greenlist_bias=delta
    )

    # print("Sampling...")
    sampled_indices = _sampling(
        scores, top_k=top_k, top_p=top_p, temperature=temperature
    )

    return sampled_indices


def get_logits(ctx: str, logits: dict, tokenizer: AutoTokenizer):
    cur_logits = logits
    pre_str = "Example12:"
    pre_tokens = tokenizer.encode(pre_str, add_special_tokens=False)
    pre_ctx_tokens = tokenizer.encode(pre_str + ctx, add_special_tokens=False)
    ctx_token = pre_ctx_tokens[len(pre_tokens) :]

    for id in ctx_token:
        cur_logits = cur_logits[id]

    assert len(cur_logits.keys()) == 1
    return torch.tensor(cur_logits["logits"], device=device)


def sample_batch_wm(
    logits: torch.Tensor,
    batch_size,
    input_ids,
    vocab_size,
    temperature,
    top_k,
    top_p,
    gamma,
    delta,
    prefix_length,
    prf,
):
    # input_ids = input_ids.clone().expand(batch_size, -1).to(device)

    # Expand logits to batch_size
    logits_batch = logits.expand(batch_size, -1)

    tokens = kgw_sampling(
        input_ids=input_ids,
        scores=logits_batch,
        gamma=gamma,
        delta=delta,
        prefix_length=prefix_length,
        vocab_size=vocab_size,
        temperature=temperature,
        top_k=top_k,
        top_p=top_p,
        prf=prf,
    ).squeeze(1)

    return tokens.cpu().numpy()


def run(
    combinations,
    model_name,
    samples,
    gamma,
    delta,
    prefix_length,
    fill_parts,
    sample_iter,
    device,
):
    num_iters = samples
    assert num_iters % len(fill_parts) == 0
    batch_size = int(samples // len(fill_parts))  # Each fill_part is a batch

    tokenizer = AutoTokenizer.from_pretrained(model_path)
    if model_name in ["opt27b", "opt13b"]: # ATTENTION: vocab size for OPT models
        vocab_size = 50272
    else:
        vocab_size = tokenizer.vocab_size

    print("Loading remote logits...")
    with open(f"./logits/ngram-p1-logits-{model_name}.pickle", "rb") as f:
        remote_logits_1 = pickle.load(f)

    with open(f"./logits/ngram-p2-logits-{model_name}.pickle", "rb") as f:
        remote_logits_2 = pickle.load(f)

    print("Transporting logits to cuda...")
    remote_logits = [remote_logits_1, remote_logits_2]

    def convert_logits_to_tensor(d):
        for key, value in d.items():
            if isinstance(value, dict):
                convert_logits_to_tensor(value)
            elif key == "logits":
                d[key].to(device)

    convert_logits_to_tensor(remote_logits[0])
    convert_logits_to_tensor(remote_logits[1])

    print("Convert done. Starting sampling...")
    rng.manual_seed(hash_key)
    prf = torch.randperm(vocab_size, device=device, generator=rng)

    with torch.no_grad():
        for idx in range(2):
            print(f"Processing prompt {idx}...")

            for combination in combinations:
                temperature = combination["temperature"]
                top_p = combination["topp"]
                top_k = combination["topk"]

                print(
                    f"Running combination: temperature={temperature}, topp={top_p}, topk={top_k}"
                )

                json_file_name = f"{json_file_paths[idx]}-{model_name}-{scheme}-temp-{temperature}-topp-{top_p}-topk-{top_k}-gamma-{gamma}-delta-{delta}-prefixlen-{prefix_length}-{samples}-{len(fill_parts)}-iter-{sample_iter}.json"
                # if already exists, skip
                if os.path.exists(json_file_name):
                    print(f"File already exists: {json_file_name}")
                    continue

                mapping_S_wm = {}
                mapping_S_uw = {}

                # Each prompt has two windows fill_part
                for fill_part in fill_parts:
                    print(f"For prompt {idx}, fill part: {fill_part}")
                    print(f"Processing fill part: {fill_part}")
                    input_ids = tokenizer.encode(
                        prompts[idx] + fill_part, return_tensors="pt"
                    ).to(device)
                    input_ids = input_ids.repeat(batch_size, 1)
                    assert num_iters % batch_size == 0
                    assert num_iters // batch_size % len(fill_parts) == 0
                    for iter in range(num_iters // batch_size // len(fill_parts)):
                        print(f"Iter: {iter + 1}")

                        logits = get_logits(fill_part, remote_logits[idx], tokenizer)
                        wm_tokens = sample_batch_wm(
                            logits=logits,
                            batch_size=batch_size,
                            input_ids=input_ids,
                            vocab_size=vocab_size,
                            temperature=temperature,
                            top_k=top_k,
                            top_p=top_p,
                            gamma=gamma,
                            delta=delta,
                            prefix_length=prefix_length,
                            prf=prf,
                        )

                        if fill_part not in mapping_S_wm:
                            mapping_S_wm[fill_part] = {}
                            mapping_S_wm[fill_part]["S_wm"] = [0] * vocab_size

                        for token in wm_tokens:
                            mapping_S_wm[fill_part]["S_wm"][token] += 1

                results = {
                    "watermarked": {str(k): v for k, v in mapping_S_wm.items()},
                    "unwatermarked": {str(k): v for k, v in mapping_S_uw.items()},
                }

                with open(
                    json_file_name,
                    "w",
                ) as json_file:
                    ujson.dump(results, json_file, separators=(",", ":"))

                # Clear CUDA cache to free memory after each combination
                torch.cuda.empty_cache()
                print("Cleared CUDA cache after combination.")


if __name__ == "__main__":
    import argparse

    parser = argparse.ArgumentParser(description="KGW Sampling Batch for Water-Prob-V1")
    parser.add_argument(
        "--model_name",
        type=str,
        default="llama-2-7b-hf",
        required=False,
        help="model_name parameter",
    )
    parser.add_argument("--samples", type=int, required=False, help="samples parameter")
    parser.add_argument(
        "--gamma", type=float, default=0.5, required=False, help="gamma parameter"
    )
    parser.add_argument(
        "--delta", type=float, default=2.0, required=False, help="delta parameter"
    )
    parser.add_argument(
        "--prefix_length", type=int, default=4, required=False, help="prefix_length parameter"
    )
    parser.add_argument("--device", type=int, required=True, help="device parameter")
    parser.add_argument(
        "--option", default="experiment", type=str, required=False, help="top_k parameter"
    )
    parser.add_argument(
        "--fill_length", default=50, type=int, required=False, help="fill_length parameter"
    )
    parser.add_argument(
        "--model_path",
        default="/data2/shared_model/llama-2-7b-hf",
        type=str,
        required=False,
        help="model_path parameter",
    )
    parser.add_argument(
        "--scheme", default="time", type=str, required=False, help="scheme parameter"
    )
    parser.add_argument(
        "--sample_iter", type=int, required=True, help="sample_iter parameter"
    )

    args = parser.parse_args()

    model_path = args.model_path

    if args.option == "all":
        combinations = [
            {"temperature": 1.0, "topp": 1.0, "topk": 0},
            {"temperature": 0.8, "topp": 1.0, "topk": 0},
            {"temperature": 0.7, "topp": 1.0, "topk": 0},
            {"temperature": 0.6, "topp": 1.0, "topk": 0},
            {"temperature": 1.2, "topp": 1.0, "topk": 0},
            {"temperature": 1.4, "topp": 1.0, "topk": 0},
            {"temperature": 1.6, "topp": 1.0, "topk": 0},
            {"temperature": 1.0, "topp": 0.7, "topk": 0},
            {"temperature": 1.0, "topp": 0.8, "topk": 0},
            {"temperature": 1.0, "topp": 0.9, "topk": 0},
            {"temperature": 1.0, "topp": 1.0, "topk": 100},
            {"temperature": 1.0, "topp": 1.0, "topk": 200},
            {"temperature": 1.0, "topp": 1.0, "topk": 500},
            {"temperature": 0.8, "topp": 1.0, "topk": 50},
            {"temperature": 0.7, "topp": 1.0, "topk": 50},
            {"temperature": 0.6, "topp": 1.0, "topk": 50},
            {"temperature": 0.8, "topp": 0.7, "topk": 0},
            {"temperature": 0.7, "topp": 0.7, "topk": 0},
            {"temperature": 0.6, "topp": 0.7, "topk": 0},
            {"temperature": 0.6, "topp": 0.7, "topk": 50},
            {"temperature": 1.2, "topp": 0.7, "topk": 50},
            {"temperature": 0.8, "topp": 0.7, "topk": 50},
        ]
    elif args.option == "temp":
        combinations = [
            {"temperature": 1.5, "topp": 1.0, "topk": 0},
            {"temperature": 1.4, "topp": 1.0, "topk": 0},
            {"temperature": 1.3, "topp": 1.0, "topk": 0},
            {"temperature": 1.2, "topp": 1.0, "topk": 0},
            {"temperature": 1.1, "topp": 1.0, "topk": 0},
            {"temperature": 1.0, "topp": 1.0, "topk": 0},
            {"temperature": 0.9, "topp": 1.0, "topk": 0},
            {"temperature": 0.8, "topp": 1.0, "topk": 0},
            {"temperature": 0.7, "topp": 1.0, "topk": 0},
            {"temperature": 0.6, "topp": 1.0, "topk": 0},
            {"temperature": 0.5, "topp": 1.0, "topk": 0},
            {"temperature": 0.4, "topp": 1.0, "topk": 0},
            {"temperature": 0.3, "topp": 1.0, "topk": 0},
            {"temperature": 0.2, "topp": 1.0, "topk": 0},
            {"temperature": 0.1, "topp": 1.0, "topk": 0},
        ]
    elif args.option == "joint":
        combinations = [
            {"temperature": 0.8, "topp": 1.0, "topk": 50},
            {"temperature": 0.7, "topp": 1.0, "topk": 50},
            {"temperature": 0.6, "topp": 1.0, "topk": 50},
            {"temperature": 0.8, "topp": 0.7, "topk": 0},
            {"temperature": 0.7, "topp": 0.7, "topk": 0},
            {"temperature": 0.6, "topp": 0.7, "topk": 0},
            {"temperature": 0.6, "topp": 0.7, "topk": 50},
            {"temperature": 1.2, "topp": 0.7, "topk": 50},
            {"temperature": 0.8, "topp": 0.7, "topk": 50},
        ]
    elif args.option == "temp-most":
        combinations = [
            {"temperature": 1.2, "topp": 1.0, "topk": 0},
            {"temperature": 1.1, "topp": 1.0, "topk": 0},
            {"temperature": 1.0, "topp": 1.0, "topk": 0},
            {"temperature": 0.9, "topp": 1.0, "topk": 0},
            {"temperature": 0.8, "topp": 1.0, "topk": 0},
        ]
    elif args.option == "experiment":
        combinations = [
            # modified
            # TODO
            {"temperature": 1.0, "topp": 1.0, "topk": 0},
        ]
    print("Device: ", args.device)
    os.environ["CUDA_VISIBLE_DEVICES"] = f"{args.device}"
    device = torch.device("cuda" if torch.cuda.is_available() else "cpu")

    # KGW Config
    hash_key = 15485863
    rng = torch.Generator(device=device)
    rng.manual_seed(hash_key)

    # Fixed seed random sampling fill_parts
    import random

    random.seed(64)
    fill_parts = random.sample(combinations_main, args.fill_length)

    print("Fill parts: ")
    print(fill_parts)

    f_scheme_map = {
        "time": _f_time,
        "additive": _f_additive,
        "skip": _f_skip,
        "min": _f_min,
    }
    scheme = args.scheme
    _f = f_scheme_map[scheme]

    run(
        combinations=combinations,
        model_name=args.model_name,
        samples=args.samples,
        gamma=args.gamma,
        delta=args.delta,
        prefix_length=args.prefix_length,
        fill_parts=fill_parts,
        sample_iter=args.sample_iter,
        device=device,
    )
    sys.exit(0)
