'''ResNet in PyTorch.
For Pre-activation ResNet, see 'preact_resnet.py'.
Reference:
[1] Kaiming He, Xiangyu Zhang, Shaoqing Ren, Jian Sun
    Deep Residual Learning for Image Recognition. arXiv:1512.03385
'''
import torch
import torch.nn as nn
import torch.nn.functional as F
import math


def get_closest_factors(num): 
    num_root = int(math.sqrt(num))
    while num % num_root != 0: 
        num_root -= 1
    return num_root, int(num / num_root)


class PooledConv(nn.Module):
  def __init__ (self, in_planes, planes, kernel_size, stride, padding=0, bias=False, 
  pool_type='mean', max_num_pools=2, noise_std=0.2):
    super(PooledConv, self).__init__()
    self.in_planes = in_planes
    self.planes = planes
    self.kernel_size = kernel_size
    self.kw, self.kh = get_closest_factors(planes)
    self.stride = stride
    self.padding = padding
    self.bias = bias 
    self.pool_type = pool_type
    self.max_num_pools = max_num_pools
    self.noise_std = noise_std
    self.conv = nn.Conv2d(in_channels=self.in_planes, out_channels=self.planes, 
                           kernel_size=kernel_size, stride=self.stride, padding=self.padding, bias=self.bias)

  def forward(self,x): 
    out = self.conv(x)
    if self.pool_type is not None:
      out_shape = out.shape
      reshaped_out = out.permute(0, 2, 3, 1).reshape(out_shape[0], out_shape[-1] * out_shape[-2], self.kw, self.kh)
      max_kernel_size = min(self.kw, self.kh)

      k = 3
      c = 1
      pooled_output = 0
      while (c <= self.max_num_pools) & (k < max_kernel_size):
        if self.pool_type == 'mean':
          pooled_output = pooled_output + F.avg_pool2d(reshaped_out, kernel_size=k, stride=1, padding=(k-1)//2, count_include_pad=False)
        elif self.pool_type == 'max':
          pooled_output = pooled_output + F.max_pool2d(reshaped_out, kernel_size=k, stride=1, padding=(k-1)//2)
        else: 
          raise ValueError(f'Pool type {self.pool_type} not recognized.')
        k = 2 * k + 1
        c += 1
      out = pooled_output
      out = out.reshape(out_shape[0], out_shape[-1], out_shape[-2], self.kw * self.kh).permute(0, 3, 1, 2)
    if self.noise_std > 0:
      out = out + torch.randn(out.size(), device=out.device) * self.noise_std
    return out


class BasicBlock(nn.Module):
  expansion = 1
  def __init__(self, in_planes, planes, pool_type, max_num_pools, noise_std, stride=1, activation=F.relu):
    super(BasicBlock, self).__init__()
    self.activation = activation
    self.conv1 = PooledConv(in_planes, planes, kernel_size=3, stride=stride, padding=1, bias=False, 
    pool_type=pool_type, max_num_pools=max_num_pools, noise_std=noise_std)
    self.bn1 = nn.BatchNorm2d(planes)
    self.conv2 = PooledConv(planes, planes, kernel_size=3, stride=1, padding=1, bias=False, 
    pool_type=pool_type, max_num_pools=max_num_pools, noise_std=noise_std)
    self.bn2 = nn.BatchNorm2d(planes)

    self.shortcut = nn.Sequential()
    if stride != 1 or in_planes != self.expansion * planes:
      self.shortcut = nn.Sequential(
        PooledConv(in_planes, self.expansion * planes, kernel_size=1, stride=stride, bias=False, 
        pool_type=pool_type, max_num_pools=max_num_pools, noise_std=noise_std),
        nn.BatchNorm2d(planes)
      )

  def forward(self, x):
    out = self.activation(self.bn1(self.conv1(x)))
    out = self.bn2(self.conv2(out))
    out += self.shortcut(x)
    out = self.activation(out)
    return out


class Bottleneck(nn.Module):
  expansion = 4

  def __init__(self, in_planes, planes, pool_type, max_num_pools, noise_std, stride=1, activation=F.relu):
    super(Bottleneck, self).__init__()
    self.activation = activation
    self.conv1 = PooledConv(in_planes, planes, kernel_size=1, bias=False, 
    pool_type=pool_type, max_num_pools=max_num_pools, noise_std=noise_std)
    self.bn1 = nn.BatchNorm2d(planes)
    self.conv2 = PooledConv(planes, planes, kernel_size=3, stride=stride, padding=1, bias=False, 
    pool_type=pool_type, max_num_pools=max_num_pools, noise_std=noise_std)
    self.bn2 = nn.BatchNorm2d(planes)
    self.conv3 = PooledConv(planes, self.expansion * planes, kernel_size=1, bias=False, 
    pool_type=pool_type, max_num_pools=max_num_pools, noise_std=noise_std)
    self.bn3 = nn.BatchNorm2d(self.expansion * planes)

    self.shortcut = nn.Sequential()
    if stride != 1 or in_planes != self.expansion * planes:
      self.shortcut = nn.Sequential(
        PooledConv(in_planes, self.expansion * planes, kernel_size=1, stride=stride, bias=False, 
        pool_type=pool_type, max_num_pools=max_num_pools, noise_std=noise_std),
        nn.BatchNorm2d(self.expansion * planes)
      )

  def forward(self, x):
    out = self.activation(self.bn1(self.conv1(x)))
    out = self.activation(self.bn2(self.conv2(out)))
    out = self.bn3(self.conv3(out))
    out += self.shortcut(x)
    out = self.activation(out)
    return out


class ResNet(nn.Module):
  def __init__(self, block, num_blocks, num_classes,
               pool_type, max_num_pools, noise_std, activation=F.relu):
    super(ResNet, self).__init__()
    self.in_planes = 64
    self.pool_type = pool_type
    self.max_num_pools = max_num_pools
    self.noise_std = noise_std
    self.activation = activation

    self.conv1 = PooledConv(3, 64, kernel_size=3, stride=1, padding=1, bias=False, 
    pool_type=pool_type, max_num_pools=max_num_pools, noise_std=noise_std)
    self.bn1 = nn.BatchNorm2d(64)
    self.layer1 = self._make_layer(block, 64, num_blocks[0], stride=1, activation=self.activation, 
    pool_type=pool_type, max_num_pools=max_num_pools, noise_std=noise_std)
    self.layer2 = self._make_layer(block, 128, num_blocks[1], stride=2, activation=self.activation, 
    pool_type=pool_type, max_num_pools=max_num_pools, noise_std=noise_std)
    self.layer3 = self._make_layer(block, 256, num_blocks[2], stride=2, activation=self.activation, 
    pool_type=pool_type, max_num_pools=max_num_pools, noise_std=noise_std)
    self.layer4 = self._make_layer(block, 512, num_blocks[3], stride=2, activation=self.activation,
    pool_type=pool_type, max_num_pools=max_num_pools, noise_std=noise_std)
    self.linear = nn.Linear(512 * block.expansion, num_classes)

  def _make_layer(self, block, planes, num_blocks, stride, activation, pool_type, max_num_pools, noise_std):
    strides = [stride] + [1] * (num_blocks - 1)
    layers = []
    for stride in strides:
      layers.append(block(self.in_planes, planes, stride=stride, activation=activation, 
      pool_type=pool_type, max_num_pools=max_num_pools, noise_std=noise_std))
      self.in_planes = planes * block.expansion
    return nn.Sequential(*layers)

  def forward(self, x):
    out = self.activation(self.bn1(self.conv1(x)))
    out = self.layer1(out)
    out = self.layer2(out)
    out = self.layer3(out)
    out = self.layer4(out)
    out = F.avg_pool2d(out, 4)
    out = out.view(out.size(0), -1)
    out = self.linear(out)
    return out


def ResNet18(num_classes, pool_type, max_num_pools, noise_std):
  return ResNet(BasicBlock, [2, 2, 2, 2], num_classes, 
  pool_type, max_num_pools, noise_std)

