# Ito Vision

**Ito Vision** is a Python library built with PyTorch that provides a unified framework for iterative refinement methods.  
It implements a wide range of existing approaches together with flexible components for discretizations, parameterizations, samplers, and schedulers.  
The modular design makes it straightforward to reproduce existing results while also enabling researchers to prototype and integrate new methods with minimal effort.  

The library is organized into five core modules:  
- **discretizations** – numerical schemes for generating time steps (e.g., linear or logarithmic schedules),  
- **methods** – implementations of iterative refinement algorithms,  
- **parameterizations** – choices of network targets (e.g., predicting $x_0$, $\epsilon$, or score) and corresponding loss weighting strategies,  
- **samplers** – algorithms for drawing samples from the defined processes,  
- **schedulers** – routines to control step sizes and noise levels over time.  

For more background on the design choices and mathematical framework, please refer to our paper.


## Installation

Use `pip`:

```bash
pip install ito-vision
```

## Quickstart

```python
from ito_vision.discretizations import LinearDiscretization
from ito_vision.methods import IRSDE
from ito_vision.parametrizations import IdentityParametrization
from ito_vision.samplers import EulerMaruyamaSampler
from ito_vision.schedulers import CosineScheduler


parametrization = IdentityParametrization(target="x0", loss_weight_type="uniform")
scheduler = CosineScheduler()
discretization = LinearDiscretization()
sampler = EulerMaruyamaSampler(N=30, Lambda=0, quiet=False)
method = IRSDE(scheduler=scheduler, tau=0.2, parametrization=parametrization)

backbone = ...
optimizer = torch.optim.Adam(backbone.parameters(), lr=1e-4)

train_dataloader = ...
val_dataloader = ...

# train loop
for batch in train_dataloader:
    y, x0 = batch["y"], batch["x0"]

    optimizer.zero_grad()

    loss = method.loss(backbone, x0, y=y)

    loss.backward()
    optimizer.step()

# sampling
for batch in val_dataloader:
    y, x0 = batch["y"], batch["x0"]

    pred_x0, _, _ = method.sample(
        discretization,
        sampler,
        backbone,
        method.base_distribution(y),
        y,
        return_trajectory=False,
    )
```

## Modules

### Methods

| Method                                              | Name                                                                                | Reference                                            |
| --------------------------------------------------- | ----------------------------------------------------------------------------------- | ---------------------------------------------------- |
| [BBDM](ito_vision/methods/bbdm.py)                  | Brownian Bridge Diffusion Models                                                    | [github](https://github.com/xuekt98/BBDM)            |
| [DDBM_VE](ito_vision/methods/ddbm_ve.py)            | Denoising Diffusion Bridge Models                                                   | [github](https://github.com/alexzhou907/DDBM)        |
| [DDBM_VP](ito_vision/methods/ddbm_vp.py)            | Denoising Diffusion Bridge Models                                                   | [github](https://github.com/alexzhou907/DDBM)        |
| [DDPM](ito_vision/methods/ddpm.py)                  | Denoising Diffusion Probabilistic Models                                            | [page](https://hojonathanho.github.io/diffusion/)    |
| [FlowMatching](ito_vision/methods/flow_matching.py) | Flow Matching                                                                       | [paper](https://arxiv.org/abs/2210.02747)            |
| [GOUB](ito_vision/methods/goub.py)                  | Image Restoration Through Generalized Ornstein-Uhlenbeck Bridge                     | [github](https://github.com/Hammour-steak/GOUB)      |
| [I2SB](ito_vision/methods/i2sb.py)                  | I2SB: Image-to-Image Schrödinger Bridge                                             | [page](https://i2sb.github.io/)                      |
| [InDI](ito_vision/methods/indi.py)                  | Inversion by Direct Iteration (InDI)                                                | [paper](https://arxiv.org/abs/2303.11435)            |
| [IRSDE](ito_vision/methods/ir_sde.py)               | Image Restoration with Mean-Reverting Stochastic Differential Equations             | [page](https://algolzw.github.io/ir-sde/index.html)  |
| [ResShift](ito_vision/methods/resshift.py)          | ResShift: Efficient Diffusion Model for Image Super-Resolution by Residual Shifting | [page](https://zsyoaoa.github.io/projects/resshift/) |
| [UniDB](ito_vision/methods/unidb.py)                | UniDB: A Unified Diffusion Bridge Framework via Stochastic Optimal Control          | [page](https://unidb-soc.github.io/UniDB_page/)      |

### Discretizations

| Discretization                                               |
| ------------------------------------------------------------ |
| [DDBMDiscretization](ito_vision/discretizations/ddbm.py)     |
| [KarrasDiscretization](ito_vision/discretizations/karras.py) |
| [LinearDiscretization](ito_vision/discretizations/linear.py) |

### Parametrizations

| Parametrization                                                             |
| --------------------------------------------------------------------------- |
| [DDBMParametrization](ito_vision/parametrizations/ddbm.py)                  |
| [DDBMLLIParametrization](ito_vision/parametrizations/ddbm_llie.py)          |
| [IdentityParametrization](ito_vision/parametrizations/identity.py)          |
| [IdentityLLIEParametrization](ito_vision/parametrizations/identity_llie.py) |

### Samplers

| Sampler                                                       |
| ------------------------------------------------------------- |
| [AncestralSampler](ito_vision/samplers/ancestral.py)          |
| [EIODESampler](ito_vision/samplers/ei_ode.py)                 |
| [EulerMaruyamaSampler](ito_vision/samplers/euler_maruyama.py) |
| [LangevinHeunSampler](ito_vision/samplers/langevin_heun.py)   |
| [MeanODESampler](ito_vision/samplers/mean_ode.py)             |
| [RungeKutta2Sampler](ito_vision/samplers/runge_kutta2.py)     |

### Schedulers

| Scheduler                                                                   |
| --------------------------------------------------------------------------- |
| [LinearScheduler](ito_vision/schedulers/linear.py)                          |
| [CosineScheduler](ito_vision/schedulers/cosine.py.py)                       |
| [ExponentialScheduler](ito_vision/schedulers/exponential.py)                |
| [QuadraticScheduler](ito_vision/schedulers/quadratic.py)                    |
| [InversedScheduler](ito_vision/schedulers/inversed.py)                      |
| [QuadraticSymmetricScheduler](ito_vision/schedulers/quadratic_symmetric.py) |
| [ConstantScheduler](ito_vision/schedulers/constant.py)                      |

