from collections import OrderedDict
from copy import deepcopy
from math import sqrt

from run_parallel import run_parallel


family_idx = 49
family_name = "truncated_scale_tune"


base_config = {
    "optimizer_params": [{
        "name": "nesgd",
        "lr": [0.001],
        "spectral_scale": 1.0,
        "weight_decay": 0.0,
        "lr_schedule": "warmup-constant-linear",
        "warm_up_fraction": 0.05,
        "cool_down_fraction": 0.5,
        "truncate_loss": 3.2,
    }],
    "training_params": {
        "tokens_processed": 524288, # 2^19
        "val_tokens_processed": 8388608, #2^23
        "batch_size": 64,
        "num_epochs": 1,
        "context_length": 1024,
        "gradnorm": 0.0,
        "tensorcore_precision": "high",   #Can be highest, high, or medium
        "autocast": True,
        "mixed_precision": "bfloat16",
        "compile": True,
    },
    "logging_params": {
        "val_tokens_processed": 8388608,
        "log_step": 256,
        "val_step": 256,
        "save_ckpt_step": 512,
        "load_ckpt_step": 0,
        "keep_last": 2,
        "ckpt_dir": "",
    },
    "gpt_model": {
        "n_embd": 768,
        "n_layer": 12,
        "n_head": 12,
        "vocab_size": 50257,
        "flash_attention": True,
    },
    "dataset": {
        "name": "fineweb1B"
    },
}

# Generate configs for this experiment.
tuned_lrs = {
    "nesgd": {"muon": 0.001, "other": 1e-5},
    "nesgd-lmo": {"muon": 0.01, "other": 1e-3},
    "nesgd-adam_2": {"muon": 0.001, "other": 1e-4},
    "nesgd-adam_2-lmo": {"muon": 0.01, "other": 1e-3},
    "nesgd-adam_infty": {"muon": 0.001, "other": 1e-5},
    "nesgd-adam_infty-lmo": {"muon": 0.01, "other": 1e-3},
    "nesgd-l2_prod": {"muon": 0.1, "other": 1e-6},
    "nesgd-l2_prod-lmo": {"muon": 0.1, "other": 0.01},
    "nesgd-l2_prod-adam_2": {"muon": 0.1, "other": 1e-3},
    "nesgd-l2_prod-adam_2-lmo": {"muon": 0.1, "other": 0.01},
    "nesgd-l2_prod-adam_infty": {"muon": 0.1, "other": 1e-4},
    "nesgd-l2_prod-adam_infty-lmo": {"muon": 0.1, "other": 0.01},
    "nesgd-hybrid_prod": {"muon": 0.001, "other": 1e-5},
    "nesgd-hybrid_prod-lmo": {"muon": 0.01, "other": 1e-3},
    "nesgd-hybrid_prod-adam_2": {"muon": 0.001, "other": 1e-3},
    "nesgd-hybrid_prod-adam_2-lmo": {"muon": 0.01, "other": 0.01},
    "nesgd-hybrid_prod-adam_infty": {"muon": 0.001, "other": 1e-4},
    "nesgd-hybrid_prod-adam_infty-lmo": {"muon": 0.01, "other": 0.01},
}
scales = [0.3, 1.0, 3.0, 10.0, 30.0, 100.0]

# Sweep other lr.
experiment_configs = OrderedDict()
for alg, lrs in tuned_lrs.items():
    for scale in scales:
        muon_lr = scale * lrs["muon"]
        other_lr = scale * lrs["other"]
        s = muon_lr / other_lr
        spectral_scale = s if "lmo" in alg else sqrt(s)

        opt_settings = {
            "name": alg + "-momo",
            "lr": [other_lr],
            "spectral_scale": spectral_scale,
        }
        current_config = deepcopy(base_config)
        current_config["optimizer_params"][0].update(dict(opt_settings))
        run_name = f"{family_idx}_{family_name}/{alg}_{scale}"
        experiment_configs[run_name] = deepcopy(current_config)

# Launch runs in parallel.
run_parallel(experiment_configs)
