import torch
import torch.nn as nn
from torch.nn import functional as F
from torchvision.models._utils import IntermediateLayerGetter

from dassl.engine import TRAINER_REGISTRY, TrainerX
from dassl.optim import build_optimizer, build_lr_scheduler

from clip import clip
from clip.model import convert_weights

from .coop import load_clip_to_cpu
from .imagenet_templates import IMAGENET_TEMPLATES, IMAGENET_TEMPLATES_SELECT
from tqdm import tqdm
import numpy as np
import math
import time

CUSTOM_TEMPLATES = {
    "OxfordPets": "a photo of a {}, a type of pet.",
    "OxfordFlowers": "a photo of a {}, a type of flower.",
    "FGVCAircraft": "a photo of a {}, a type of aircraft.",
    "DescribableTextures": "{} texture.",
    "EuroSAT": "a centered satellite photo of {}.",
    "StanfordCars": "a photo of a {}.",
    "Food101": "a photo of {}, a type of food.",
    "SUN397": "a photo of a {}.",
    "Caltech101": "a photo of a {}.",
    "UCF101": "a photo of a person doing {}.",
    "ImageNet": "a photo of a {}.",
    "ImageNetSketch": "a photo of a {}.",
    "ImageNetV2": "a photo of a {}.",
    "ImageNetA": "a photo of a {}.",
    "ImageNetR": "a photo of a {}."
}
label_names = [
    "person", "bicycle", "car", "motorcycle", "airplane", "bus", "train", "truck",
    "boat", "traffic light", "fire hydrant", "stop sign", "parking meter", "bench",
    "bird", "cat", "dog", "horse", "sheep", "cow", "elephant", "bear", "zebra", "giraffe",
    "backpack", "umbrella", "handbag", "tie", "suitcase", "frisbee", "skis", "snowboard",
    "sports ball", "kite", "baseball bat", "baseball glove", "skateboard", "surfboard",
    "tennis racket", "bottle", "wine glass", "cup", "fork", "knife", "spoon", "bowl",
    "banana", "apple", "sandwich", "orange", "broccoli", "carrot", "hot dog", "pizza",
    "donut", "cake", "chair", "couch", "potted plant", "bed", "dining table", "toilet",
    "tv", "laptop", "mouse", "remote", "keyboard", "cell phone", "microwave", "oven",
    "toaster", "sink", "refrigerator", "book", "clock", "vase", "scissors", "teddy bear",
    "hair drier", "toothbrush"
]

MEASURE_TIME = False
WTA = False
RESNET101 = True
DATASET = "COCO"

CLASS_ADAPT_GLOBAL = True
CLASS_ADAPT_LOCAL = True

TASK_ADAPT_GLOBAL = True
TASK_ADAPT_LOCAL = True

COCO_CUT = 40000 #1000~32000
#VOC_CUT = 32000 #1000~32000
VOC_CUT=0

K=3
GZSL=False
FEWSHOT=False
UNSEEN=True



@TRAINER_REGISTRY.register()
class ZeroshotCLIP(TrainerX):
    def build_model(self):
        cfg = self.cfg
        classnames = self.dm.dataset.classnames

        print(f"Loading CLIP (backbone: {cfg.MODEL.BACKBONE.NAME})")
        clip_model = load_clip_to_cpu(cfg)
        clip_model.to(self.device)

        try:
            temp = CUSTOM_TEMPLATES[cfg.DATASET.NAME]
        except:
            print('!! WARNING: Not found template for {}'.format(cfg.DATASET.NAME))
            temp = "a photo of a {}."

        prompts = [temp.format(c.replace("_", " ")) for c in classnames]
        print(f"Prompts: {prompts}")
        prompts = torch.cat([clip.tokenize(p) for p in prompts])
        prompts = prompts.to(self.device)

        with torch.no_grad():
            text_features = clip_model.encode_text(prompts)
            text_features = text_features / text_features.norm(dim=-1, keepdim=True)

        self.text_features = text_features
        self.clip_model = clip_model

    def model_inference(self, image):
        image_features = self.clip_model.encode_image(image)
        image_features = image_features / image_features.norm(dim=-1, keepdim=True)
        logit_scale = self.clip_model.logit_scale.exp()
        logits = logit_scale * image_features @ self.text_features.t()

        return logits, None, None


def read_npy(numpy_file, device):
    all_features = np.load(numpy_file, allow_pickle=True)


    print('load complete:', numpy_file)
    return torch.tensor(all_features).to(torch.float16).to(
        device)  # text_caption.to(device) #torch.tensor(all_features).to(device)


@TRAINER_REGISTRY.register()
class ZeroshotCLIP_dense(TrainerX):
    def build_model(self):
        cfg = self.cfg
        classnames = self.dm.dataset.classnames

        print(f"Loading CLIP (backbone: {cfg.MODEL.BACKBONE.NAME})")
        clip_model = load_clip_to_cpu(cfg)
        clip_model.to(self.device)

        try:
            temp = CUSTOM_TEMPLATES[cfg.DATASET.NAME]
        except:
            print('!! WARNING: Not found template for {}'.format(cfg.DATASET.NAME))
            temp = "a photo of a {}."

        prompts = [temp.format(c.replace("_", " ")) for c in classnames]
        print(f"Prompts: {prompts}")
        prompts = torch.cat([clip.tokenize(p) for p in prompts])
        prompts = prompts.to(self.device)

        with torch.no_grad():
            text_features = clip_model.encode_text(prompts)
            text_features = text_features / text_features.norm(dim=-1, keepdim=True)

        self.text_features = text_features
        self.clip_model = clip_model

        self.visual_encoder = IntermediateLayerGetter(self.clip_model.visual, {"layer4": "0"})
        self.positional_embedding = self.clip_model.visual.attnpool.positional_embedding[1::]
        self.v_linear_weight = self.clip_model.visual.attnpool.v_proj.weight
        self.v_linear_bias = self.clip_model.visual.attnpool.v_proj.bias
        self.c_linear_weight = self.clip_model.visual.attnpool.c_proj.weight
        self.c_linear_bias = self.clip_model.visual.attnpool.c_proj.bias

        if DATASET == "COCO":
            if RESNET101:
                self.caption = read_npy('coco_llm_caption_selection_32000_low_quality_sentence.npy', self.device)
                self.caption = self.caption / self.caption.norm(dim=-1, keepdim=True)
                self.caption_ori = read_npy('COCO_caption_selection_101_syn_class_rel_dup_200.npy', self.device)
                self.caption_ori = self.caption_ori / self.caption_ori.norm(dim=-1, keepdim=True)
                self.caption_local = read_npy('COCO_caption_selection_local_101.npy', self.device)
                self.caption_local = self.caption_local / self.caption_local.norm(dim=-1, keepdim=True)
                self._multi_label_caption = read_npy('coco_llm_multi_label_caption_selection_32000_low_quality.npy', self.device)
                self._multi_label_caption = self._multi_label_caption / self._multi_label_caption.norm(dim=-1,
                                                                                                       keepdim=True)
                self._multi_label_caption_ori = read_npy('COCO_multi_label_caption_selection_101_32000_syn.npy', self.device)
                self._multi_label_caption_ori = self._multi_label_caption_ori / self._multi_label_caption_ori.norm(dim=-1,
                                                                                                       keepdim=True)
                self.caption = self.caption_ori
                self._multi_label_caption = torch.cat((self._multi_label_caption[:400,:,:],self._multi_label_caption_ori[:,:,:]), 0)
            else:
                self.caption = read_npy('COCO_caption_selection.npy', self.device)
                self.caption = self.caption / self.caption.norm(dim=-1, keepdim=True)

                self.caption_local = read_npy('COCO_caption_selection_local.npy', self.device)
                self.caption_local = self.caption_local / self.caption_local.norm(dim=-1, keepdim=True)

                self._multi_label_caption = read_npy('COCO_multi_label_caption_selection_50_32000.npy', self.device)
                self._multi_label_caption = self._multi_label_caption / self._multi_label_caption.norm(dim=-1,
                                                                                                       keepdim=True)

        elif DATASET == "NUSWIDE":
            if RESNET101:
                self.caption = read_npy('nus_llm_caption_selection_101_87500_one_cls.npy', self.device)
                self.caption = self.caption / self.caption.norm(dim=-1, keepdim=True)

                self._multi_label_caption = read_npy('nus_llm_multi_label_caption_selection_101_87500_one_cls.npy', self.device)
                self._multi_label_caption = self._multi_label_caption / self._multi_label_caption.norm(dim=-1,
                                                                                                       keepdim=True)
            else:
                self.caption = read_npy('nus_llm_caption_selection_50_57600_one_cls.npy', self.device)
                self.caption = self.caption / self.caption.norm(dim=-1, keepdim=True)
                self._multi_label_caption = read_npy('nus_llm_multi_label_caption_selection_50_57600_one_cls.npy', self.device)
                self._multi_label_caption = self._multi_label_caption / self._multi_label_caption.norm(dim=-1,
                                                                                                       keepdim=True)
                


        elif DATASET == "VOC":
            if RESNET101:
                self.caption = read_npy('VOC_caption_selection_100_101.npy', self.device)
                self.caption = self.caption / self.caption.norm(dim=-1, keepdim=True)

                self._multi_label_caption = read_npy('VOC_multi_label_caption_selection_101_v5_40000.npy', self.device)
                self._multi_label_caption = self._multi_label_caption / self._multi_label_caption.norm(dim=-1,
                                                                                                       keepdim=True)
               # self._multi_label_caption = self._multi_label_caption.permute(1,0,2) 200 100 512
            else:
                self.caption = read_npy('VOC_caption_selection_100_50.npy', self.device)
                self.caption = self.caption / self.caption.norm(dim=-1, keepdim=True)
                
                self._multi_label_caption = read_npy('VOC_multi_label_caption_selection_50_v5_40000.npy', self.device)
                self._multi_label_caption = self._multi_label_caption / self._multi_label_caption.norm(dim=-1,
                                                                                                       keepdim=True)
                
                
    def encode_image(self, x):
        def stem(x):
            for conv, bn in [(self.visual_encoder.conv1, self.visual_encoder.bn1), \
                             (self.visual_encoder.conv2, self.visual_encoder.bn2),
                             (self.visual_encoder.conv3, self.visual_encoder.bn3)]:
                x = self.visual_encoder.relu(bn(conv(x)))
            x = self.visual_encoder.avgpool(x)
            return x

        x = x.type(self.visual_encoder.conv1.weight.dtype)
        x = stem(x)
        x = self.visual_encoder.layer1(x)
        x = self.visual_encoder.layer2(x)
        x = self.visual_encoder.layer3(x)
        x = self.visual_encoder.layer4(x)
        return x

    def model_inference(self, image, label):
        image_feat = self.encode_image(image)
        b, c, h, w = image_feat.shape
        x = image_feat.reshape(b, c, h * w).permute(2, 0, 1)

        x = F.linear(x, self.v_linear_weight, self.v_linear_bias)
        x = F.linear(x, self.c_linear_weight, self.c_linear_bias)
        image_features = x

        image_feature_, _ = self.clip_model.visual.attnpool(image_feat)
        image_features = image_features / image_features.norm(dim=-1, keepdim=True)
        image_feature_ = image_feature_ / image_feature_.norm(dim=-1, keepdim=True)

        def get_global_p_three_step(caption, image_feature_, temp):  # caption: [40, 40, 20, 512], image_feature_: [g, 4, 512]
            # i =40, j=40, k=20, b = 4, g= 1
            
            ## 1 step
            similarity = temp * torch.einsum('ijkc,gcb->gijkb', caption, image_feature_.permute(0,2,1))  # [g,40,40, 20, 4]
            attention_weights = F.softmax(similarity, dim=3) # [g,40,40, 20, 4]
            # torch.Size([g, 40, 40, 20, 4]) # caption: [40, 40, 20, 512]
            weighted_sum = torch.einsum('gijbk,ijkc->gijbc', attention_weights.permute(0,1, 2, 4, 3),caption) 
            # torch.Size([g, 40, 40, 4, 512])
            weighted_sum = weighted_sum.permute(0,3, 1, 2, 4)  
            # torch.Size([g, 4, 40, 40, 512]) 
            
            ## 2 step
            # # torch.Size([g, 4, 40, 40, 512]) # torch.Size([g, 4, 512])
            similarity_2 = temp * torch.einsum('gbijc,gbc->gbij', weighted_sum, image_feature_)  # [g,4,40,40]
            attention_weights_2 = F.softmax(similarity_2, dim=3) #   [g,4,40,40]
            weighted_sum_2 = torch.einsum('gbij,gbijc->gbic', attention_weights_2, weighted_sum)  # [g,4,40,512]
            
            ## 3 step
            # torch.Size([1, 4, 40, 512])  #torch.Size([g, 4, 512])
            similarity_3 = temp * torch.einsum('gbic,gbc->gbi', weighted_sum_2, image_feature_)  # [g,4,40]
            attention_weights_3 = F.softmax(similarity_3, dim=2)  # [g,4,40]
            global_p = torch.einsum('gbi,gbic->gbc', attention_weights_3, weighted_sum_2)  # [g,4,512]

            
            return global_p

        
        if TASK_ADAPT_GLOBAL:
        
            dim1, dim2, depth = self._multi_label_caption.shape
            multi_label_caption = self._multi_label_caption.reshape(-1,depth)

            if DATASET=='VOC':
                multi_label_caption = multi_label_caption[:COCO_CUT]
                dim1_1 =40
                dim2 =20
                TEMP1 = 55
                TEMP2 = 30
                CLS_GLO_TEMP = 1/100
                CLS_LOC_TEMP = 1/100
            elif DATASET=='NUSWIDE':
                dim1_1 = 40
                dim2 = 36
                TEMP1 = 40
                TEMP2 = 10
                CLS_GLO_TEMP = 1/100
                CLS_LOC_TEMP = 1/100
            elif DATASET=='COCO':
                multi_label_caption = multi_label_caption[:COCO_CUT]
                TEMP1 = 80
                TEMP2 = 40
                CLS_GLO_TEMP = 1
                CLS_LOC_TEMP = 5
                if multi_label_caption.shape[0] == 1000:
                    dim1_1=10
                    dim2 =10
                elif multi_label_caption.shape[0] == 2000:
                    dim1_1=10
                    dim2 =20
                elif multi_label_caption.shape[0] == 4000:
                    dim1_1=20
                    dim2 =10
                elif multi_label_caption.shape[0] == 8000:
                    dim1_1=20
                    dim2 =20
                elif multi_label_caption.shape[0] == 16000:
                    dim1_1=20
                    dim2 =40
                elif multi_label_caption.shape[0] == 32000:
                    dim1_1=40
                    dim2 =20
                elif multi_label_caption.shape[0] == 40000:
                    dim1_1=40
                    dim2 =25
                

            multi_step_input = multi_label_caption.reshape(dim1_1, dim1_1, dim2, depth)
            global_p = get_global_p_three_step(multi_step_input, image_feature_.unsqueeze(0), TEMP1)
            global_p = global_p[0]
            image_feature_temp = global_p 
            
        else:
            if DATASET=='VOC':
                CLS_GLO_TEMP = 1/100
                CLS_LOC_TEMP = 1/100
            elif DATASET=='NUSWIDE':
                CLS_GLO_TEMP = 1/100
                CLS_LOC_TEMP = 1/100
            elif DATASET=='COCO':
                CLS_GLO_TEMP = 1
                CLS_LOC_TEMP = 5
            
            image_feature_temp = image_feature_
        
        logit_scale = self.clip_model.logit_scale.exp()  # logit_scale = self.clip_model.logit_scale.exp()
        
        
        ####global start #######
            
        if CLASS_ADAPT_GLOBAL:
            A = torch.matmul(self.caption, image_feature_temp.t())  # 80 16 100
            A_up = F.softmax(A*CLS_GLO_TEMP, dim=1) # origin 
            #A_up = F.softmax(A/100, dim=1) # edit for voc
            P_up = torch.matmul(A_up.permute(0, 2, 1), self.caption)  # 80 100 512
            
            logits_ = logit_scale * (torch.sum(P_up * image_feature_temp.unsqueeze(0), dim=2).permute(1,0) + image_feature_ @ self.text_features.t()) / 2.0 
        else:
            logits_ = logit_scale * image_feature_ @ self.text_features.t() #origin 
            
            
        ####global end #######
        
        ####local start#######

        if TASK_ADAPT_LOCAL:
            grid_max = image_feat.shape[-1]

            local_logit_list = []

            updated_img_feat = get_global_p_three_step(multi_step_input, image_features, TEMP2)
            
            A = torch.einsum('gbc,nkc->gnkb',updated_img_feat, self.caption) # [gnkb]
            A_up = F.softmax(A *CLS_LOC_TEMP, dim=2) 
            local_logit_list = logit_scale * torch.sum(A_up * A, dim=2) # ([gnb])        
            logits = (local_logit_list.permute(0, 2, 1) + logit_scale * image_features @ self.text_features.t()) / 2.0  
        

        a = (label[0, :] == 1).nonzero(as_tuple=False)
        
        
        ####local end #######
        
        #### WTA start #########

        if WTA == False:
            prob_spatial = torch.nn.functional.softmax(logits, dim=0) 
            logits = torch.sum(logits * prob_spatial, dim=0)
        else:
            # WTA
            grid_size = int(math.sqrt(logits.shape[0]))
            bs = logits.shape[1]
            num_classes = logits.shape[2]
            wta_result = torch.zeros((grid_size * grid_size, bs, num_classes), device=logits.device)  
            local_logits = logits.reshape(grid_size, grid_size, bs, num_classes)  # [[14,14,100,80]
            flat_logits = local_logits.view(-1, *local_logits.shape[-2:])  # Shape [196, 100, 80]
            max_values, _ = flat_logits.max(dim=-1, keepdim=True)  # Shape [196, 100, 1]
            wta_result = torch.where(flat_logits == max_values, flat_logits, torch.full_like(flat_logits, -1.0))
            ## strong WTA
            result = wta_result.permute(1, 0, 2)  # [100, 196 ,80]
            mask = result != -1
            result_masked = torch.where(mask, result, torch.full_like(result, float('-inf')))
            strong_wta, _ = torch.max(result_masked, dim=1)  # [100, 80]
            strong_wta[strong_wta == float('-inf')] = -1.0
            logits = strong_wta
            
        #### WTA end #########

        return logits_, logits, None, None

    @torch.no_grad()
    def test(self, split=None):
        """A generic testing pipeline."""
        self.set_model_mode("eval")
        self.evaluator.reset()

        if split is None:
            split = self.cfg.TEST.SPLIT

        if split == "val" and self.val_loader is not None:
            data_loader = self.val_loader
            print("Do evaluation on {} set".format(split))
        else:
            data_loader = self.test_loader
            print("Do evaluation on test set")
            
            
        time_queue = []
        warmup_phase = 1000

        for batch_idx, batch in enumerate(tqdm(data_loader)):
            input, label = self.parse_batch_test(batch)
            # output = self.model_inference(input)

            if MEASURE_TIME:
                st = time.time()

            output, output_pos, image_features_, text_features_ = self.model_inference(input, label)

            if MEASURE_TIME:
                ed = time.time()
                elapsed  = ed - st
                if batch_idx > warmup_phase:
                    time_queue.append(elapsed)
                    print (np.mean(time_queue))
                

            self.evaluator.process(output, label, output_pos)

        results = self.evaluator.evaluate()

        for k, v in results.items():
            tag = "{}/{}".format(split, k)
            self.write_scalar(tag, v, self.epoch)

        return list(results.values())[0]


@TRAINER_REGISTRY.register()
class ZeroshotCLIP2(ZeroshotCLIP):
    """Prompt ensembling."""

    # templates = IMAGENET_TEMPLATES
    templates = IMAGENET_TEMPLATES_SELECT

    def build_model(self):
        cfg = self.cfg
        classnames = self.dm.dataset.classnames

        print(f"Loading CLIP (backbone: {cfg.MODEL.BACKBONE.NAME})")
        clip_model = load_clip_to_cpu(cfg)
        clip_model.to(self.device)

        for params in clip_model.parameters():
            params.requires_grad_(False)

        # add custom-made prompt
        if cfg.DATASET.NAME != "ImageNet":
            self.templates += [CUSTOM_TEMPLATES[cfg.DATASET.NAME]]

        num_temp = len(self.templates)
        print(f"Prompt ensembling (n={num_temp})")

        mean_text_features = 0
        for i, temp in enumerate(self.templates):
            prompts = [temp.format(c.replace("_", " ")) for c in classnames]
            prompts = torch.cat([clip.tokenize(p) for p in prompts]).to(self.device)
            text_features = clip_model.encode_text(prompts)
            text_features = text_features / text_features.norm(dim=-1, keepdim=True)
            mean_text_features = mean_text_features + text_features
        mean_text_features = mean_text_features / num_temp
        mean_text_features = mean_text_features / mean_text_features.norm(dim=-1, keepdim=True)

        self.text_features = mean_text_features
        self.clip_model = clip_model
