import torch
import torch.nn as nn
from torch.nn import functional as F
from torchvision.models._utils import IntermediateLayerGetter

from dassl.engine import TRAINER_REGISTRY, TrainerX
from dassl.optim import build_optimizer, build_lr_scheduler

from clip import clip
from clip.model import convert_weights

from .coop import load_clip_to_cpu
from .imagenet_templates import IMAGENET_TEMPLATES, IMAGENET_TEMPLATES_SELECT
from tqdm import tqdm
import numpy as np
import math
CUSTOM_TEMPLATES = {
    "OxfordPets": "a photo of a {}, a type of pet.",
    "OxfordFlowers": "a photo of a {}, a type of flower.",
    "FGVCAircraft": "a photo of a {}, a type of aircraft.",
    "DescribableTextures": "{} texture.",
    "EuroSAT": "a centered satellite photo of {}.",
    "StanfordCars": "a photo of a {}.",
    "Food101": "a photo of {}, a type of food.",
    "SUN397": "a photo of a {}.",
    "Caltech101": "a photo of a {}.",
    "UCF101": "a photo of a person doing {}.",
    "ImageNet": "a photo of a {}.",
    "ImageNetSketch": "a photo of a {}.",
    "ImageNetV2": "a photo of a {}.",
    "ImageNetA": "a photo of a {}.",
    "ImageNetR": "a photo of a {}."
}
label_names = [
    "person", "bicycle", "car", "motorcycle", "airplane", "bus", "train", "truck",
    "boat", "traffic light", "fire hydrant", "stop sign", "parking meter", "bench",
    "bird", "cat", "dog", "horse", "sheep", "cow", "elephant", "bear", "zebra", "giraffe",
    "backpack", "umbrella", "handbag", "tie", "suitcase", "frisbee", "skis", "snowboard",
    "sports ball", "kite", "baseball bat", "baseball glove", "skateboard", "surfboard",
    "tennis racket", "bottle", "wine glass", "cup", "fork", "knife", "spoon", "bowl",
    "banana", "apple", "sandwich", "orange", "broccoli", "carrot", "hot dog", "pizza",
    "donut", "cake", "chair", "couch", "potted plant", "bed", "dining table", "toilet",
    "tv", "laptop", "mouse", "remote", "keyboard", "cell phone", "microwave", "oven",
    "toaster", "sink", "refrigerator", "book", "clock", "vase", "scissors", "teddy bear",
    "hair drier", "toothbrush"
]

WTA = False
LOCAL = True
GLOBAL = False
RESNET101 = True
DATASET = "NUSWIDE"

@TRAINER_REGISTRY.register()
class ZeroshotCLIP(TrainerX):
    def build_model(self):
        cfg = self.cfg
        classnames = self.dm.dataset.classnames

        print(f"Loading CLIP (backbone: {cfg.MODEL.BACKBONE.NAME})")
        clip_model = load_clip_to_cpu(cfg)
        clip_model.to(self.device)

        try:
            temp = CUSTOM_TEMPLATES[cfg.DATASET.NAME]
        except:
            print('!! WARNING: Not found template for {}'.format(cfg.DATASET.NAME))
            temp = "a photo of a {}."

        prompts = [temp.format(c.replace("_", " ")) for c in classnames]
        print(f"Prompts: {prompts}")
        prompts = torch.cat([clip.tokenize(p) for p in prompts])
        prompts = prompts.to(self.device)

        with torch.no_grad():
            text_features = clip_model.encode_text(prompts)
            text_features = text_features / text_features.norm(dim=-1, keepdim=True)

        self.text_features = text_features
        self.clip_model = clip_model

    def model_inference(self, image):
        image_features = self.clip_model.encode_image(image)
        image_features = image_features / image_features.norm(dim=-1, keepdim=True)
        logit_scale = self.clip_model.logit_scale.exp()
        logits = logit_scale * image_features @ self.text_features.t()

        return logits, None, None


def read_npy(numpy_file, device):
    all_features = np.load(numpy_file, allow_pickle=True)

    # for ii, l_n in enumerate(label_names):
    #     temp = torch.tensor(all_features[ii])
    #     if ii == 0:
    #         text_caption = temp[:16].unsqueeze(0)
    #     else:
    #         text_caption = torch.cat((text_caption, temp[:16].unsqueeze(0)),0) # 80 16 512

    print('load complete:', numpy_file)
    return torch.tensor(all_features).to(torch.float16).to(
        device)  # text_caption.to(device) #torch.tensor(all_features).to(device)


@TRAINER_REGISTRY.register()
class ZeroshotCLIP_dense(TrainerX):
    def build_model(self):
        cfg = self.cfg
        classnames = self.dm.dataset.classnames

        print(f"Loading CLIP (backbone: {cfg.MODEL.BACKBONE.NAME})")
        clip_model = load_clip_to_cpu(cfg)
        clip_model.to(self.device)

        try:
            temp = CUSTOM_TEMPLATES[cfg.DATASET.NAME]
        except:
            print('!! WARNING: Not found template for {}'.format(cfg.DATASET.NAME))
            temp = "a photo of a {}."

        prompts = [temp.format(c.replace("_", " ")) for c in classnames]
        print(f"Prompts: {prompts}")
        prompts = torch.cat([clip.tokenize(p) for p in prompts])
        prompts = prompts.to(self.device)

        with torch.no_grad():
            text_features = clip_model.encode_text(prompts)
            text_features = text_features / text_features.norm(dim=-1, keepdim=True)

        self.text_features = text_features
        self.clip_model = clip_model

        self.visual_encoder = IntermediateLayerGetter(self.clip_model.visual, {"layer4": "0"})
        self.positional_embedding = self.clip_model.visual.attnpool.positional_embedding[1::]
        self.v_linear_weight = self.clip_model.visual.attnpool.v_proj.weight
        self.v_linear_bias = self.clip_model.visual.attnpool.v_proj.bias
        self.c_linear_weight = self.clip_model.visual.attnpool.c_proj.weight
        self.c_linear_bias = self.clip_model.visual.attnpool.c_proj.bias
        
        if DATASET == "COCO":
            if RESNET101:
                self.caption = read_npy('COCO_caption_selection_101.npy', self.device)
                self.caption = self.caption / self.caption.norm(dim=-1, keepdim=True)

                self.caption_local = read_npy('COCO_caption_selection_local_101.npy', self.device)
                self.caption_local = self.caption_local / self.caption_local.norm(dim=-1, keepdim=True)

                self._multi_label_caption = read_npy('COCO_multi_label_caption_selection_101_v3.npy', self.device)
                self._multi_label_caption = self._multi_label_caption / self._multi_label_caption.norm(dim=-1, keepdim=True)
            else:
                self.caption = read_npy('COCO_caption_selection.npy', self.device)
                self.caption = self.caption / self.caption.norm(dim=-1, keepdim=True)

                self.caption_local = read_npy('COCO_caption_selection_local.npy', self.device)
                self.caption_local = self.caption_local / self.caption_local.norm(dim=-1, keepdim=True)

                self._multi_label_caption = read_npy('COCO_multi_label_caption_selection_50_v3.npy', self.device)
                self._multi_label_caption = self._multi_label_caption / self._multi_label_caption.norm(dim=-1, keepdim=True)
                
        elif DATASET == "NUSWIDE":
            if RESNET101:
                self.caption = read_npy('nus_caption_selection_101.npy', self.device)
                self.caption = self.caption / self.caption.norm(dim=-1, keepdim=True)

                self._multi_label_caption = read_npy('nus_multi_label_caption_selection_101_v3_7000.npy', self.device)
                self._multi_label_caption = self._multi_label_caption / self._multi_label_caption.norm(dim=-1, keepdim=True)
            else:
                self.caption = read_npy('nus_caption_selection.npy', self.device)
                self.caption = self.caption / self.caption.norm(dim=-1, keepdim=True)

                self._multi_label_caption = read_npy('nus_multi_label_caption_selection_50_v3.npy', self.device)
                self._multi_label_caption = self._multi_label_caption / self._multi_label_caption.norm(dim=-1, keepdim=True)
                
                
        elif DATASET == "VOC":
            if RESNET101:
                self.caption = read_npy('VOC_caption_selection_101.npy', self.device)
                self.caption = self.caption / self.caption.norm(dim=-1, keepdim=True)

                self._multi_label_caption = read_npy('VOC_multi_label_caption_selection_101_v3.npy', self.device)
                self._multi_label_caption = self._multi_label_caption / self._multi_label_caption.norm(dim=-1, keepdim=True)
            else:
                self.caption = read_npy('VOC_caption_selection.npy', self.device)
                self.caption = self.caption / self.caption.norm(dim=-1, keepdim=True)

                self._multi_label_caption = read_npy('VOC_multi_label_caption_selection_50_v3.npy', self.device)
                self._multi_label_caption = self._multi_label_caption / self._multi_label_caption.norm(dim=-1, keepdim=True)
                

    def encode_image(self, x):
        def stem(x):
            for conv, bn in [(self.visual_encoder.conv1, self.visual_encoder.bn1), \
                             (self.visual_encoder.conv2, self.visual_encoder.bn2),
                             (self.visual_encoder.conv3, self.visual_encoder.bn3)]:
                x = self.visual_encoder.relu(bn(conv(x)))
            x = self.visual_encoder.avgpool(x)
            return x

        x = x.type(self.visual_encoder.conv1.weight.dtype)
        x = stem(x)
        x = self.visual_encoder.layer1(x)
        x = self.visual_encoder.layer2(x)
        x = self.visual_encoder.layer3(x)
        x = self.visual_encoder.layer4(x)
        return x

    def model_inference(self, image, label):
        image_feat = self.encode_image(image)
        b, c, h, w = image_feat.shape
        x = image_feat.reshape(b, c, h * w).permute(2, 0, 1)

        x = F.linear(x, self.v_linear_weight, self.v_linear_bias)
        x = F.linear(x, self.c_linear_weight, self.c_linear_bias)
        image_features = x

        image_feature_, _ = self.clip_model.visual.attnpool(image_feat)
        image_features = image_features / image_features.norm(dim=-1, keepdim=True)
        image_feature_ = image_feature_ / image_feature_.norm(dim=-1, keepdim=True)
        
        
        """

        def get_global_p(caption):  # [80,16,512]
            #             cls_num,cap_num , C = caption.shape
            #             caption = caption.reshape(cls_num*cap_num,C)

            #             indice =  torch.randperm(cls_num*cap_num)
            #             caption = caption[indice]
            #             caption = caption.reshape(cls_num, cap_num,C)
            # caption = caption[:50,:,:]

            A = torch.matmul(caption, image_feature_.t())  # [80 16 512] [512 100] => [80 16 100]
            A_up = F.softmax(A * 80, dim=1)  # [80 16 100]
            # print('A_up',A_up)
            A_up_transposed = A_up.transpose(1, 2)  # [80, 100, 16]
            p_up = torch.bmm(A_up_transposed, caption)  # [80, 100, 16] [80, 16, 512] => [80,100,512]
            p_up = p_up.permute(1, 0, 2)  # [100,80,512]
            # p_up finished

            image_feature_expanded = image_feature_.unsqueeze(1)  # [100, 1, 512]
            p_up_sim = torch.bmm(image_feature_expanded, p_up.permute(0, 2, 1))  # [100,1,512] [100,512,80] => [100,80]
            p_up_sim = p_up_sim.squeeze(1)  # [100, 80]
            prob_p_up = torch.nn.functional.softmax(p_up_sim * 80, dim=1)  # [100,80]
            # print('prob_p_up',prob_p_up)
            p_up_sim_expanded = prob_p_up.unsqueeze(-1)
            p_up_mul = p_up * p_up_sim_expanded  # [100,80,512]  [100,80,1] => [100,80,512]
            global_p = torch.sum(p_up_mul, dim=1)  # [100,80,512] => [100,512]
            # global_p finished

            return global_p

        global_p = get_global_p(self._multi_label_caption)

        image_feature_temp = global_p  # (0.5*image_feature_ + 0.5* global_p) / 1.0
        caption_logit = torch.matmul(self.caption, image_feature_temp.t())
        A = torch.matmul(self.caption, image_feature_temp.t())  # 80 16 100
        A_up = F.softmax(A, dim=1)
        """
        caption_logit = torch.matmul(self.caption,image_feature_.t())
        A = torch.matmul(self.caption, image_feature_.t()) #80 16 100
        A_up = F.softmax(A, dim=1) #80 16 100
        P_up = torch.matmul(A_up.permute(0, 2, 1), self.caption)  # 80 100 512


        caption_logit_ = torch.einsum('hbd,dcn->hbcn', self.caption, image_features.permute(2, 0, 1))
        caption_logit_ = caption_logit_.permute(2, 3, 0, 1)

        A_ = caption_logit_  # torch.matmul(self.caption, image_feature_.t()) #80 16 100
        A_up_ = F.softmax(A_ * 5, dim=3)

        logit_scale = self.clip_model.logit_scale.exp()  # logit_scale = self.clip_model.logit_scale.exp()

        # local
        
        if LOCAL :
            logits = logit_scale * (torch.sum(A_up_ * caption_logit_, dim=3) + image_features @ self.text_features.t()) / 2.0
        else:
            logits = logit_scale * image_features @ self.text_features.t()

        # global
        
        if GLOBAL:
            logits_ = logit_scale * (torch.sum(P_up * image_feature_.unsqueeze(0), dim=2).permute(1,0) + image_feature_ @ self.text_features.t()) / 2.0
        else:
            logits_ = logit_scale * image_feature_ @ self.text_features.t()


        a = (label[0, :] == 1).nonzero(as_tuple=False)

    

        # Original local logits [14X14,100,80]
        if WTA==False:
            prob_spatial = torch.nn.functional.softmax(logits, dim=0)
            logits = torch.sum(logits * prob_spatial, dim=0)
        # WTA
        else: 
            grid_size = int(math.sqrt(logits.shape[0]))
            bs = logits.shape[1]
            num_classes = logits.shape[2]
            wta_result = torch.zeros((grid_size * grid_size, bs, num_classes), device=logits.device)  # 모든 값을 0으로 초기화
            local_logits = logits.reshape(grid_size,grid_size,bs,num_classes) #[[14,14,100,80]

            # Flatten the first two dimensions of local_logits so that we can work with each patch individually
            # Now each patch is a separate element in the zeroth dimension
            flat_logits = local_logits.view(-1, *local_logits.shape[-2:])  # Shape [196, 100, 80]

            # Find the max values in each patch
            max_values, _ = flat_logits.max(dim=-1, keepdim=True)  # Shape [196, 100, 1]

            # Use torch.where to keep max values and set others to -1
            wta_result = torch.where(flat_logits == max_values, flat_logits, torch.full_like(flat_logits, -1.0))

            ## strong WTA

            result = wta_result.permute(1,0,2)  # [100, 196 ,80]

            # Create a mask for values that are not -1
            mask = result != -1

            # Use the mask to replace all -1 with the minimum float value
            # because -1 could be a valid maximum when dealing with negative values
            result_masked = torch.where(mask, result, torch.full_like(result, float('-inf')))

            # Now, instead of looping, use torch.max to find the maximum value across the grid
            # This will ignore the -inf values and effectively find the maximum non--1 value
            strong_wta, _ = torch.max(result_masked, dim=1)  # [100, 80]

            # Replace the -inf back to -1 if there was no valid maximum found
            strong_wta[strong_wta == float('-inf')] = -1.0


            #print (logits_[0].topk(k=5)[1])
            #print (origin_local_logits[0].topk(k=5)[1])
            #print (strong_wta[0].topk(k=5)[1])

            # 결과를 logits에 저장
            logits = strong_wta


        return logits_, logits, None, None

    @torch.no_grad()
    def test(self, split=None):
        """A generic testing pipeline."""
        self.set_model_mode("eval")
        self.evaluator.reset()

        if split is None:
            split = self.cfg.TEST.SPLIT

        if split == "val" and self.val_loader is not None:
            data_loader = self.val_loader
            print("Do evaluation on {} set".format(split))
        else:
            data_loader = self.test_loader
            print("Do evaluation on test set")

        for batch_idx, batch in enumerate(tqdm(data_loader)):
            input, label = self.parse_batch_test(batch)
            # output = self.model_inference(input)
            output, output_pos, image_features_, text_features_ = self.model_inference(input, label)
            self.evaluator.process(output, label, output_pos)

        results = self.evaluator.evaluate()

        for k, v in results.items():
            tag = "{}/{}".format(split, k)
            self.write_scalar(tag, v, self.epoch)

        return list(results.values())[0]


@TRAINER_REGISTRY.register()
class ZeroshotCLIP2(ZeroshotCLIP):
    """Prompt ensembling."""

    # templates = IMAGENET_TEMPLATES
    templates = IMAGENET_TEMPLATES_SELECT

    def build_model(self):
        cfg = self.cfg
        classnames = self.dm.dataset.classnames

        print(f"Loading CLIP (backbone: {cfg.MODEL.BACKBONE.NAME})")
        clip_model = load_clip_to_cpu(cfg)
        clip_model.to(self.device)

        for params in clip_model.parameters():
            params.requires_grad_(False)

        # add custom-made prompt
        if cfg.DATASET.NAME != "ImageNet":
            self.templates += [CUSTOM_TEMPLATES[cfg.DATASET.NAME]]

        num_temp = len(self.templates)
        print(f"Prompt ensembling (n={num_temp})")

        mean_text_features = 0
        for i, temp in enumerate(self.templates):
            prompts = [temp.format(c.replace("_", " ")) for c in classnames]
            prompts = torch.cat([clip.tokenize(p) for p in prompts]).to(self.device)
            text_features = clip_model.encode_text(prompts)
            text_features = text_features / text_features.norm(dim=-1, keepdim=True)
            mean_text_features = mean_text_features + text_features
        mean_text_features = mean_text_features / num_temp
        mean_text_features = mean_text_features / mean_text_features.norm(dim=-1, keepdim=True)

        self.text_features = mean_text_features
        self.clip_model = clip_model
