import torch
import torch.nn as nn
from torch.nn import functional as F
from torchvision.models._utils import IntermediateLayerGetter

from dassl.engine import TRAINER_REGISTRY, TrainerX
from dassl.optim import build_optimizer, build_lr_scheduler

from clip import clip
from clip.model import convert_weights

from .coop import load_clip_to_cpu
from .imagenet_templates import IMAGENET_TEMPLATES, IMAGENET_TEMPLATES_SELECT
from tqdm import tqdm
import numpy as np

CUSTOM_TEMPLATES = {
    "OxfordPets": "a photo of a {}, a type of pet.",
    "OxfordFlowers": "a photo of a {}, a type of flower.",
    "FGVCAircraft": "a photo of a {}, a type of aircraft.",
    "DescribableTextures": "{} texture.",
    "EuroSAT": "a centered satellite photo of {}.",
    "StanfordCars": "a photo of a {}.",
    "Food101": "a photo of {}, a type of food.",
    "SUN397": "a photo of a {}.",
    "Caltech101": "a photo of a {}.",
    "UCF101": "a photo of a person doing {}.",
    "ImageNet": "a photo of a {}.",
    "ImageNetSketch": "a photo of a {}.",
    "ImageNetV2": "a photo of a {}.",
    "ImageNetA": "a photo of a {}.",
    "ImageNetR": "a photo of a {}."
}
label_names = [
        "person", "bicycle", "car", "motorcycle", "airplane", "bus", "train", "truck",
        "boat", "traffic light", "fire hydrant", "stop sign", "parking meter", "bench",
        "bird", "cat", "dog", "horse", "sheep", "cow", "elephant", "bear", "zebra", "giraffe",
        "backpack", "umbrella", "handbag", "tie", "suitcase", "frisbee", "skis", "snowboard",
        "sports ball", "kite", "baseball bat", "baseball glove", "skateboard", "surfboard",
        "tennis racket", "bottle", "wine glass", "cup", "fork", "knife", "spoon", "bowl",
        "banana", "apple", "sandwich", "orange", "broccoli", "carrot", "hot dog", "pizza",
        "donut", "cake", "chair", "couch", "potted plant", "bed", "dining table", "toilet",
        "tv", "laptop", "mouse", "remote", "keyboard", "cell phone", "microwave", "oven",
        "toaster", "sink", "refrigerator", "book", "clock", "vase", "scissors", "teddy bear",
        "hair drier", "toothbrush"
    ]

@TRAINER_REGISTRY.register()
class ZeroshotCLIP(TrainerX):
    def build_model(self):
        cfg = self.cfg
        classnames = self.dm.dataset.classnames

        print(f"Loading CLIP (backbone: {cfg.MODEL.BACKBONE.NAME})")
        clip_model = load_clip_to_cpu(cfg)
        clip_model.to(self.device)

        try:
            temp = CUSTOM_TEMPLATES[cfg.DATASET.NAME]
        except:
            print('!! WARNING: Not found template for {}'.format(cfg.DATASET.NAME))
            temp = "a photo of a {}."

        prompts = [temp.format(c.replace("_", " ")) for c in classnames]
        print(f"Prompts: {prompts}")
        prompts = torch.cat([clip.tokenize(p) for p in prompts])
        prompts = prompts.to(self.device)

        with torch.no_grad():
            text_features = clip_model.encode_text(prompts)
            text_features = text_features / text_features.norm(dim=-1, keepdim=True)

        self.text_features = text_features
        self.clip_model = clip_model

    def model_inference(self, image):
        image_features = self.clip_model.encode_image(image)
        image_features = image_features / image_features.norm(dim=-1, keepdim=True)
        logit_scale = self.clip_model.logit_scale.exp()
        logits = logit_scale * image_features @ self.text_features.t()

        return logits, None, None
    
def read_npy(numpy_file, device):
    all_features = np.load(numpy_file,allow_pickle=True)
    
    
    # for ii, l_n in enumerate(label_names):
    #     temp = torch.tensor(all_features[ii])
    #     if ii == 0:
    #         text_caption = temp[:16].unsqueeze(0)
    #     else:
    #         text_caption = torch.cat((text_caption, temp[:16].unsqueeze(0)),0) # 80 16 512

    
    
    print('load complete:', numpy_file)
    return torch.tensor(all_features).to(torch.float16).to(device) #text_caption.to(device) #torch.tensor(all_features).to(device)

@TRAINER_REGISTRY.register()
class ZeroshotCLIP_dense(TrainerX):
    def build_model(self):
        cfg = self.cfg
        classnames = self.dm.dataset.classnames

        print(f"Loading CLIP (backbone: {cfg.MODEL.BACKBONE.NAME})")
        clip_model = load_clip_to_cpu(cfg)
        clip_model.to(self.device)
        
        try:
            temp = CUSTOM_TEMPLATES[cfg.DATASET.NAME]
        except:
            print('!! WARNING: Not found template for {}'.format(cfg.DATASET.NAME))
            temp = "a photo of a {}."

        prompts = [temp.format(c.replace("_", " ")) for c in classnames]
        print(f"Prompts: {prompts}")
        prompts = torch.cat([clip.tokenize(p) for p in prompts])
        prompts = prompts.to(self.device)

        with torch.no_grad():
            text_features = clip_model.encode_text(prompts)
            text_features = text_features / text_features.norm(dim=-1, keepdim=True)

        self.text_features = text_features
        self.clip_model = clip_model

        self.visual_encoder = IntermediateLayerGetter(self.clip_model.visual, {"layer4": "0"})
        self.positional_embedding = self.clip_model.visual.attnpool.positional_embedding[1::]
        self.v_linear_weight = self.clip_model.visual.attnpool.v_proj.weight
        self.v_linear_bias = self.clip_model.visual.attnpool.v_proj.bias
        self.c_linear_weight = self.clip_model.visual.attnpool.c_proj.weight
        self.c_linear_bias = self.clip_model.visual.attnpool.c_proj.bias
        
        self.caption = read_npy('COCO_caption_selection_101.npy', self.device)
        self.caption = self.caption / self.caption.norm(dim=-1, keepdim=True)
        
        self.caption_local = read_npy('COCO_caption_selection_local_101.npy', self.device)
        self.caption_local = self.caption_local / self.caption_local.norm(dim=-1, keepdim=True)
        
        self._multi_label_caption = read_npy('COCO_multi_label_caption_selection_101.npy', self.device)
        self._multi_label_caption = self._multi_label_caption / self._multi_label_caption.norm(dim=-1, keepdim=True)
    def encode_image(self, x):
        def stem(x):
            for conv, bn in [(self.visual_encoder.conv1, self.visual_encoder.bn1), \
                (self.visual_encoder.conv2, self.visual_encoder.bn2), (self.visual_encoder.conv3, self.visual_encoder.bn3)]:
                x = self.visual_encoder.relu(bn(conv(x)))
            x = self.visual_encoder.avgpool(x)
            return x

        x = x.type(self.visual_encoder.conv1.weight.dtype)
        x = stem(x)
        x = self.visual_encoder.layer1(x)
        x = self.visual_encoder.layer2(x)
        x = self.visual_encoder.layer3(x)
        x = self.visual_encoder.layer4(x)
        return x

    def model_inference(self, image, label):
        image_feat = self.encode_image(image)
        b, c, h, w = image_feat.shape
        x = image_feat.reshape(b, c, h * w).permute(2, 0, 1)

        x = F.linear(x, self.v_linear_weight, self.v_linear_bias)
        x = F.linear(x, self.c_linear_weight, self.c_linear_bias)
        image_features = x
        
        image_feature_, _ = self.clip_model.visual.attnpool(image_feat)
        
        
        
        def get_global_p(caption): # [80,16,512]
            A = torch.matmul(caption, image_feature_.t())  # [80 16 512] [512 100] => [80 16 100]
            A_up = F.softmax(A*5, dim=1) #[80 16 100]
            A_up_transposed = A_up.transpose(1, 2)  # [80, 100, 16]
            p_up = torch.bmm(A_up_transposed, caption)  #[80, 100, 16] [80, 16, 512] => [80,100,512]
            p_up = p_up.permute(1,0,2) # [100,80,512]
            # p_up finished

            image_feature_expanded = image_feature_.unsqueeze(1)  # [100, 1, 512]
            p_up_sim = torch.bmm(image_feature_expanded, p_up.permute(0,2, 1)) #[100,1,512] [100,512,80] => [100,80]
            p_up_sim = p_up_sim.squeeze(1)  # [100, 80]
            prob_p_up = torch.nn.functional.softmax(p_up_sim/0.5, dim=1) # [100,80]
            p_up_sim_expanded = prob_p_up.unsqueeze(-1)
            p_up_mul = p_up * p_up_sim_expanded #[100,80,512]  [100,80,1] => [100,80,512]
            global_p = torch.sum(p_up_mul, dim=1)  # [100,80,512] => [100,512]
            # global_p finished

            return global_p
        
        
      #  print('lable',label.shape)
    
       # print(F.softmax(100*torch.matmul(self.text_features, self.text_features.t())[0,1:]))
        
        
        image_features = image_features / image_features.norm(dim=-1, keepdim=True)
        image_feature_ = image_feature_ / image_feature_.norm(dim=-1, keepdim=True)
        
        
        
        
        
        #image guiding caption update
        #C= softmax(CV)V + C
        # C : 80 16 512
        # V : 100 1024
        # P : 80 1024
        
       # print('self.caption',self.caption.shape)
       # print('image_feature',image_feature_.shape)
        
#         A_cap = torch.matmul(self.caption, image_feature_.t()) # 80 16 100
#         A_cap_up = F.softmax(A_cap, dim = 2) # 80 16 100
        
#         A_cap = self.caption.permute(0,2,1) * self.text_features.unsqueeze(2) # 16 80 100
#         A_cap_up = F.softmax(A_cap, dim = 2) # 80 16 100
        
#       #  alpha = 0.9
#         #self.caption  = ((1-alpha) * torch.matmul(A_cap_up , image_feature_) + alpha * self.caption ) 
#         self.caption  = (  torch.sum(A_cap_up * self.text_features.unsqueeze(2), dim =2) + self.caption ) / 2.0
        
        
      #  A_P = F.softmax(self.caption.permute(1,0,2) * self.text_features.unsqueeze(0), dim = 0) #16 80 512
      #  A_P_up = torch.sum(A_P * self.caption.permute(1,0,2), dim = 0).squeeze() #80 512
        
        
        #global
        
#         A = torch.matmul(self.caption, image_feature_.t()) #80 16 100
#         A_up = F.softmax(A, dim=1)
        
       
        global_p = get_global_p(self._multi_label_caption)
        
      #  image_feature_ = global_p
        
        image_feature_temp = global_p #(0.5*image_feature_ + 0.5* global_p) / 1.0
        
        caption_logit = torch.matmul(self.caption,image_feature_temp.t())
        A = torch.matmul(self.caption, image_feature_temp.t()) #80 16 100
        A_up = F.softmax(A, dim=1)
        
        # caption_logit = torch.matmul(self.caption,image_feature_.t())
        # A = torch.matmul(self.caption, image_feature_.t()) #80 16 100
        # A_up = F.softmax(A, dim=1) #80 16 100
        
        
        P_up = torch.matmul(A_up.permute(0,2,1), self.caption) #80 100 512
        
        
        
#         _, index = torch.topk(A_up,1,dim=1)
        
#         cls_num , cap_num, _ = A_up.shape
#         B, _ = image_feature_.shape
#         P_up_ = torch.zeros_like(P_up)
        
#         for i in range(B):
#             for j in range(cls_num):
#                 P_up_[j,i,:] = P_up[j,i,:] + (self.caption[j,index[j,0,i]] - P_up[j,i,:]) * 2.0
        
        
        P_up = P_up / P_up.norm(dim=-1, keepdim=True)
        
        
        
        
        # caption_logit_g = torch.matmul(self.caption,global_p.t())
        # A_g = torch.matmul(self.caption, global_p.t()) #80 16 100
        # A_up_g = F.softmax(A_g, dim=1)
        
        
        
#         A_P_up = F.softmax(A, dim=1) * 1
        
#         A_P_up1 = F.softmax(A*0.1, dim=1) 
        
        
        #local
        caption_logit_ = torch.einsum('hbd,dcn->hbcn',self.caption, image_features.permute(2,0,1))
        caption_logit_ = caption_logit_.permute(2,3,0,1)
        
        A_ = caption_logit_ #torch.matmul(self.caption, image_feature_.t()) #80 16 100
        A_up_ = F.softmax(A_, dim=3)
        
        logit_scale = self.clip_model.logit_scale.exp() # logit_scale = self.clip_model.logit_scale.exp()
       # logits_ = logit_scale * image_feature_ @ self.text_features.t()   # B * C,  cls * C, = B * cls
        #logits = logit_scale * image_features @ self.text_features.t()    #  HW * B * C,  cls * C,  HW * B * cls
        
        #image_features 49 100 512 x 512 80 -> 49 100 80
        
        # 100 49  512 x  100 80  512 ->100 49 80
        
        # 80 16 512 / 512 49 100 -> 80 16 49 100 
        
       # print(image_features.shape)
       # print(P_up.shape)
       # print(torch.matmul(image_features.permute(1,0,2), P_up.permute(1,2,0)).shape)
        #local
        logits = logit_scale *   (torch.sum(A_up_ * caption_logit_, dim = 3) + image_features @ self.text_features.t()  ) / 2.0
     #   logits = logit_scale *   (torch.sum(A_up_ * caption_logit_, dim = 3) + torch.matmul(image_features.permute(1,0,2), P_up.permute(1,2,0)).permute(1,0,2)+ image_features @ self.text_features.t()  ) / 3.0 #caption_logit_.mean(-1)
        #global
      #  logits_ = logit_scale * ( torch.sum(A_up * caption_logit, dim = 1).permute(1,0)  +  image_feature_ @ self.text_features.t() )/2.0
        
      #  logits_ = logit_scale * ( torch.sum(P_up * image_feature_.unsqueeze(0), dim =2).permute(1,0) +  image_feature_ @ self.text_features.t() )/2.0
    
        logits_ = logit_scale * ( torch.sum(P_up * image_feature_.unsqueeze(0), dim =2).permute(1,0) +  image_feature_temp @ self.text_features.t() +  image_feature_ @ self.text_features.t() )/3.0
    
    
        #logits_ = logit_scale * (torch.sum(A_up * caption_logit, dim = 1).permute(1,0)  + torch.matmul(A_P_up , image_feature_.t()).permute(1,0)+ image_feature_ @ self.text_features.t())/3.0 
       # logits_ = logit_scale * (torch.sum(A_up * caption_logit, dim = 1).permute(1,0) + torch.sum(A_P_up * caption_logit, dim = 1).permute(1,0)   + torch.sum(A_P_up1 * caption_logit, dim = 1).permute(1,0)  + image_feature_ @ self.text_features.t())/4.0   
        
        
        # caption_logit = torch.matmul(self.caption,image_feature_.t())
        # 80 16 512 / 49 100 512
        
        
        
        # 80 16 100 49 ->  49 100 80 16
        
        # 49 100 80 16 
      #  print (caption_logit.shape)
        
        
        
        
         #[100, 1024] [80 16 1024]
        
        
        #caption_logit = torch.matmul(A_up.permute(0,2,1), self.caption) # 80 100 1024
        
        
        
        
        
#         A = torch.matmul(image_feature_[i,:], self.text_features.permute(1,0)) *5  # 49 80
#             A_up = F.softmax(A, dim=0) # 49 80
#             temp_text_feats = torch.matmul(A_up.t(), image_feature_[i,:]) # 80 1024
        
        
        
        a = (label[0,:]==1).nonzero(as_tuple=False)
      #  print(a)
        
        # for a_ in a:
        #     print (label_names[a_])
        
        # for ii, l_n in enumerate(label_names):
        #     if ii in torch.where(label==1):
        #         print(l_n)
                
        # for ii, l_n in enumerate(label_names):
        #     print(l_n,caption_logit[ii].mean())
        
     #   print(caption_logit.shape)

           
        
        # [100, 1,  1024]  80 100 1024 => 100 80 1024 / 100 1 80
      #  logits_ = logit_scale * (torch.matmul( image_feature_.unsqueeze(1), caption_logit.permute(1, 2 ,0)).squeeze()  + image_feature_ @ self.text_features.t())/2.0
    
    
    
      #  \caption_logit.mean(1).permute(1,0)
      # logits_ =  logit_scale * (caption_logit.mean(1).permute(1,0) + image_feature_ @ self.text_features.t())/2.0
        
        
        
        
        prob_spatial = torch.nn.functional.softmax(logits, dim=0)
        logits = torch.sum(logits * prob_spatial, dim=0)

        return logits_, logits, None, None

    @torch.no_grad()
    def test(self, split=None):
        """A generic testing pipeline."""
        self.set_model_mode("eval")
        self.evaluator.reset()

        if split is None:
            split = self.cfg.TEST.SPLIT

        if split == "val" and self.val_loader is not None:
            data_loader = self.val_loader
            print("Do evaluation on {} set".format(split))
        else:
            data_loader = self.test_loader
            print("Do evaluation on test set")

        for batch_idx, batch in enumerate(tqdm(data_loader)):
            input, label = self.parse_batch_test(batch)
            # output = self.model_inference(input)
            output, output_pos, image_features_, text_features_ = self.model_inference(input, label)
            self.evaluator.process(output, label, output_pos)

        results = self.evaluator.evaluate()

        for k, v in results.items():
            tag = "{}/{}".format(split, k)
            self.write_scalar(tag, v, self.epoch)

        return list(results.values())[0]


@TRAINER_REGISTRY.register()
class ZeroshotCLIP2(ZeroshotCLIP):
    """Prompt ensembling."""

    # templates = IMAGENET_TEMPLATES
    templates = IMAGENET_TEMPLATES_SELECT

    def build_model(self):
        cfg = self.cfg
        classnames = self.dm.dataset.classnames

        print(f"Loading CLIP (backbone: {cfg.MODEL.BACKBONE.NAME})")
        clip_model = load_clip_to_cpu(cfg)
        clip_model.to(self.device)

        for params in clip_model.parameters():
            params.requires_grad_(False)

        # add custom-made prompt
        if cfg.DATASET.NAME != "ImageNet":
            self.templates += [CUSTOM_TEMPLATES[cfg.DATASET.NAME]]

        num_temp = len(self.templates)
        print(f"Prompt ensembling (n={num_temp})")

        mean_text_features = 0
        for i, temp in enumerate(self.templates):
            prompts = [temp.format(c.replace("_", " ")) for c in classnames]
            prompts = torch.cat([clip.tokenize(p) for p in prompts]).to(self.device)
            text_features = clip_model.encode_text(prompts)
            text_features = text_features / text_features.norm(dim=-1, keepdim=True)
            mean_text_features = mean_text_features + text_features
        mean_text_features = mean_text_features / num_temp
        mean_text_features = mean_text_features / mean_text_features.norm(dim=-1, keepdim=True)

        self.text_features = mean_text_features
        self.clip_model = clip_model
