import torch
import torch.nn as nn
from torch.nn import functional as F
from torchvision.models._utils import IntermediateLayerGetter

from dassl.engine import TRAINER_REGISTRY, TrainerX
from dassl.optim import build_optimizer, build_lr_scheduler

from clip import clip
from clip.model import convert_weights

from .coop import load_clip_to_cpu
from .imagenet_templates import IMAGENET_TEMPLATES, IMAGENET_TEMPLATES_SELECT
from tqdm import tqdm

import matplotlib.pyplot as plt
import numpy as np
import matplotlib.patheffects as pe
import time

VISUALIZE = False
ERASE_FP = False
FILTER_FP = False
MEASURE_TIME = False

label_names = [
    "person", "bicycle", "car", "motorcycle", "airplane", "bus", "train", "truck",
    "boat", "traffic light", "fire hydrant", "stop sign", "parking meter", "bench",
    "bird", "cat", "dog", "horse", "sheep", "cow", "elephant", "bear", "zebra", "giraffe",
    "backpack", "umbrella", "handbag", "tie", "suitcase", "frisbee", "skis", "snowboard",
    "sports ball", "kite", "baseball bat", "baseball glove", "skateboard", "surfboard",
    "tennis racket", "bottle", "wine glass", "cup", "fork", "knife", "spoon", "bowl",
    "banana", "apple", "sandwich", "orange", "broccoli", "carrot", "hot dog", "pizza",
    "donut", "cake", "chair", "couch", "potted plant", "bed", "dining table", "toilet",
    "tv", "laptop", "mouse", "remote", "keyboard", "cell phone", "microwave", "oven",
    "toaster", "sink", "refrigerator", "book", "clock", "vase", "scissors", "teddy bear",
    "hair drier", "toothbrush"
]

CUSTOM_TEMPLATES = {
    "OxfordPets": "a photo of a {}, a type of pet.",
    "OxfordFlowers": "a photo of a {}, a type of flower.",
    "FGVCAircraft": "a photo of a {}, a type of aircraft.",
    "DescribableTextures": "{} texture.",
    "EuroSAT": "a centered satellite photo of {}.",
    "StanfordCars": "a photo of a {}.",
    "Food101": "a photo of {}, a type of food.",
    "SUN397": "a photo of a {}.",
    "Caltech101": "a photo of a {}.",
    "UCF101": "a photo of a person doing {}.",
    "ImageNet": "a photo of a {}.",
    "ImageNetSketch": "a photo of a {}.",
    "ImageNetV2": "a photo of a {}.",
    "ImageNetA": "a photo of a {}.",
    "ImageNetR": "a photo of a {}."
}


@TRAINER_REGISTRY.register()
class ZeroshotCLIP(TrainerX):
    def build_model(self):
        cfg = self.cfg
        classnames = self.dm.dataset.classnames

        print(f"Loading CLIP (backbone: {cfg.MODEL.BACKBONE.NAME})")
        clip_model = load_clip_to_cpu(cfg)
        clip_model.to(self.device)

        try:
            temp = CUSTOM_TEMPLATES[cfg.DATASET.NAME]
        except:
            print('!! WARNING: Not found template for {}'.format(cfg.DATASET.NAME))
            temp = "a photo of a {}."

        prompts = [temp.format(c.replace("_", " ")) for c in classnames]
        print(f"Prompts: {prompts}")
        prompts = torch.cat([clip.tokenize(p) for p in prompts])
        prompts = prompts.to(self.device)

        with torch.no_grad():
            text_features = clip_model.encode_text(prompts)
            text_features = text_features / text_features.norm(dim=-1, keepdim=True)

        self.text_features = text_features
        self.clip_model = clip_model

    def model_inference(self, image):
        image_features = self.clip_model.encode_image(image)
        image_features = image_features / image_features.norm(dim=-1, keepdim=True)
        logit_scale = self.clip_model.logit_scale.exp()
        logits = logit_scale * image_features @ self.text_features.t()

        return logits, None, None


def entropy_from_logits(logits):
    # 로그 확률을 직접 계산
    log_probs = F.log_softmax(logits, dim=-1)
    probs = torch.exp(log_probs)
    # 확률 분포에 대한 entropy 계산
    ent = -torch.sum(probs * log_probs, dim=-1)
    return ent


def read_npy(numpy_file, device):
    all_features = np.load(numpy_file, allow_pickle=True)

    # for ii, l_n in enumerate(label_names):
    #     temp = torch.tensor(all_features[ii])
    #     if ii == 0:
    #         text_caption = temp[:16].unsqueeze(0)
    #     else:
    #         text_caption = torch.cat((text_caption, temp[:16].unsqueeze(0)),0) # 80 16 512

    print('load complete:', numpy_file)
    return torch.tensor(all_features).to(torch.float16).to(
        device)


@TRAINER_REGISTRY.register()
class ZeroshotCLIP_dense(TrainerX):
    def build_model(self):
        cfg = self.cfg
        classnames = self.dm.dataset.classnames

        print(f"Loading CLIP (backbone: {cfg.MODEL.BACKBONE.NAME})")
        clip_model = load_clip_to_cpu(cfg)
        clip_model.to(self.device)

        try:
            temp = CUSTOM_TEMPLATES[cfg.DATASET.NAME]
        except:
            print('!! WARNING: Not found template for {}'.format(cfg.DATASET.NAME))
            temp = "a photo of a {}."
            # temp = "a photo of a part of {}"
            # temp = "a photo of the small {}."
            # temp = "a photo of no {}."

        prompts = [temp.format(c.replace("_", " ")) for c in classnames]
        print(f"Prompts: {prompts}")
        prompts = torch.cat([clip.tokenize(p) for p in prompts])
        prompts = prompts.to(self.device)

        with torch.no_grad():
            text_features = clip_model.encode_text(prompts)
            text_features = text_features / text_features.norm(dim=-1, keepdim=True)

        self.text_features = text_features
        self.clip_model = clip_model

        self.visual_encoder = IntermediateLayerGetter(self.clip_model.visual, {"layer4": "0"})
        self.positional_embedding = self.clip_model.visual.attnpool.positional_embedding[1::]
        self.v_linear_weight = self.clip_model.visual.attnpool.v_proj.weight
        self.v_linear_bias = self.clip_model.visual.attnpool.v_proj.bias
        self.c_linear_weight = self.clip_model.visual.attnpool.c_proj.weight
        self.c_linear_bias = self.clip_model.visual.attnpool.c_proj.bias

        self.caption = read_npy('COCO_caption_selection_101.npy', self.device)
        self.caption = self.caption / self.caption.norm(dim=-1, keepdim=True)  # [80,16,1024]

    def encode_image(self, x):
        def stem(x):
            for conv, bn in [(self.visual_encoder.conv1, self.visual_encoder.bn1), \
                             (self.visual_encoder.conv2, self.visual_encoder.bn2),
                             (self.visual_encoder.conv3, self.visual_encoder.bn3)]:
                x = self.visual_encoder.relu(bn(conv(x)))
            x = self.visual_encoder.avgpool(x)
            return x

        x = x.type(self.visual_encoder.conv1.weight.dtype)
        x = stem(x)
        x = self.visual_encoder.layer1(x)
        x = self.visual_encoder.layer2(x)
        x = self.visual_encoder.layer3(x)  # torch.Size([B, 512, 28, 28])
        x = self.visual_encoder.layer4(x)  # torch.Size([B, 1024, 14, 14])
        return x  # torch.Size([B, 2048, 7, 7])

    def model_inference(self, image):
        image_feat = self.encode_image(image)  # [B,C,H,W]
        b, c, h, w = image_feat.shape
        x = image_feat.reshape(b, c, h * w).permute(2, 0, 1)  # [7x7,B,2048]

        x = F.linear(x, self.v_linear_weight, self.v_linear_bias)
        x = F.linear(x, self.c_linear_weight, self.c_linear_bias)
        image_features = x

        image_feature_, _ = self.clip_model.visual.attnpool(image_feat)  # [7x7,B,2048]-> #[B,1024]

        image_features = image_features / image_features.norm(dim=-1, keepdim=True)
        image_feature_ = image_feature_ / image_feature_.norm(dim=-1, keepdim=True)

        caption_logit = torch.matmul(self.caption, image_feature_.t())
        A = torch.matmul(self.caption, image_feature_.t())  # 80 16 100
        A_up = F.softmax(A, dim=1)  # 80 16 100

        logit_scale = self.clip_model.logit_scale.exp()  # logit_scale = self.clip_model.logit_scale.exp()
        pup_logit = torch.sum(A_up * caption_logit, dim=1).permute(1, 0)

        # original logits
        origin_logit = logit_scale * image_feature_ @ self.text_features.t()

        # Calip Text Update / softmax(VT)V => HW axis
        text_sim = image_features @ self.text_features.t()  # [HW,B,cls]
        text_sim = torch.nn.functional.softmax(0.5*text_sim,dim=0)
        text_up= torch.bmm( image_features.permute(1,2,0), text_sim.permute(1,0,2))  # [B,C,HW] [B,HW,cls] => [B,C,cls]
        text_up_logits = logit_scale * image_feature_.unsqueeze(1) @ text_up  #[B,1,C] [B,C,cls] => [B,cls]
        text_up_logits = text_up_logits.squeeze()
        
        # Calip Image Update / softmax(VT)T => cls axis
        img_sim = image_features @ self.text_features.t()  # [HW,B,cls]
        img_sim = torch.nn.functional.softmax(0.5*img_sim,dim=2) #[HW,B,cls]
        img_up = img_sim @ self.text_features #[HW,B,cls] [cls,C] = [HW,B,C]
        img_up = torch.mean(img_up, dim=0) + torch.max(img_up, dim=0)[0]  # [B,C]
        img_up_logits = logit_scale * img_up @ self.text_features.t()  # [B,C] [C,cls] => [B,cls]

        
        #print(text_up_logits)
        #print(img_up_logits)
        #print(origin_logit)

        #logits_ =  text_up_logits *3 + img_up_logits*1 + origin_logit *1

        logits = logit_scale * image_features @ self.text_features.t()  # HW * B * cls = HW * B * C,  cls * C
        patch_logits = logits.clone()

        prob_spatial = torch.nn.functional.softmax(logits, dim=0)  # [7x7,B,cls] #
        logits = torch.sum(logits * prob_spatial, dim=0)  # [B,cls] = [7x7,B,cls] * [7x7,B,cls]

        return (origin_logit,text_up_logits,img_up_logits), logits, patch_logits, None, None

    def attnpool_sliding_window(self, feature_map, input_window, output_size):
        """
        Slides a window over the feature_map and applies the attention pooling.

        :param feature_map: Input feature map. Shape: [B, C, H, W]
        :param input_window: Size of the window to slide. For 5x5 window, it's 5.
        :param output_size: The resulting feature map size. For 3x3 result, it's 3.
        :return: Processed feature map. Shape: [B, output_size, output_size, 1024]
        """
        b, c, h, w = feature_map.shape
        stride = (h - input_window) // (output_size - 1)

        pooled_features = []
        for i in range(output_size):
            for j in range(output_size):
                # Extract the window
                sub_feat = feature_map[:, :, i * stride:i * stride + input_window, j * stride:j * stride + input_window]

                # Apply attnpool
                pooled, _ = self.clip_model.visual.attnpool(sub_feat)  # [B, 1024]
                pooled_features.append(pooled)

        # Reshape to the desired output size
        pooled_features = torch.stack(pooled_features, dim=1).reshape(b, output_size, output_size, -1)
        pooled_features = pooled_features.to(feature_map.device)
        return pooled_features

    @torch.no_grad()
    def test(self, split=None):
        """A generic testing pipeline."""
        self.set_model_mode("eval")
        self.evaluator.reset()

        if split is None:
            split = self.cfg.TEST.SPLIT

        if split == "val" and self.val_loader is not None:
            data_loader = self.val_loader
            print("Do evaluation on {} set".format(split))
        else:
            data_loader = self.test_loader
            print("Do evaluation on test set")

        folder_path = 'visualize/'
        # 폴더가 없으면 생성
        import os
        if not os.path.exists(folder_path):
            os.makedirs(folder_path)
            
        
        ## find optimal ratio
        
        origin_logit_list = []
        text_up_logits_list = []
        img_up_logits_list = []
        label_list = []
        

        for batch_idx, batch in enumerate(tqdm(data_loader)):
            input, label = self.parse_batch_test(batch)

            bs = label.shape[0]
            
            if MEASURE_TIME:
                st = time.time()
                
            (origin_logit,text_up_logits,img_up_logits), output_pos, patch_output, image_features_, text_features_ = self.model_inference(input)

            if MEASURE_TIME:
                ed = time.time()
                print (ed-st)
 
            #self.evaluator.process(output, label, output_pos)
            origin_logit_list.append(origin_logit)
            text_up_logits_list.append(text_up_logits)
            img_up_logits_list.append(img_up_logits)
            label_list.append(label)
    
        
        #logits = torch.cat(origin_logit_list,dim=0) * 1.36 ## adjust scailing
        logits = torch.cat(origin_logit_list,dim=0)
        logits1 = torch.cat(text_up_logits_list,dim=0)
        logits2 = torch.cat(img_up_logits_list,dim=0)
        label = torch.cat(label_list,dim=0)
        
        beta2 = 2.0
        beta3 = 2.0
        module_size = 10
        
        print(logits.max(), logits1.max(), logits2.max())
        """
        # original beta search
        beta2_list = [i * (beta2 - 0.001) / module_size + 0.001 for i in range(module_size)]
        beta3_list = [i * (beta3 - 0.001) / module_size + 0.001 for i in range(module_size)]
        """ 
        beta2_list = [0.1 * i for i in range(1, int(beta2 / 0.1) + 1)]
        beta3_list = [0.1 * i for i in range(1, int(beta3 / 0.1) + 1)]
        
        
        print('-' * 20)
        print('Starting searching...')
        print('     beta1 = 1.0')
        print('     beta2 searching range: [0.1, ' + str(beta2) + ']')
        print('     beta3 searching range: [0.1, ' + str(beta3) + ']')
        print('-' * 20)

        best_acc = 0.
        best_beta2 = 0.
        best_beta3 = 0.

        for beta2 in beta2_list:
            for beta3 in beta3_list:
                print ('beta2:',beta2,'beta3:',beta3)
                
                logits = logits + logits1 * beta2 + logits2 * beta3
                self.evaluator.reset()
                self.evaluator.process(logits, label)
                results = self.evaluator.evaluate()
                print (results)
                
                key,acc = list(results.items())[0]
                
                if acc > best_acc:
                    print('New best setting, beta1: {:.4f}; beta2: {:.4f}; beta3: {:.4f}; Acc: {:.2f}'.format(1, beta2, beta3, acc))
                    best_acc = acc
                    best_beta2 = beta2
                    best_beta3 = beta3

                

        for k, v in results.items():
            tag = "{}/{}".format(split, k)
            self.write_scalar(tag, v, self.epoch)

        return list(results.values())[0]


@TRAINER_REGISTRY.register()
class ZeroshotCLIP2(ZeroshotCLIP):
    """Prompt ensembling."""

    # templates = IMAGENET_TEMPLATES
    templates = IMAGENET_TEMPLATES_SELECT

    def build_model(self):
        cfg = self.cfg
        classnames = self.dm.dataset.classnames

        print(f"Loading CLIP (backbone: {cfg.MODEL.BACKBONE.NAME})")
        clip_model = load_clip_to_cpu(cfg)
        clip_model.to(self.device)

        for params in clip_model.parameters():
            params.requires_grad_(False)

        # add custom-made prompt
        if cfg.DATASET.NAME != "ImageNet":
            self.templates += [CUSTOM_TEMPLATES[cfg.DATASET.NAME]]

        num_temp = len(self.templates)
        print(f"Prompt ensembling (n={num_temp})")

        mean_text_features = 0
        for i, temp in enumerate(self.templates):
            prompts = [temp.format(c.replace("_", " ")) for c in classnames]
            prompts = torch.cat([clip.tokenize(p) for p in prompts]).to(self.device)
            text_features = clip_model.encode_text(prompts)
            text_features = text_features / text_features.norm(dim=-1, keepdim=True)
            mean_text_features = mean_text_features + text_features
        mean_text_features = mean_text_features / num_temp
        mean_text_features = mean_text_features / mean_text_features.norm(dim=-1, keepdim=True)

        self.text_features = mean_text_features
        self.clip_model = clip_model
