import random
from datetime import datetime
import concurrent.futures
from functools import partial
import sys
import scipy.optimize as opt
import math
import numpy as np
if sys.platform == 'win32':
    import matplotlib.pyplot as plt


MAX_WORKERS = 1000
G = 0.5


def randomize_locations(n, k, r, max_per_row=None, max_per_col=None):
    if max_per_col is not None:
        while True:
            used, unused = randomize_locations(n, k, r, max_per_row=max_per_row)
            cols = {i: 0 for i in range(n)}
            all_good = True
            for i, j in used:
                cols[j] += 1
                if cols[j] > max_per_col:
                    all_good = False
                    break
            if all_good:
                return used, unused

    max_per_row = min(max_per_row, n) if max_per_row is not None else n
    vars_per_row = [1] * k
    options = list(range(k))
    for _ in range(r - k):
        choice = random.choice(options)
        vars_per_row[choice] += 1
        if vars_per_row[choice] == max_per_row:
            options.remove(choice)

    unused = [(a, b) for a in range(k) for b in range(n)]
    used = []

    for i in range(k):
        for j in random.sample(range(n), vars_per_row[i]):
            unused.remove((i, j))
            used.append((i, j))

    return used, unused


def rand(*size, order=1):
    return np.random.normal(scale=math.sqrt(G / order), size=size)


def run_try(k, i, equations, x0_size, checker, initializations):
    if isinstance(x0_size, dict):
        x0_size = x0_size[k]
    e = equations(k)
    for i in range(initializations):
        x0 = rand(x0_size)
        x, info, status, message = opt.fsolve(e, x0=x0,
                                              full_output=True)
        if status == 1:
            if checker is not None:
                checker(e, x)
            return i
    return None


def run_concurrent_solver(equations, x0_size, checker, ks, tries, initializations):
    return run_concurrent(ks, tries, run_try, equations, x0_size, checker, initializations)


def run_concurrent(ks, tries, func, *args):
    start = datetime.now()
    futures = []
    with concurrent.futures.ThreadPoolExecutor(max_workers=MAX_WORKERS) as executor:
        for k in ks:
            for i in range(tries):
                future = executor.submit(partial(func, k, i, *args))
                future.__k_value__ = k
                future.__run_num__ = i
                futures.append(future)

        success = {k: 0 for k in ks}
        failures = {k: 0 for k in ks}
        for i, future in enumerate(concurrent.futures.as_completed(futures)):
            k = future.__k_value__
            sol_round = future.result()
            if sol_round is not None:
                success[k] += 1
                result = f"success on {sol_round}'s round"
            else:
                failures[k] += 1
                result = 'failure'
            print(
                f'{datetime.now()} finished run {future.__run_num__} of k={k}, {result}, it has {success[k]} success and {failures[k]} failures')
    results = {k: success[k] / tries for k in ks}
    print(results)
    print(f'took {datetime.now() - start}')
    if sys.platform == 'win32':
        plt.plot(results.keys(), results.values())
        plt.show()
