import json
from src.xlogomini.components.code import actions
from src.xlogomini.components.constraints.code_constraints import CodeConstraints


class Code():
    def __init__(self, astJson=None):
        if astJson:  # Try to parse the given Json.
            try:
                cursor, open_bodies, success = self._parse_json(astJson)
                assert success
                self.astJson = astJson
                self.cursor = cursor
                self.open_bodies = open_bodies
            except Exception as e:
                print(e)
                raise e
        else:  # Create an empty src
            self.astJson = {"run": [{"type": "cursor"}]}
            self.cursor = self.astJson["run"]
            self.open_bodies = []

        self.block_cnt, self.depth = self._get_block_cnt(block_cnt={}, body=astJson['run'], depth=0)
        self.n_blocks = sum(self.block_cnt.values())

    def _get_block_cnt(self, block_cnt, body, depth):
        """
        Get the block count for each block, stored in a dict, and the depth of the code
        """
        max_depth = depth
        for b in body:
            blockType = b['type']
            if blockType not in block_cnt.keys():
                block_cnt[blockType] = 0
            # update count
            block_cnt[blockType] += 1

            # blocks used inside repeat
            if blockType == 'repeat':
                block_cnt, repeat_depth = self._get_block_cnt(block_cnt, b['body'], depth + 1)
                max_depth = max(depth, repeat_depth)

        return block_cnt, max_depth

    def get_pen_colors(self, pc, body):
        """
        Get a set of pen colors that have been set by the code. `pc` should be a set.
        """
        for b in body:
            blockType = b['type']
            if blockType == 'setpc':
                pc.add(b['value'])

            # blocks used inside repeat
            if blockType == 'repeat':
                pc = self.get_pen_colors(pc, b['body'])

        return pc

    def is_done(self):
        return self.cursor is None

    def getJson(self):
        return self.astJson

    def _parse_json(self, astJson):
        '''
        Checks if astJson in the required format and extracts cursor and open bodies information.
        '''
        cursors_list = []  # Used as a static variable and populated by _parse_body with the path to cursor when cursor found

        def _parse_body(block_body, cursor_valid, cursor_path):
            '''
            A recursive helper function that parses the given body.
            block_body:list[dict] --> A block's body that potentially contains other blocks
            cursor_valid:bool --> True if block_body is on a valid path for sequential writing
            cursor_path:list[list] --> The path from root to this block
            '''
            if not cursor_path is None:
                cursor_path.append(block_body)
            for i in range(len(block_body)):
                child = block_body[i]
                cursor_valid_iter = cursor_valid and (
                        i == len(block_body) - 1)  # If this block is cursor invalid, all children will be too
                if child["type"] == "repeat":
                    assert len(child.keys()) == 3  # type, body, times
                    assert isinstance(child["times"], int)
                    assert 0 < child["times"] < 13
                    assert _parse_body(child["body"], cursor_valid_iter,
                                       None if not cursor_valid_iter else cursor_path.copy())
                elif child in actions.SET_VARIABLES:
                    assert len(child.keys()) == 2  # type, color
                    assert (isinstance(child["value"], str) or child['value'] is None)
                    assert child["value"] in ["red", "black", "blue", "green", "yellow", "white", None]
                elif child in actions.BASIC_ACTIONS:
                    assert len(child.keys()) == 1
                else:
                    assert child["type"] == "cursor", f"Unknown block, found {child}"
                    assert cursor_valid_iter
                    cursors_list.append(cursor_path)
            return True

        assert len(astJson.keys()) == 1  # Root should have only key 'run'
        assert _parse_body(astJson["run"], True, [])
        assert len(cursors_list) <= 1
        success = True
        cursor = cursors_list[0][-1] if len(cursors_list) == 1 else None
        open_bodies = cursors_list[0][:-1] if len(cursors_list) else []
        return cursor, open_bodies, success

    def check_constraints(self, constraint):
        constraint = CodeConstraints(constraint)
        # check at_most and exactly for used blocks
        for block_name, cnt in self.block_cnt.items():
            if (cnt > constraint.at_most(block_name)) or (cnt < constraint.at_least(block_name)):
                return False

        # check at_most and exactly for 'all'
        if self.n_blocks > constraint.at_most('all') or (self.n_blocks < constraint.at_least('all')):
            return False

        # check start-by
        if len(constraint.start) > self.n_blocks:
            return False
        if len(constraint.start) > 0:
            for i in range(len(constraint.start)):
                if self.astJson['run'][i]['type'] != constraint.start[i]:
                    return False
        return True

    def toString(self):
        return json.dumps(self.astJson, sort_keys=True, indent=2)

    def __repr__(self):
        def print_json(obj, indent=0):
            output = ""
            if "type" in obj:
                if obj["type"] == "repeat":
                    output += "  " * indent + "repeat(" + str(obj["times"]) + "){\n"
                    for element in obj["body"]:
                        output += print_json(element, indent + 1)
                    output += "  " * indent + "}\n"
                elif obj["type"] == "setpc":
                    output += "  " * indent + obj["type"] + '(' + obj["value"] + ')\n'
                else:
                    output += "  " * indent + obj["type"] + "\n"
            else:
                for element in obj:
                    output += print_json(element, indent)
            return output

        code_str = ""
        for b in self.astJson['run']:
            code_str += print_json(b, 0)
        return code_str
