from typing import Optional

from serde import Serializable

from ..payloads import PayloadInstance
from ..tasks import TaskInstance
from .decoder import Decoder


class SimpleMappingDecoder(Decoder):
    """A decoder that looks for a keyword in the encoder's output to determine the secret payload."""

    def __init__(
        self,
        mapping: dict[str, PayloadInstance],
    ):
        """Initialize the SimpleMappingDecoder with a mapping of keywords to payloads.

        Args:
            mapping: maps strings to lookup in the encoder's output to payloads
        """
        self.mapping = mapping

    def get_secret_payload(
        self, task: TaskInstance, encoder_answer: str
    ) -> Optional[PayloadInstance]:
        """Get the secret payload from the encoder's answer.

        Uses the mapping to determine the secret payload from the encoder's answer.

        Args:
            task: the task to be performed by the encoder
            encoder_answer: the encoder's answer to the task

        Returns:
            The secret payload if it is found in the encoder's answer, else None
        """
        for key, val in self.mapping.items():
            if key.lower() in encoder_answer.lower():
                return val
        return None

    def serialize(self) -> Serializable:
        return {"mapping": self.mapping}
