import ast

import astor


def rename_variables(code: str) -> str:
    """Rename all the variables in a code string to generic var{i} names.

    Args:
        code: the code string from which to remove variables

    Returns:
        the code with variables replaced with generic names
    """

    class VariableVisitor(ast.NodeVisitor):
        def __init__(self):
            self.variables = set()

        def visit_Name(self, node):
            if isinstance(node.ctx, (ast.Store, ast.Param)):
                self.variables.add(node.id)
            self.generic_visit(node)

        def visit_FunctionDef(self, node):
            for arg in node.args.args:
                self.variables.add(arg.arg)
            self.generic_visit(node)

        def visit_Lambda(self, node):
            for arg in node.args.args:
                self.variables.add(arg.arg)
            self.generic_visit(node)

    class RenameVariableTransformer(ast.NodeTransformer):
        def __init__(self, name_mapping):
            self.name_mapping = name_mapping

        def visit_Name(self, node):
            if node.id in self.name_mapping:
                node.id = self.name_mapping[node.id]
            return node

        def visit_FunctionDef(self, node):
            for arg in node.args.args:
                if arg.arg in self.name_mapping:
                    arg.arg = self.name_mapping[arg.arg]
            return self.generic_visit(node)

        def visit_Lambda(self, node):
            for arg in node.args.args:
                if arg.arg in self.name_mapping:
                    arg.arg = self.name_mapping[arg.arg]
            return self.generic_visit(node)

    # Step 1: Collect variable names
    tree = ast.parse(code)
    visitor = VariableVisitor()
    visitor.visit(tree)

    # Step 2: Create a mapping of old names to new generic names
    sorted_variables = sorted(visitor.variables)
    name_mapping = {name: f"var_{i}" for i, name in enumerate(sorted_variables)}

    # Step 3: Apply the renaming transformation
    transformer = RenameVariableTransformer(name_mapping)
    transformed_tree = transformer.visit(tree)

    # Step 4: Convert AST back to source code
    return astor.to_source(transformed_tree)


def remove_docstrings(code: str) -> str:
    """Remove any docstrings from the code.

    Args:
        code: the code string from which to remove docstrings

    Returns:
        the code with docstrings removed
    """

    class DocstringRemover(ast.NodeTransformer):
        def visit_FunctionDef(self, node):
            node.body = self._remove_docstring(node.body)
            self.generic_visit(node)
            return node

        def visit_ClassDef(self, node):
            node.body = self._remove_docstring(node.body)
            self.generic_visit(node)
            return node

        def visit_Module(self, node):
            node.body = self._remove_docstring(node.body)
            self.generic_visit(node)
            return node

        def _remove_docstring(self, body):
            if (
                body
                and isinstance(body[0], ast.Expr)
                and isinstance(body[0].value, ast.Constant)
            ):
                return body[1:]
            return body

    tree = ast.parse(code)
    transformer = DocstringRemover()
    transformed_tree = transformer.visit(tree)
    return astor.to_source(transformed_tree)


if __name__ == "__main__":
    code_str = """def fun(a, b):
    '''docy docstring
    '''
    c = a + b
    # HIYA!
    for i in range(c):
        print(j := i)
    return a + b + c / 2
    """

    print(rename_variables(code_str))

    print(remove_docstrings(code_str))
