from enum import Enum
from pathlib import Path
from typing import Optional

import pandas as pd

from episode.tasks.task import Task
from logger import logger


class SeededStrategiesPromptType(str, Enum):
    NONE = "noseededstrats"
    SYSTEM_PROMPT = "seededsysprompt"
    STRATEGIZE_INSTRUCTIONS = "seededstrategize"


class SteganographyCategory(str, Enum):
    FORMAT_BASED = "Format-based"
    LINGUISTIC = "Linguistic"
    STATISTICAL = "Statistical"
    MISC = "Miscellaneous"


def get_seeded_strategies_from_csv(
    seed_file_path: Path = Path("data", "seed_strategies.csv"),
    task_filters: Optional[list[Task]] = None,
    category_filters: Optional[list[SteganographyCategory]] = None,
    applicable_filter: Optional[str] = None,
) -> str:
    """Load seed strategies from CSV file, apply filters, and return string for model prompts.

    Args:
        seed_file_path: Path to seed strategies CSV file.
        NAME_filters: List of task names to filter on (eg: TaskName.SIMPLE_QA, TaskName.AQUA_RAT, TaskName.HUMAN_EVAL).
        task_type_filters: List of task types to filter on (eg: TaskType.QA, TaskType.SUMMARIZATION, TaskType.CODING).
        category_filters: List of categories to filter on (eg: SteganographyCategory.STRUCTURAL, SteganographyCategory.LINGUISTIC, SteganographyCategory.STATISTICAL, SteganographyCategory.MISC).
        applicable_filter: Filter on whether the strategy is applicable.

    Returns:
        seed_strategies_prompt: Multi-line string of seed strategies to be used in model prompts.
    """

    # Load strategies from file
    seed_df = pd.read_csv(seed_file_path)

    # Apply filters
    if task_filters:
        seed_df = seed_df[
            seed_df["Task Name"].isin([task.__NAME for task in task_filters])
        ]
        seed_df = seed_df[
            seed_df["Task Type"].isin([task.TASK_TYPE for task in task_filters])
        ]
    if category_filters:
        seed_df = seed_df[
            seed_df["Category"].isin([category.value for category in category_filters])
        ]
    if applicable_filter:
        seed_df = seed_df[seed_df["Applicable"] == applicable_filter]

    # Create string for prompt with techniques
    seed_strategies_prompt = None
    if seed_df.empty:
        logger.warning("No seed strategies found after applying filters.")
    else:
        seed_df["Prompt"] = (
            "(Technique Type: "
            + seed_df["Category"]
            + ") "
            + seed_df["Title"]
            + " - "
            + seed_df["Technique"]
        )
        seed_strategies_prompt = "\n".join(
            [f" - {prompt}" for prompt in seed_df["Prompt"]]
        )
        # logger.info(f"Seed strategies prompt: {seed_strategies_prompt}")

    return seed_strategies_prompt
