from concurrent.futures import ThreadPoolExecutor
from typing import Optional

from logger import logger

BANNED_OPERATIONS = ["eval", "exec", "open"]

builtins_as_dict = (
    __builtins__ if isinstance(__builtins__, dict) else __builtins__.__dict__
)
BUILTINS_DICT = {
    key: value
    for key, value in builtins_as_dict.items()
    if key not in BANNED_OPERATIONS
}


def safe_exec(code: str, timeout: int = 1) -> tuple[bool, Optional[Exception]]:
    """Try to exec some code with a restricted set of builtins.

    We have restricted builtins which are missing:
    * import
    * eval
    * exec
    * open

    Args:
        code: the code to try to run

    Returns:
        (true, None) if the code executes with no exceptions. Else, (false, exception)
    """

    locals_ = {"__builtins__": BUILTINS_DICT}

    def exec_code():
        exec(code, locals_)

    with ThreadPoolExecutor() as executor:
        future = executor.submit(exec_code)
        try:
            # Wait for the result with a timeout
            future.result(timeout=timeout)
            return (True, None)
        except TimeoutError:
            logger.warning("Code execution timed out")
            return (False, TimeoutError("Code execution exceeded timeout"))
        except Exception as e:
            logger.warning(f"Code raised exception: {repr(e)}")
            return (False, e)
    try:
        exec(code, locals_)
        return (True, None)
    except Exception as e:
        logger.warning(f"Code raised exception: {repr(e)}")
        return (False, e)


if __name__ == "__main__":
    code = """pass
"""
    print(safe_exec(code))
