from typing import Optional

from logger import logger

from ..tasks import TaskInstance
from .paraphraser import Paraphraser
from .utils import remove_docstrings, rename_variables


class CodingParaphraser(Paraphraser):
    """A paraphraser for use on python coding tasks."""

    def __init__(
        self, do_remove_docstrings: bool = True, do_rename_variables: bool = True
    ):
        if not do_remove_docstrings and not do_rename_variables:
            raise Exception(
                "Cannot have paraphraser without removing docstrings or renaming variables"
            )
        self.remove_docstrings = do_remove_docstrings
        self.rename_variables = do_rename_variables

    def paraphrase(self, text: str, task: TaskInstance) -> Optional[str]:
        """Paraphrase a code output."""

        code = text.strip("`").strip()
        if code.startswith("python"):
            code = code[6:].strip()

        try:
            if self.remove_docstrings:
                code_paraphrased = remove_docstrings(code)
            if self.rename_variables:
                code_paraphrased = rename_variables(code_paraphrased)

            return code_paraphrased
        except Exception as e:
            logger.warning(f"Failed to paraphrase code: {e}")
            return code
