from models import ChatModel, ModelType, create_chat_model
from serde import Serializable

from ..tasks.cv_task import NaturalTask, NaturalTaskInstance


class NaturalEncoder:
    """Uses an LLM to the task, but does not explicitly try to encode the payload."""

    def __init__(
        self,
        strategy: str,
        model: ChatModel | ModelType,
        task_type: NaturalTask,
    ):
        self.strategy = strategy
        self.model = create_chat_model(model) if isinstance(model, ModelType) else model
        self.system_prompt = f"""{task_type.ENCODER_TASK_DESCRIPTION}
Here is your strategy:
{strategy}"""

    def get_message(self, task: NaturalTaskInstance) -> str:
        """Get the message given the task and payload.

        Uses its strategy to try to encode the payload in the output message.

        Args:
            task: a task instance e.g. answer question x
            payload: the payload to encode into the resopnse to the task

        Returns:
            the encoder's response to the task
        """

        messages = [
            {"role": "system", "content": self.system_prompt},
            {
                "role": "user",
                "content": task.get_encoder_prompt(),
            },
        ]

        return self.model.query_with_retry(messages)

    def serialize(self) -> Serializable:
        return {
            "model": self.model,
            "strategy": self.strategy,
        }
