#include "../../inc/lacam2/lacam_instance.hpp"

LACAMInstance::LACAMInstance(const std::string& map_filename,
                   const std::vector<uint>& start_indexes,
                   const std::vector<uint>& goal_indexes,
                   const std::vector<uint>& dummy_goal_indexes)
    : G(map_filename),
      starts(Config()),
      goals(Config()),
      dummy_goals(Config()),
      N(start_indexes.size())
{
  for (auto k : start_indexes) starts.push_back(G.U[k]);
  for (auto k : goal_indexes) goals.push_back(G.U[k]);
  for (auto k : dummy_goal_indexes) dummy_goals.push_back(G.U[k]);
}

//for start kit
LACAMInstance::LACAMInstance(SharedEnvironment* env)
  :G(env->map,env->rows,env->cols),starts(Config()), goals(Config()),N(env->num_of_agents)
{
  for (int i = 0; i < env->curr_states.size(); i++)
  {
    starts.push_back(G.U[env->curr_states[i].location]);
  }
  for (int i = 0; i < env->goal_locations.size(); i++)
  {
    goals.push_back(G.U[env->goal_locations[i].front().first]);
  }
}

// for load instance
static const std::regex r_instance =
    std::regex(R"(\d+\t.+\.map\t\d+\t\d+\t(\d+)\t(\d+)\t(\d+)\t(\d+)\t.+)");

LACAMInstance::LACAMInstance(const std::string& scen_filename,
                   const std::string& map_filename, const uint _N)
    : G(Graph(map_filename)), starts(Config()), goals(Config()), N(_N)
{
  // load start-goal pairs
  std::ifstream file(scen_filename);
  if (!file) {
    info(0, 0, scen_filename, " is not found");
    return;
  }
  std::string line;
  std::smatch results;

  while (getline(file, line)) {
    // for CRLF coding
    if (*(line.end() - 1) == 0x0d) line.pop_back();

    if (std::regex_match(line, results, r_instance)) {
      uint x_s = std::stoi(results[1].str()); //x is the column, y is the row
      uint y_s = std::stoi(results[2].str());
      uint x_g = std::stoi(results[3].str());
      uint y_g = std::stoi(results[4].str());
      if (x_s < 0 || G.width <= x_s || x_g < 0 || G.width <= x_g) continue;
      if (y_s < 0 || G.height <= y_s || y_g < 0 || G.height <= y_g) continue;
      auto s = G.U[G.width * y_s + x_s];
      auto g = G.U[G.width * y_g + x_g];
      if (s == nullptr || g == nullptr) continue;
      starts.push_back(s);
      goals.push_back(g);
    }

    if (starts.size() == N) break;
  }
}

LACAMInstance::LACAMInstance(const std::string& map_filename, std::mt19937* MT,
                   const uint _N)
    : G(Graph(map_filename)), starts(Config()), goals(Config()), N(_N)
{
  // random assignment
  const auto V_size = G.size();

  // set starts
  auto s_indexes = std::vector<uint>(V_size);
  std::iota(s_indexes.begin(), s_indexes.end(), 0);
  std::shuffle(s_indexes.begin(), s_indexes.end(), *MT);
  size_t i = 0;
  while (true) {
    if (i >= V_size) return;
    starts.push_back(G.V[s_indexes[i]]);
    if (starts.size() == N) break;
    ++i;
  }

  // set goals
  auto g_indexes = std::vector<uint>(V_size);
  std::iota(g_indexes.begin(), g_indexes.end(), 0);
  std::shuffle(g_indexes.begin(), g_indexes.end(), *MT);
  size_t j = 0;
  while (true) {
    if (j >= V_size) return;
    goals.push_back(G.V[g_indexes[j]]);
    if (goals.size() == N) break;
    ++j;
  }
}

bool LACAMInstance::is_valid(const int verbose) const
{
  if (N != starts.size() || N != goals.size()) {
    info(1, verbose, "invalid N, check instance");
    return false;
  }
  return true;
}

std::ostream& operator<<(std::ostream& os, const Solution& solution)
{
  auto N = solution.front().size();
  for (size_t i = 0; i < N; ++i) {
    os << std::setw(5) << i << ":";
    for (size_t k = 0; k < solution[i].size(); ++k) {
      if (k > 0) os << "->";
      os << std::setw(5) << solution[i][k]->index;
    }
    os << std::endl;
  }
  return os;
}


void LACAMInstance::set_dummygoals(const std::vector<uint>& dummy_goal_indexes)
{
  dummy_goals = Config();
  for (auto k : dummy_goal_indexes) dummy_goals.push_back(G.U[k]);
}
