import torch
import torch.nn as nn
import torch.optim as optim
import torch.nn.functional as F
import numpy as np
import matplotlib.pyplot as plt
import random
from collections import deque


# ---------- Environment: Ackley ----------
def ackley(x):
    """
    Ackley function with global minimum at x = 0
    Typically defined on [-5, 5] but we'll use [-5, 5] for better convergence
    """
    a = 20
    b = 0.2
    c = 2 * np.pi
    d = len(x)

    sum1 = sum([xi ** 2 for xi in x])
    sum2 = sum([np.cos(c * xi) for xi in x])

    term1 = -a * np.exp(-b * np.sqrt(sum1 / d))
    term2 = -np.exp(sum2 / d)

    return term1 + term2 + a + np.exp(1)


# ---------- Improved RL Policy Network ----------
class ImprovedChildPolicy(nn.Module):
    def __init__(self, dim, hidden_dim=128):
        super().__init__()
        self.dim = dim

        # Input: parent1 + parent2 + parent fitness + population stats
        input_dim = dim * 2 + 2 + 4  # parents + parent_fits + [best, worst, mean, std]

        # Shared layers
        self.shared_net = nn.Sequential(
            nn.Linear(input_dim, hidden_dim),
            nn.ReLU(),
            nn.Dropout(0.1),
            nn.Linear(hidden_dim, hidden_dim),
            nn.ReLU(),
            nn.Dropout(0.1),
            nn.Linear(hidden_dim, hidden_dim // 2),
            nn.ReLU()
        )

        # Separate heads for mean and std
        self.mean_net = nn.Sequential(
            nn.Linear(hidden_dim // 2, hidden_dim // 4),
            nn.ReLU(),
            nn.Linear(hidden_dim // 4, dim),
            nn.Tanh()  # Bounded output
        )

        self.std_net = nn.Sequential(
            nn.Linear(hidden_dim // 2, hidden_dim // 4),
            nn.ReLU(),
            nn.Linear(hidden_dim // 4, dim),
            nn.Sigmoid()  # Always positive
        )

    def forward(self, state):
        shared_features = self.shared_net(state)

        # Mean scaled to Ackley problem bounds [-5, 5]
        mean = self.mean_net(shared_features) * 5

        # Std with reasonable bounds for Ackley (0.05 to 1.5)
        std = self.std_net(shared_features) * 1.45 + 0.05

        return mean, std

    def sample_action(self, state):
        mean, std = self.forward(state)
        dist = torch.distributions.Normal(mean, std)
        action = dist.sample()
        log_prob = dist.log_prob(action).sum(dim=-1)
        entropy = dist.entropy().sum(dim=-1)
        return action, log_prob, entropy


# ---------- Improved Value Network ----------
class ImprovedValueNetwork(nn.Module):
    def __init__(self, dim, hidden_dim=128):
        super().__init__()
        input_dim = dim * 2 + 2 + 4  # Same as policy

        self.net = nn.Sequential(
            nn.Linear(input_dim, hidden_dim),
            nn.ReLU(),
            nn.Dropout(0.1),
            nn.Linear(hidden_dim, hidden_dim),
            nn.ReLU(),
            nn.Dropout(0.1),
            nn.Linear(hidden_dim, hidden_dim // 2),
            nn.ReLU(),
            nn.Linear(hidden_dim // 2, 1)
        )

    def forward(self, state):
        return self.net(state).squeeze(-1)


# ---------- Improved State Representation ----------
def create_state(parent1, parent2, population, fitnesses):
    """Create rich state representation for Ackley"""
    p1_fit = ackley(parent1)
    p2_fit = ackley(parent2)

    # Population statistics
    pop_stats = [
        np.min(fitnesses),  # best fitness
        np.max(fitnesses),  # worst fitness
        np.mean(fitnesses),  # mean fitness
        np.std(fitnesses)  # fitness std
    ]

    # Combine all features
    state_list = list(parent1) + list(parent2) + [p1_fit, p2_fit] + pop_stats
    return torch.tensor(state_list, dtype=torch.float32)


# ---------- Improved Reward Function ----------
def compute_reward(child, parent1, parent2, population, fitnesses):
    """Multi-component reward function for Ackley"""
    child_fitness = ackley(child)
    parent1_fitness = ackley(parent1)
    parent2_fitness = ackley(parent2)

    # Component 1: Improvement over parents
    best_parent_fitness = min(parent1_fitness, parent2_fitness)
    # Avoid division by zero for Ackley (minimum is 0)
    if best_parent_fitness > 0:
        improvement_reward = (best_parent_fitness - child_fitness) / (best_parent_fitness + 1e-8)
    else:
        improvement_reward = max(0, -child_fitness)  # Reward for getting closer to 0

    # Component 2: Population ranking reward
    better_than_count = sum(1 for f in fitnesses if child_fitness < f)
    ranking_reward = better_than_count / len(fitnesses) - 0.5  # Center around 0

    # Component 3: Diversity bonus (distance from nearest neighbor)
    min_distance = min(np.linalg.norm(np.array(child) - np.array(ind))
                       for ind in population)
    diversity_reward = min(min_distance / 10.0, 1.0)  # Cap at 1.0, adjusted for Ackley range

    # Component 4: Minimize child fitness - This feels like cheating tho
    proximity_reward = -child_fitness  # Reward for minimizing fitness

    # Combined reward with weights
    total_reward = (1.0 * improvement_reward +
                    1.0 * ranking_reward +
                    1.0 * diversity_reward
                    + 1.0 * proximity_reward)

    return total_reward, {
        'improvement': improvement_reward,
        'ranking': ranking_reward,
        'diversity': diversity_reward,
        'proximity': proximity_reward,
        'child_fitness': child_fitness
    }


# ---------- Experience Buffer ----------
class ReplayBuffer:
    def __init__(self, capacity=10000):
        self.capacity = capacity
        self.buffer = deque(maxlen=capacity)

    def push(self, state, action, reward, log_prob, entropy, value):
        self.buffer.append((state, action, reward, log_prob, entropy, value))

    def sample(self, batch_size):
        batch = random.sample(self.buffer, min(batch_size, len(self.buffer)))
        states, actions, rewards, log_probs, entropies, values = zip(*batch)
        return (torch.stack(states), torch.stack(actions),
                torch.tensor(rewards, dtype=torch.float32),
                torch.stack(log_probs), torch.stack(entropies), torch.stack(values))

    def __len__(self):
        return len(self.buffer)


# ---------- Improved SPO Update ----------
def improved_spo_update(policy, value_net, optimizer_policy, optimizer_value,
                        buffer, batch_size=64, epochs=4, epsilon=0.2):
    if len(buffer) < batch_size:
        return 0, 0

    states, actions, rewards, old_log_probs, entropies, old_values = buffer.sample(batch_size)

    # Compute advantages
    with torch.no_grad():
        values = value_net(states)
        advantages = rewards - values
        returns = rewards

        # Normalize advantages
        advantages = (advantages - advantages.mean()) / (advantages.std() + 1e-8)

    total_policy_loss = 0
    total_value_loss = 0

    for epoch in range(epochs):
        # Get current policy outputs
        mean, std = policy(states)
        dist = torch.distributions.Normal(mean, std)
        new_log_probs = dist.log_prob(actions).sum(dim=-1)
        new_entropy = dist.entropy().sum(dim=-1)

        # SPO policy loss with quadratic penalty
        ratio = torch.exp(new_log_probs - old_log_probs.detach())
        mb_advantages = advantages.detach()

        # SPO objective: ratio * advantages - quadratic penalty
        policy_loss_elements = (ratio * mb_advantages -
                                (mb_advantages.abs() / (2 * epsilon)) * (ratio - 1).pow(2))
        policy_loss = -policy_loss_elements.mean()

        # Add entropy bonus for exploration
        entropy_loss = -0.01 * new_entropy.mean()

        total_loss = policy_loss + entropy_loss

        optimizer_policy.zero_grad()
        total_loss.backward()
        torch.nn.utils.clip_grad_norm_(policy.parameters(), 0.5)
        optimizer_policy.step()

        # Value function update
        current_values = value_net(states)
        value_loss = F.mse_loss(current_values, returns.detach())

        optimizer_value.zero_grad()
        value_loss.backward()
        torch.nn.utils.clip_grad_norm_(value_net.parameters(), 0.5)
        optimizer_value.step()

        total_policy_loss += total_loss.item()
        total_value_loss += value_loss.item()

    return total_policy_loss / epochs, total_value_loss / epochs


# ---------- Training Function ----------
def train_improved_rl(dim=10, training_episodes=2000):
    policy = ImprovedChildPolicy(dim)
    value_net = ImprovedValueNetwork(dim)

    optimizer_policy = optim.Adam(policy.parameters(), lr=1e-5, weight_decay=1e-6)
    optimizer_value = optim.Adam(value_net.parameters(), lr=1e-5, weight_decay=1e-6)

    buffer = ReplayBuffer(capacity=10000)

    print("Training Improved RL Agent for Ackley Function...")

    reward_history = []

    for episode in range(training_episodes):
        # Generate diverse populations
        pop_size = random.randint(50, 100)
        population = [np.random.uniform(-5,5, dim) for _ in range(pop_size)]
        fitnesses = np.array([ackley(ind) for ind in population])

        episode_rewards = []
        episode_info = []

        # Generate experience
        for step in range(20):
            # Smart parent selection (mix of tournament and fitness-based)
            if random.random() < 0.8:
                # Tournament selection
                parent1 = tournament_selection(population, fitnesses, 5)
                parent2 = tournament_selection(population, fitnesses, 5)
            else:
                # Random selection for diversity
                parent1 = random.choice(population)
                parent2 = random.choice(population)

            # Create state
            state = create_state(parent1, parent2, population, fitnesses)

            # Sample action
            action, log_prob, entropy = policy.sample_action(state.unsqueeze(0))
            action = action.squeeze(0)

            # Ensure bounds for Ackley function
            child = torch.clamp(action, -5, 5).detach().numpy()

            # Compute reward
            reward, info = compute_reward(child, parent1, parent2, population, fitnesses)

            # Get value estimate
            value = value_net(state.unsqueeze(0)).squeeze(0)

            # Store experience
            buffer.push(state, action, reward, log_prob.squeeze(0),
                        entropy.squeeze(0), value)

            episode_rewards.append(reward)
            episode_info.append(info)

        avg_reward = np.mean(episode_rewards)
        reward_history.append(avg_reward)

        # Update networks
        if len(buffer) > 100:
            policy_loss, value_loss = improved_spo_update(
                policy, value_net, optimizer_policy, optimizer_value, buffer
            )

            if episode % 100 == 0:
                avg_improvement = np.mean([info['improvement'] for info in episode_info])
                avg_ranking = np.mean([info['ranking'] for info in episode_info])
                avg_proximity = np.mean([info['proximity'] for info in episode_info])
                print(f"Episode {episode}: Reward={avg_reward:.4f}, "
                      f"Improvement={avg_improvement:.4f}, Ranking={avg_ranking:.4f}, "
                      f"Proximity={avg_proximity:.4f}")

    return policy, value_net, reward_history


# ---------- Tournament Selection ----------
def tournament_selection(population, fitnesses, tournament_size=3):
    tournament_idx = random.sample(range(len(population)), min(tournament_size, len(population)))
    tournament_fitnesses = [fitnesses[i] for i in tournament_idx]
    winner_idx = tournament_idx[np.argmin(tournament_fitnesses)]
    return population[winner_idx]


# ---------- Standard Operations ----------
def standard_crossover_mutation(parent1, parent2, mutation_rate=0.1, mutation_strength=0.2):
    # Uniform crossover
    child = []
    for i in range(len(parent1)):
        if random.random() < 0.5:
            child.append(parent1[i])
        else:
            child.append(parent2[i])

    # Gaussian mutation with adaptive strength based on distance from optimum
    for i in range(len(child)):
        if random.random() < mutation_rate:
            # Slightly larger mutation for Ackley to help escape local minima
            child[i] += np.random.normal(0, mutation_strength)
            child[i] = np.clip(child[i], -5,5)

    return child


# ---------- Improved RL-Guided GA ----------
def improved_rl_ga(policy, dim=10, pop_size=50, generations=150, rl_ratio=1.0):
    population = [np.random.uniform(-5,5, dim) for _ in range(pop_size)]
    best_history = []
    mean_history = []

    policy.eval()

    for gen in range(generations):
        fitnesses = np.array([ackley(ind) for ind in population])
        best_history.append(fitnesses.min())
        mean_history.append(fitnesses.mean())

        new_population = []

        # Strong elitism
        elite_count = max(2, int(pop_size * 0.15))
        sorted_idx = np.argsort(fitnesses)
        for i in range(elite_count):
            new_population.append(population[sorted_idx[i]])

        # Generate rest of population
        while len(new_population) < pop_size:
            parent1 = tournament_selection(population, fitnesses, 3)
            parent2 = tournament_selection(population, fitnesses, 3)

            if random.random() < rl_ratio:
                # Use RL policy with improved state
                state = create_state(parent1, parent2, population, fitnesses)

                with torch.no_grad():
                    action, _, _ = policy.sample_action(state.unsqueeze(0))
                    child = torch.clamp(action.squeeze(0), -5,5).numpy()
                    new_population.append(child.tolist())
            else:
                # Use standard operations
                child = standard_crossover_mutation(parent1, parent2)
                new_population.append(child)

        population = new_population

    return best_history, mean_history ,population


# ---------- Standard GA ----------
def standard_ga(dim=10, pop_size=50, generations=150):
    population = [np.random.uniform(-5,5, dim) for _ in range(pop_size)]
    best_history = []
    mean_history = []

    for gen in range(generations):
        fitnesses = np.array([ackley(ind) for ind in population])
        best_history.append(fitnesses.min())
        mean_history.append(fitnesses.mean())

        new_population = []

        # Elitism
        elite_count = max(2, int(pop_size * 0.15))
        sorted_idx = np.argsort(fitnesses)
        for i in range(elite_count):
            new_population.append(population[sorted_idx[i]])

        while len(new_population) < pop_size:
            parent1 = tournament_selection(population, fitnesses, 3)
            parent2 = tournament_selection(population, fitnesses, 3)
            child = standard_crossover_mutation(parent1, parent2)
            new_population.append(child)

        population = new_population

    return best_history, mean_history ,population


# ---------- Main Execution ----------
if __name__ == "__main__":
    dim = 10
    generations = 150

    print("Training improved RL agent for Ackley function...")
    policy, value_net, reward_history = train_improved_rl(dim=dim, training_episodes=50000)

    # Plot training progress
    plt.figure(figsize=(16, 5))
    plt.subplot(1, 3, 1)
    plt.plot(reward_history)
    plt.title("RL Training Progress",fontsize=18)
    plt.xlabel("Episode",fontsize=18)
    plt.ylabel("Average Reward",fontsize=20)
    plt.grid(True, alpha=0.3)

    print("Running comparisons...")
    best_ga, mean_ga ,popga = standard_ga(dim=dim, generations=generations)
    best_rl, mean_rl,poprl = improved_rl_ga(policy, dim=dim, generations=generations)

    # Plot comparison
    plt.subplot(1, 3, 2)
    plt.plot(best_ga, label="Standard GA", color='blue', linewidth=2)
    plt.plot(best_rl, label="EARL", color='red', linewidth=2)
    plt.xlabel("Generation",fontsize=18)
    plt.ylabel("Best Fitness",fontsize=18)
    plt.title("Performance Comparison",fontsize=20)
    plt.legend(fontsize=18)
    plt.grid(True, alpha=0.3)
    plt.yscale('log')  # Log scale helps visualize Ackley convergence

    # Plot final best solutions
    plt.subplot(1, 3, 3)
    final_ga_best_idx = np.argmin([ackley(ind) for ind in
                                   [np.random.uniform(-5,5, dim) for _ in range(50)]])

    # Show convergence to optimum
    distances_ga = [np.linalg.norm(np.array(ind)) for ind in popga]
    distances_rl = [np.linalg.norm(np.array(ind)) for ind in poprl]

    plt.hist(distances_ga, alpha=0.5, label='Standard GA', bins=15)
    plt.hist(distances_rl, alpha=0.5, label='EARL', bins=15)
    plt.xlabel("Distance from Optimum",fontsize=18)
    plt.ylabel("Frequency",fontsize=18)
    plt.title("Final Population Distribution",fontsize=20)
    plt.legend(fontsize=16)
    plt.grid(True, alpha=0.3)

    plt.tight_layout()
    plt.show()

    print(f"\nResults for Ackley Function (optimum = 0):")
    print(f"Standard GA - Best: {min(best_ga):.6f}, Final: {best_ga[-1]:.6f}")
    print(f"RL-Guided GA - Best: {min(best_rl):.6f}, Final: {best_rl[-1]:.6f}")

    if min(best_rl) < min(best_ga):
        improvement = ((min(best_ga) - min(best_rl)) / (min(best_ga) + 1e-8)) * 100
        print(f"RL improvement: {improvement:.2f}%")
    else:
        degradation = ((min(best_rl) - min(best_ga)) / (min(best_ga) + 1e-8)) * 100
        print(f"RL degradation: {degradation:.2f}%")

    print(f"\nClosest to optimum:")
    print(f"Standard GA: {min(best_ga):.8f}")
    print(f"RL-GA: {min(best_rl):.8f}")