from sklearn.linear_model import LogisticRegression
from sklearn.svm import SVC, LinearSVC
from sklearn.neural_network import MLPClassifier
from sklearn.ensemble import GradientBoostingClassifier, AdaBoostClassifier


def get_classifier(classifier_name, **kwargs):
    """
    Returns a classifier instance based on the specified name.
    
    Parameters:
    -----------
    classifier_name : str
        Name of the classifier to use. Options are:
        - 'logistic': Logistic Regression
        - 'linear_svm': Linear Support Vector Machine
        - 'nonlinear_svm': Non-linear Support Vector Machine (RBF kernel)
        - 'mlp': Multi-Layer Perceptron
        - 'gbm': Gradient Boosting Machine
        - 'adaboost': AdaBoost
    
    **kwargs : dict
        Additional parameters to pass to the classifier constructor.
        
    Returns:
    --------
    classifier : sklearn.base.BaseEstimator
        Instance of the specified classifier
    """
    default_params = {
        'logistic': {'max_iter': 1000},
        'linear_svm': {'max_iter': 1000},
        'nonlinear_svm': {'kernel': 'rbf', 'gamma': 0.5},
        'mlp': {'hidden_layer_sizes': (10,10), 'max_iter': 1000, 'solver': 'lbfgs', 'tol': 1e-4},
        'gbm': {'n_estimators': 100, 'learning_rate': 0.1},
        'adaboost': {'n_estimators': 100, 'learning_rate': 1.0}
    }
    
    # Update default parameters with user-specified ones
    params = default_params.get(classifier_name, {}).copy()
    params.update(kwargs)
    
    classifiers = {
        'logistic': LogisticRegression,
        'linear_svm': LinearSVC,
        'nonlinear_svm': SVC,
        'mlp': MLPClassifier,
        'gbm': GradientBoostingClassifier,
        'adaboost': AdaBoostClassifier
    }
    
    if classifier_name not in classifiers:
        raise ValueError(f"Unknown classifier: {classifier_name}. Choose from: {list(classifiers.keys())}")
    
    return classifiers[classifier_name](**params) 
