"""
Generate plots for different sampling schedules.
"""

import argparse
import logging
import matplotlib.pyplot as plt
import scienceplots

plt.style.use("science")

logging.basicConfig(level=logging.INFO)

from analysis_engine import AnalysisEngine
from config import ANALYSIS_TYPES


def main(fail: bool = False, analysis_types=None, schedule: str = 'end', output_dir: str|None = None):
    """
    Main function to run analysis.

    Parameters
    ----------
    fail : bool
        Whether to fail on errors (for debugging)
    analysis_types : list of str, optional
        List of analysis types to run. If None, runs all available types.
    schedule : str
        Schedule to run
    output_dir : str, optional
        Directory to save plots to. If None, uses PLOT_OUTPUT_DIR.
    """
    # Initialize the analysis engine with output directory
    engine = AnalysisEngine(output_dir=output_dir)

    if analysis_types is None:
        analysis_types = ANALYSIS_TYPES

    available_types = set(engine._plotters.keys())
    invalid_types = set(analysis_types) - available_types
    if invalid_types:
        logging.error(f"Invalid analysis types: {invalid_types}")
        logging.error(f"Available types: {available_types}")
        return

    logging.info("Starting full analysis pipeline...")
    results = engine.run_full_analysis(analysis_types, fail_on_error=fail, schedule=schedule)

    total_plots = sum(len(plots) for plots in results.values())
    logging.info(f"Analysis completed. Generated {total_plots} plots across {len(results)} analysis types.")

    for analysis_type, plots in results.items():
        if plots:
            logging.info(f"  {analysis_type}: {len(plots)} plots")
        else:
            logging.warning(f"  {analysis_type}: No plots generated")


if __name__ == "__main__":
    parser = argparse.ArgumentParser(description='Generate plots for paper')
    parser.add_argument('--fail', action='store_true',
                       help='Fail on individual analysis errors (for debugging)')
    parser.add_argument('--analysis_types', '-p', nargs='+',
                       help='Analysis types to run',
                       choices=ANALYSIS_TYPES,
                       default=None)
    parser.add_argument('--schedule', type=str,
                       help='Schedule to run',
                       choices=['end', 'uniform', 'start', 'linear', 'block', 'pair'],
                       default='end')
    parser.add_argument('--output_dir', '-o', type=str,
                       help='Directory to save plots to',
                       default=None)

    args = parser.parse_args()

    main(args.fail, args.analysis_types, args.schedule, args.output_dir)