# Flow Matching Fine-tuning Research Project

This project investigates fine-tuning techniques based on the Flow Matching algorithm, covering two main application scenarios: image generation and robotic manipulation. The project is divided into two core modules: CIFAR-10 image generation experiments and robotic manipulation control experiments.

## Project Overview

### Research Objectives
- Explore the application effectiveness of Flow Matching algorithms in different domains
- Investigate the performance improvement of Maximum Likelihood Estimation (MLE) fine-tuning techniques
- Implement Residual Fine-tuning methods
- Validate algorithm effectiveness in multiple robotic manipulation scenarios

### Project Structure
```
.
├── preliminary_exp/          # CIFAR-10 image generation experiments
│   ├── train_cifar10_ddp*.py    # Distributed training scripts
│   ├── compute_fid*.py          # FID evaluation scripts
│   ├── utils_cifar.py           # Utility functions
│   └── requirements.txt         # Dependencies
└── robotic_manipulation/    # Robotic manipulation experiments
    ├── run.py                   # Training entry script
    ├── test.py                  # Testing script
    ├── test_res_finetune.py    # Residual fine-tuning testing script
    ├── lightning_module.py      # PyTorch Lightning module
    ├── finetune_mlp.py         # MLP fine-tuning script
    ├── finetune_contract.py    # Contract-based fine-tuning script
    ├── conf/                    # Hydra configuration directory
    │   ├── config.yaml          # Main configuration file
    │   ├── experiment/          # Experiment environment configurations
    │   │   ├── kitchen/         # Kitchen environment
    │   │   ├── pusht/           # Block pushing task
    │   │   └── mimic/           # Imitation learning
    │   └── model/               # Model configurations
    ├── external/                # External dependencies
    ├── utils/                   # Utility modules
    └── requirements.txt         # Dependencies
```

## Experimental Scenarios

### 1. CIFAR-10 Image Generation
**Objective**: Validate Flow Matching algorithm's image generation capability on CIFAR-10 dataset

**Key Features**:
- Support for multiple Flow Matching variants: OT-CFM, I-CFM, FM, SI
- Distributed Data Parallel (DDP) training
- Maximum Likelihood Estimation (MLE) fine-tuning
- Residual Fine-tuning
- FID score evaluation

**Key Scripts**:
- `train_cifar10_ddp3.py`: MLE fine-tuning implementation
- `train_cifar10_ddp4.py`: Residual fine-tuning implementation
- `compute_fid.py`: FID score computation

### 2. Robotic Manipulation Control
**Objective**: Apply Flow Matching algorithm to robotic manipulation tasks

**Supported Scenarios**:
- **Push-T**: Block pushing task
- **Franka Kitchen**: Kitchen robot control tasks
- **Robomimic**: Dual-arm robot imitation learning

**Key Features**:
- Unified framework based on PyTorch Lightning and Hydra
- Multi-environment support
- Conditional generation (observation-conditioned action generation)
- Multi-GPU distributed training
- Mixed precision training

## Technical Architecture

### Flow Matching Algorithm
The project implements multiple Flow Matching variants:
- **OT-CFM** (Optimal Transport Conditional Flow Matching)
- **I-CFM** (Independent Conditional Flow Matching)
- **FM** (Target Conditional Flow Matching)
- **SI** (Score-based Integration)

### Model Architecture
- **UNet**: Diffusion model-based UNet architecture
- **Transformer**: Attention-based sequence model
- **Conditional Encoding**: Support for observation conditioning
- **Visual Encoding**: ResNet-based image encoding (PUSH-T environment)

### Fine-tuning Techniques
1. **Maximum Likelihood Estimation (MLE) Fine-tuning**
   - Continue training on pre-trained models
   - Optimize models using MLE objective function
   - Improve sample quality and generation stability

2. **Residual Fine-tuning**
   - Two-stage generation process
   - Pre-trained model generates intermediate representations
   - Residual model refines the output
   - Support for incremental model improvements

## Quick Start

### Environment Setup

#### CIFAR-10 Experiment Environment
```bash
# Create conda environment
conda create -n fm_c python=3.9 -y
conda activate fm_c

# Install dependencies
cd preliminary_exp
pip install -r requirements.txt
```

#### Robotic Manipulation Experiment Environment
```bash
# Install dependencies
cd robotic_manipulation
pip install -r requirements.txt
```

### Dataset Setup

#### CIFAR-10 Dataset
The CIFAR-10 dataset will be automatically downloaded by PyTorch when running the training scripts. No manual setup is required.

#### Robotic Manipulation Datasets
For robotic manipulation experiments, you need to download and place the datasets in the following structure:

```
data/
├── pusht/
│   └── pusht_cchi_v7_replay.zarr
├── kitchen/
│   └── dataset/
└── mimic/
    └── dataset/
```

**Dataset Sources**:
- **Push-T**: Download from the original Push-T dataset repository
- **Kitchen**: Franka Kitchen dataset from the original source
- **Mimic**: Robomimic dataset from the official repository

**Note**: Update the dataset paths in the configuration files if you place datasets in different locations.

### Running Examples

#### CIFAR-10 Training
```bash
# Single GPU training
python train_cifar10_ddp3.py --output_dir ./results/ --batch_size 128

# Multi-GPU distributed training
torchrun --nproc_per_node=4 train_cifar10_ddp3.py --parallel True --batch_size 32

# Residual fine-tuning
python train_cifar10_ddp4.py \
    --pretrained_path ./pretrained_model.pt \
    --res_model_type unet \
    --lr 1e-5 \
    --output_dir ./residual_results/
```

#### Robotic Manipulation Training
```bash
# Default configuration training (Kitchen environment)
python run.py

# Specific environment training
python run.py env=pusht
python run.py env=mimic

# Custom parameter training
python run.py env=kitchen training.epochs=1000 training.batch_size=128

# MLE fine-tuning
python run.py execution=mle_finetune checkpoint.load_checkpoint=path/to/model.pth
```

### Model Evaluation

#### CIFAR-10 FID Evaluation
```bash
python compute_fid.py \
    --input_dir ./results/ \
    --step 400000 \
    --integration_steps 100 \
    --integration_method dopri5
```

#### Robotic Manipulation Testing
```bash
# Test model
python test.py

# Test specific checkpoint
python test.py checkpoint.load_checkpoint=path/to/checkpoint.pth
```

## Configuration Guide

### CIFAR-10 Experiment Configuration
Configure training parameters via command-line arguments:
- `--batch_size`: Batch size
- `--lr`: Learning rate
- `--output_dir`: Output directory
- `--parallel`: Whether to use distributed training

### Robotic Manipulation Configuration
Using Hydra configuration system, main configuration groups:
- **env**: Environment configuration (kitchen, pusht, mimic)
- **execution**: Execution mode (fm_train, test, mle_finetune)
- **model**: Model type (unet, transformer)
- **training**: Training parameters

Example configuration:
```yaml
env: kitchen
execution: fm_train
model: unet
training:
  epochs: 5000
  batch_size: 64
  learning_rate: 1.0e-4
device: cuda
```

## Experimental Outputs

### CIFAR-10 Experiment Output Structure
```
results/
├── model_type/ (e.g., otcfm/)
│   ├── tensorboard_logs/        # Training metrics
│   ├── model_weights_step_*.pt  # Checkpoints
│   ├── training_losses.json     # Loss history
│   ├── fid_scores.json          # FID evaluation results
│   └── generated_samples_*.png  # Generated samples
```

### Robotic Manipulation Experiment Outputs
- **Checkpoints**: Saved in `checkpoint/{env_name}/` directory
- **Logs**: TensorBoard logs saved in `logs/` directory
- **Outputs**: Hydra run outputs saved in `outputs/` directory

## Performance Metrics

### CIFAR-10 Evaluation Metrics
- **FID Score**: Fréchet Inception Distance
- **Generated Sample Quality**: Visual evaluation
- **Training Loss**: Convergence curves

### Robotic Manipulation Evaluation Metrics
- **Total Reward**: Cumulative reward value
- **Reward Rate**: Ratio relative to maximum possible reward
- **Task Completion Rate**: Rate of successful task completion

## Advanced Features

### Hyperparameter Search
```bash
# Use Hydra's multirun functionality
python run.py -m training.learning_rate=1e-4,5e-4,1e-3 training.batch_size=32,64,128
```

### Debug Mode
```bash
# Enable verbose logging
python run.py logging.level=DEBUG

# Single GPU debugging
python run.py training.gpus=1
```

## Dependencies

### Core Dependencies
- PyTorch >= 2.0.0
- PyTorch Lightning >= 1.7.0, < 2.0.0
- torchcfm (Conditional Flow Matching library)
- torchdyn (Neural ODE integration)
- Hydra (Configuration management)

### Optional Dependencies
- CleanFID (FID computation)
- WandB (Experiment tracking)
- TensorBoard (Visualization)

## Troubleshooting

### Common Issues

1. **Dependency Conflicts**: Ensure using specified PyTorch Lightning version (<2.0.0)
2. **GPU Memory Issues**: Reduce batch_size or use gradient accumulation
3. **Dataset Path Errors**: Set correct dataset paths in environment configuration

### Memory Optimization
- Use chunked processing for large batches
- Enable mixed precision training
- Regular garbage collection

## Citation

If you use this code, please cite the relevant papers:

```bibtex
@article{flow_matching,
  title={Flow Matching for Generative Modeling},
  author={Lipman, Yaron and Chen, Ricky T. Q. and Ben-Hamu, Heli and Nickel, Maximilian and Le, Matt},
  journal={arXiv preprint arXiv:2210.02747},
  year={2022}
}
```

## License

This project is based on open-source code, licensed under BSD 3-Clause License.

## Contributing

Issues and Pull Requests are welcome to improve this project.

## Contact

For questions, please contact us through GitHub Issues.
