# Preliminary Experiments for Conditional Flow Matching

This folder contains preliminary experiments and development scripts for Conditional Flow Matching (CFM) models, specifically focused on CIFAR-10 image generation.

## Overview

The scripts in this folder represent various experimental approaches to training and evaluating Conditional Flow Matching models, including:

- **Maximum Likelihood Estimation (MLE) fine-tuning experiments**
- **Residual fine-tuning approaches**
- **Distributed Data Parallel (DDP) training implementations**
- **FID score computation utilities**

## File Descriptions

### Training Scripts

#### `my_train_cifar10_ddp.py` - Original DDP Training Script
- Basic distributed training implementation for CFM models
- Supports multiple flow matching variants (OT-CFM, I-CFM, FM, SI)
- Includes EMA (Exponential Moving Average) model tracking
- TensorBoard logging for training metrics

#### `my_train_cifar10_ddp2.py` - Enhanced DDP Training
- Improved version with better memory management
- Additional training configurations and optimizations

#### `my_train_cifar10_ddp3.py` - MLE Fine-tuning Implementation
- **Maximum Likelihood Estimation fine-tuning approach**
- Continues training from checkpoints with MLE objective
- Enhanced loss tracking and incremental saving
- Memory-efficient chunked processing for large batches

#### `my_train_cifar10_ddp4.py` - Residual Fine-tuning Implementation
- **Two-stage residual fine-tuning approach**
- Uses a pretrained model as the first stage
- Adds a residual model for second-stage refinement
- Supports different residual model architectures (MLP, UNet variants)
- Memory-optimized training with explicit garbage collection

### Utility Scripts

#### `my_compute_fid.py` - FID Score Computation
- Computes Fréchet Inception Distance (FID) for model evaluation
- Supports different integration methods (Euler, dopri5)
- Configurable batch sizes and sample counts
- CleanFID integration for reliable metrics

#### `my_compute_fid_res.py` - Residual Model FID Computation
- Specialized FID computation for residual fine-tuning models
- Handles two-stage generation process

#### `utils_cifar.py` - Common Utilities
- Distributed training setup functions
- Model EMA (Exponential Moving Average) implementation
- Sample generation utilities
- Data loading helpers

## Key Features

### Flow Matching Variants Supported
- **OT-CFM** (Optimal Transport Conditional Flow Matching)
- **I-CFM** (Independent Conditional Flow Matching)
- **FM** (Target Conditional Flow Matching)
- **SI** (Score-based Integration)

### Training Features
- **Distributed Data Parallel (DDP)** training support
- **Exponential Moving Average (EMA)** model tracking
- **TensorBoard** logging for real-time monitoring
- **Checkpoint resuming** capability
- **Memory optimization** with chunked processing
- **Incremental loss/FID saving** to prevent data loss

### Evaluation Features
- **FID score computation** using CleanFID
- **Sample generation** for visual inspection
- **Training loss visualization** with automatic plotting
- **Multiple integration methods** for ODE solving

## Usage Examples

### Basic Training
```bash
# Single GPU training
python my_train_cifar10_ddp3.py --output_dir ./results/ --batch_size 128

# Multi-GPU training (DDP)
torchrun --nproc_per_node=4 my_train_cifar10_ddp3.py --parallel True --batch_size 32
```

### Residual Fine-tuning
```bash
python my_train_cifar10_ddp4.py \
    --pretrained_path ./pretrained_model.pt \
    --res_model_type unet \
    --lr 1e-5 \
    --output_dir ./residual_results/
```

### FID Computation
```bash
python my_compute_fid.py \
    --input_dir ./results/ \
    --step 400000 \
    --integration_steps 100 \
    --integration_method dopri5
```

## Experimental Approaches

### 1. Maximum Likelihood Estimation (MLE) Fine-tuning
- Continues training with MLE objective after initial CFM training
- Aims to improve sample quality through likelihood maximization
- Implemented in `my_train_cifar10_ddp3.py`

### 2. Residual Fine-tuning
- Two-stage generation process:
  1. Pretrained model generates intermediate representations
  2. Residual model refines the output
- Allows for incremental model improvements
- Implemented in `my_train_cifar10_ddp4.py`

## Model Architecture

All experiments use a UNet-based architecture with:
- **Base channels**: 128 (configurable)
- **Residual blocks**: 2 per resolution level
- **Channel multipliers**: [1, 2, 2, 2] for multi-scale processing
- **Attention mechanisms**: At 16x16 resolution
- **Dropout**: 0.1 for regularization

## Output Structure

Training scripts generate the following outputs:
```
results/
├── model_type/ (e.g., otcfm/)
│   ├── tensorboard_logs/        # Training metrics
│   ├── model_weights_step_*.pt  # Checkpoints
│   ├── training_losses.json     # Loss history
│   ├── fid_scores.json          # FID evaluation results
│   └── generated_samples_*.png  # Visual samples
```

## Environment Setup

### Creating the Conda Environment

Create a new conda environment named `fm_c` with the required dependencies:

```bash
# Create conda environment
conda create -n fm_c python=3.9 -y

# Activate the environment
conda activate fm_c

# Install dependencies from requirements.txt
pip install -r requirements.txt
```



## Notes

- These are **preliminary experiments** and may contain experimental code
- Memory management is optimized for large-scale training
- All scripts include extensive logging and error handling
- Results should be validated with multiple runs for statistical significance

## Related Files

The main project implementation can be found in the parent directory's `examples/images/cifar10/` folder, which contains the production-ready versions of these scripts.
