# Robotic Manipulation with Flow Matching

This module implements Flow Matching algorithms for robotic manipulation tasks, supporting multiple environments with a unified framework based on PyTorch Lightning and Hydra.

## Supported Environments

- **Kitchen**: Kitchen robot control tasks
- **Push-T**: Block pushing task environment  
- **Mimic**: Dual-arm robot imitation learning

## Quick Start

### Installation
```bash
pip install -r requirements.txt
```

### Training
```bash
# Default training (Kitchen environment)
python run.py

# Specific environment training
python run.py env=pusht
python run.py env=mimic

# Custom parameters
python run.py env=kitchen training.epochs=1000 training.batch_size=128
```

### Testing
```bash
# Test with default configuration
python test.py

# Test specific checkpoint
python test.py checkpoint.load_checkpoint=path/to/checkpoint.pth
```

## Fine-tuning Methods

### 1. MLE Fine-tuning
```bash
python run.py execution=mle_finetune checkpoint.load_checkpoint=path/to/model.pth
```

### 2. Residual Fine-tuning
```bash
python test_res_finetune.py
```

### 3. Contract-based Fine-tuning
```bash
python finetune_contract.py --checkpoint path/to/pretrained_model.pth --mode train
```

## Project Structure

```
robotic_manipulation/
├── run.py                    # Main training script
├── test.py                   # Testing script
├── test_res_finetune.py     # Residual fine-tuning testing
├── finetune_contract.py     # Contract-based fine-tuning
├── finetune_mlp.py          # MLP fine-tuning
├── lightning_module.py      # PyTorch Lightning module
├── conf/                    # Hydra configurations
│   ├── config.yaml          # Main configuration
│   ├── experiment/          # Environment configs
│   └── model/               # Model configs
├── external/                # External dependencies
├── utils/                   # Utility modules
└── requirements.txt         # Dependencies
```

## Configuration

The project uses Hydra for configuration management. Key configuration groups:

- `env`: Environment settings (kitchen, pusht, mimic)
- `execution`: Execution mode (fm_train, test, mle_finetune)
- `model`: Model architecture (unet, transformer)
- `training`: Training parameters

Example configuration:
```yaml
env: kitchen
execution: fm_train
model: unet
training:
  epochs: 5000
  batch_size: 64
  learning_rate: 1.0e-4
```

## Dataset Setup

Place datasets in the following structure:
```
data/
├── pusht/pusht_cchi_v7_replay.zarr
├── kitchen/dataset/
└── mimic/dataset/
```

## Features

- Multi-GPU distributed training
- Mixed precision training
- Conditional generation (observation-conditioned actions)
- Multiple Flow Matching variants
- Physics-informed regularization
- TensorBoard logging

## Troubleshooting

- Ensure PyTorch Lightning version < 2.0.0
- Set correct dataset paths in configuration files
- Use appropriate batch sizes for GPU memory

For detailed documentation, see the main project README.md.
