import torch
import torch.nn as nn
import torch.nn.functional as F
from torch import Tensor
from typing import Type, Any, Callable, Union, List, Optional, Tuple
from torchvision.models.resnet import BasicBlock, Bottleneck, conv1x1, ResNet18_Weights, ResNet34_Weights, handle_legacy_interface, _ovewrite_named_param, WeightsEnum

class ResNetAE(nn.Module):

    def __init__(
        self,
        block: Type[Union[BasicBlock, Bottleneck]],
        layers: List[int],
        num_classes: int = 1000,
        stem_end_block: int = 31, 
        zero_init_residual: bool = False,
        groups: int = 1,
        width_per_group: int = 64,
        replace_stride_with_dilation: Optional[List[bool]] = None,
        norm_layer: Optional[Callable[..., nn.Module]] = None
    ) -> None:
        super(ResNetAE, self).__init__()
        if norm_layer is None:
            norm_layer = nn.BatchNorm2d
        self._norm_layer = norm_layer

        self.inplanes = 64
        self.dilation = 1
        if replace_stride_with_dilation is None:
            # each element in the tuple indicates if we should replace
            # the 2x2 stride with a dilated convolution instead
            replace_stride_with_dilation = [False, False, False]
        if len(replace_stride_with_dilation) != 3:
            raise ValueError("replace_stride_with_dilation should be None "
                             "or a 3-element tuple, got {}".format(replace_stride_with_dilation))
        self.groups = groups
        self.base_width = width_per_group
        self.conv1 = nn.Conv2d(3, self.inplanes, kernel_size=7, stride=2, padding=3,
                               bias=False)
        self.bn1 = norm_layer(self.inplanes)
        self.relu = nn.ReLU(inplace=True)
        self.maxpool = nn.MaxPool2d(kernel_size=3, stride=2, padding=1)
        self.layer1 = self._make_layer(block, 64, layers[0])
        self.layer2 = self._make_layer(block, 128, layers[1], stride=2,
                                       dilate=replace_stride_with_dilation[0])
        self.layer3 = self._make_layer(block, 256, layers[2], stride=2,
                                       dilate=replace_stride_with_dilation[1])
        self.layer4 = self._make_layer(block, 512, layers[3], stride=2,
                                       dilate=replace_stride_with_dilation[2])
        self.avgpool = nn.AdaptiveAvgPool2d((1, 1))
        self.fc = nn.Linear(512 * block.expansion, num_classes)

        self.layer4_aux = self._make_layer(block, 512, layers[3], stride=2,
                                       dilate=replace_stride_with_dilation[2])
        self.fc_aux = nn.Linear(512 * block.expansion, num_classes)

        self.stem_end_block = stem_end_block
        self.stem_end_blocks = [(1+2*(layers[0]+layers[1]+layers[2]) + i*2) for i in range(layers[3]+1)]
        
        # set the second last block as the stem end block by default
        if stem_end_block is None:
            self.stem_end_block = self.stem_end_blocks[-2]

        if self.stem_end_block == self.stem_end_blocks[0]:
            self.decoder1 = nn.Sequential(
                nn.ConvTranspose2d(256, 128, 4, stride=2, padding=1),  
                nn.ConvTranspose2d(128, 128, 4, stride=2, padding=1), 
                nn.InstanceNorm2d(128), 
                nn.ReLU(),
            )
            self.decoder2 = nn.Sequential(
                nn.ConvTranspose2d(128, 64, 4, stride=2, padding=1),  
                nn.InstanceNorm2d(64), 
                nn.ReLU(),
            )
            self.decoder3 = nn.Sequential(
                nn.ConvTranspose2d(64, 3, 4, stride=2, padding=1), 
                nn.Sigmoid(),
            )
            self.decoder = nn.Sequential(
                self.decoder1, self.decoder2, self.decoder3
            )
        else:
            self.decoder1 = nn.Sequential(
                nn.ConvTranspose2d(512, 256, 4, stride=2, padding=1), 
                nn.ConvTranspose2d(256, 128, 4, stride=2, padding=1),
                nn.InstanceNorm2d(256), 
                nn.ReLU(),
            )
            self.decoder2 = nn.Sequential(
                nn.ConvTranspose2d(128, 64, 4, stride=2, padding=1),  
                nn.ConvTranspose2d(64, 32, 4, stride=2, padding=1), 
                nn.InstanceNorm2d(128), 
                nn.ReLU(),
            )
            self.decoder3 = nn.Sequential(
                nn.ConvTranspose2d(32, 3, 4, stride=2, padding=1),  
                nn.Sigmoid(),
            )
            self.decoder = nn.Sequential(
                self.decoder1, self.decoder2, self.decoder3
            )

        for m in self.modules():
            if isinstance(m, nn.Conv2d):
                nn.init.kaiming_normal_(m.weight, mode='fan_out', nonlinearity='relu')
            elif isinstance(m, (nn.BatchNorm2d, nn.GroupNorm)):
                nn.init.constant_(m.weight, 1)
                nn.init.constant_(m.bias, 0)

        # Zero-initialize the last BN in each residual branch,
        # so that the residual branch starts with zeros, and each residual block behaves like an identity.
        # This improves the model by 0.2~0.3% according to https://arxiv.org/abs/1706.02677
        if zero_init_residual:
            for m in self.modules():
                if isinstance(m, Bottleneck):
                    nn.init.constant_(m.bn3.weight, 0)  # type: ignore[arg-type]
                elif isinstance(m, BasicBlock):
                    nn.init.constant_(m.bn2.weight, 0)  # type: ignore[arg-type]

    def _make_layer(self, block: Type[Union[BasicBlock, Bottleneck]], planes: int, blocks: int,
                    stride: int = 1, dilate: bool = False) -> nn.Sequential:
        norm_layer = self._norm_layer
        downsample = None
        previous_dilation = self.dilation
        if dilate:
            self.dilation *= stride
            stride = 1
        if stride != 1 or self.inplanes != planes * block.expansion:
            downsample = nn.Sequential(
                conv1x1(self.inplanes, planes * block.expansion, stride),
                norm_layer(planes * block.expansion),
            )

        layers = []
        layers.append(block(self.inplanes, planes, stride, downsample, self.groups,
                            self.base_width, previous_dilation, norm_layer))
        self.inplanes = planes * block.expansion
        for _ in range(1, blocks):
            layers.append(block(self.inplanes, planes, groups=self.groups,
                                base_width=self.base_width, dilation=self.dilation,
                                norm_layer=norm_layer))

        return nn.Sequential(*layers)

    def forward_stem(self, x: Tensor) -> Tensor:
        # See note [TorchScript super()]
        x = self.conv1(x)
        x = self.bn1(x)
        x = self.relu(x)
        x = self.maxpool(x)

        x = self.layer1(x)
        x = self.layer2(x)
        x = self.layer3(x)
        
        stem_out = x

        for i in range(1, len(self.stem_end_blocks)):
            block = self.stem_end_blocks[i]
            if self.stem_end_block >= block:
                stem_out = self.layer4[i-1](stem_out)
            else:
                break

        # if self.stem_end_block == self.stem_end_blocks[0]:
        #     stem_out = x
        # elif self.stem_end_block == 29:
        #     stem_out = self.layer4[0](x)
        # elif self.stem_end_block == 31:
        #     x = self.layer4[0](x)
        #     stem_out = self.layer4[1](x)
        # elif self.stem_end_block == 33:
        #     stem_out = self.layer4(x)

        return stem_out

    def forward_main_branch(self, stem_out: Tensor) -> Tensor:
        
        x = stem_out
        for i in range(1, len(self.stem_end_blocks)):
            block = self.stem_end_blocks[i]
            if self.stem_end_block >= block:
                continue
            x = self.layer4[i-1](x)
        
        # if self.stem_end_block == self.stem_end_blocks[0]:
        #     x = self.layer4(stem_out)
        # elif self.stem_end_block == 29:
        #     x = self.layer4[1](stem_out)
        #     x = self.layer4[2](x)
        # elif self.stem_end_block == 31:
        #     x = self.layer4[2](stem_out)
        # elif self.stem_end_block == 33:
        #     x = stem_out

        x = self.avgpool(x)
        x = torch.flatten(x, 1)
        x = self.fc(x)

        return x

    def forward_ae_branch(self, stem_out: Tensor) -> Tensor:

        # print(stem_out.shape)
        # x = self.decoder1(stem_out)
        # print(x.shape)
        # x = self.decoder2(x)
        # print(x.shape)
        # rec = self.decoder3(x)
        # print(rec.shape)

        rec = self.decoder(stem_out)

        return rec

    def forward(self, x: Tensor) -> Type[Tuple[Tensor, Tensor]]:
        stem_out = self.forward_stem(x)
        logits = self.forward_main_branch(stem_out)
        rec = self.forward_ae_branch(stem_out)
        return logits, rec

# def ResNet34AE(num_classes=1000, stem_end_block=31):
#     return ResNetAE(block=BasicBlock, layers=[3, 4, 6, 3], num_classes=num_classes, stem_end_block=stem_end_block)

# def ResNet18AE(num_classes=1000, stem_end_block=15):
#     return ResNetAE(block=BasicBlock, layers=[2, 2, 2, 2], num_classes=num_classes, stem_end_block=stem_end_block)

@handle_legacy_interface(weights=("pretrained", ResNet18_Weights.IMAGENET1K_V1))
def resnet18ae(*, weights: Optional[ResNet18_Weights] = None, progress: bool = True, **kwargs: Any):
    weights = ResNet18_Weights.verify(weights)

    return _resnetae(BasicBlock, [2, 2, 2, 2], weights, progress, **kwargs)


@handle_legacy_interface(weights=("pretrained", ResNet34_Weights.IMAGENET1K_V1))
def resnet34ae(*, weights: Optional[ResNet34_Weights] = None, progress: bool = True, **kwargs: Any):
    weights = ResNet34_Weights.verify(weights)

    return _resnetae(BasicBlock, [3, 4, 6, 3], weights, progress, **kwargs)

def _resnetae(
    block: Type[Union[BasicBlock, Bottleneck]],
    layers: List[int],
    weights: Optional[WeightsEnum],
    progress: bool,
    **kwargs: Any,
):
    if weights is not None:
        _ovewrite_named_param(kwargs, "num_classes", len(weights.meta["categories"]))

    model = ResNetAE(block, layers, **kwargs)

    if weights is not None:
        model.load_state_dict(weights.get_state_dict(progress=progress))

    return model

if __name__ == '__main__':
    model = resnet34ae(stem_end_block=31)
    # for name, p in model.named_parameters():
    #     print(name, p.shape)

    imgs = torch.zeros((2,3,224,224))
    logits, imgs_rec = model(imgs)

    print(logits.shape)
    print(imgs_rec.shape)