import numpy as np
import torch
import random
import argparse
import wandb

import learners

if __name__ == '__main__':
    parser = argparse.ArgumentParser(description='RDBench')

    # training
    parser.add_argument("--batch_size", help="batch_size", default=256, type=int)
    parser.add_argument("--lr",help='learning rate',default=1e-6, type=float)
    parser.add_argument("--weight_decay",help='weight_decay',default=0.0, type=float)
    parser.add_argument("--momentum",help='momentum',default=0.9, type=float)
    parser.add_argument("--num_workers", help="workers number", default=16, type=int)
    parser.add_argument("--exp", help='wandb project name', default='cifar10c_mb', type=str)
    parser.add_argument("--device", help="cuda or cpu", default='cuda', type=str)
    parser.add_argument("--num_steps", help="# of iterations", default= 30, type=int)
    parser.add_argument("--target_attr_idx", help="target_attr_idx", default= 0, type=int)
    parser.add_argument("--bias_attr_idx", help="bias_attr_idx", default= 1, type=int)
    parser.add_argument("--dataset", help="data to train", default= 'cifar10c-10', type=str)
    parser.add_argument("--percent", help="percent of bias conflit", default= "5pct", type=str)
    parser.add_argument("--sparsity", help="sparsity of biased attribute", default= "1pct", type=str)
    parser.add_argument("--corr", help="correlation between bias attribute and target attribute", default= "0.5pct", type=str)
    parser.add_argument("--use_lr_decay", action='store_true', help="whether to use learning rate decay")
    parser.add_argument("--lr_decay_step", help="learning rate decay steps", type=int, default=10000)
    parser.add_argument("--q", help="GCE parameter q", type=float, default=0.7)
    parser.add_argument("--q_d", help="GCE parameter q for debiased model", type=float, default=-0.5)
    parser.add_argument("--bias_criterion", help="criterion used to train bias model", type=str, choices=["GCE", "CE"],default="GCE")
    parser.add_argument("--debias_criterion", help="criterion used to train bias model", type=str, choices=["GCE", "CE"],default="CE")
    parser.add_argument("--lr_gamma",  help="lr gamma", type=float, default=0.1)
    parser.add_argument("--lambda_dis_align",  help="lambda_dis in Eq.2 of DisEnt", type=float, default=1.0)
    parser.add_argument("--lambda_swap_align",  help="lambda_swap_b in Eq.3 of DisEnt", type=float, default=1.0)
    parser.add_argument("--lambda_swap",  help="lambda swap (lambda_swap in Eq.4 of DisEnt)", type=float, default=1.0)
    parser.add_argument("--ema_alpha",  help="use weight mul", type=float, default=0.7)
    parser.add_argument("--curr_step", help="curriculum steps", type=int, default= 0)
    parser.add_argument("--model", help="which network, [MLP, ResNet18]", default= 'MLP', type=str)
    parser.add_argument("--tensorboard_dir", help="tensorboard directory", default= 'summary', type=str)
    parser.add_argument("--lr_decay", action="store_true") 

    # logging
    parser.add_argument("--log_dir", help='path for saving model', default='./log/cifar_log', type=str)
    parser.add_argument("--data_dir", help='path for loading data', default='./dataset', type=str)
    parser.add_argument("--valid_freq", help='frequency to evaluate on valid/test set', default=10, type=int)
    parser.add_argument("--log_freq", help='frequency to log on tensorboard', default=10, type=int)
    parser.add_argument("--tensorboard", action="store_true", help="whether to use tensorboard")
    parser.add_argument("--wandb", action="store_true", help="whether to use wandb")

    # experiment
    parser.add_argument("--alg", default="denet", help="which algorithm to train the model")
    parser.add_argument("--bias_ensm", action="store_true", help="whether to apply bias ensmble")
    parser.add_argument("--pre", type=str, default="x_", help="How to train the pretrained model")

    parser.add_argument("--fix_randomseed", action="store_true", help="fix randomseed")
    parser.add_argument("--seed",  help="seed", type=int, default=0)
    parser.add_argument("--biased_model_train_iter", type=int, default=1000, help="biased_model_stop iteration")
    parser.add_argument("--biased_model_softmax_threshold", type=float, default=0.99, help="biased_model_softmax_threshold")
    parser.add_argument("--num_bias_models", type=int, default=5, help="number of bias models")
    parser.add_argument("--resnet_pretrained", action="store_true", help="use pretrained ResNet")
    parser.add_argument("--agreement", type=int, default=3, help="number of agreement")
    parser.add_argument("--stem_end_block", type=int, default=15, help="end block of stem network in tar algorithm, its set as the second last block by default")

    #tar
    parser.add_argument('--Lambda', type=float, default=10., help='img reconstruction loss weight')
    parser.add_argument('--Lambda2', type=float, default=1, help='TV loss weight')
    
    # weighting
    parser.add_argument('--w_func', type=str, default='frac_max_s1', help='weight calcualtion function for debiased model')
    
    # shape
    parser.add_argument('--fd', type=str, default='patch-shuffle', help='feature destruction method')
    parser.add_argument('--ps', type=int, default=28, help='hyperparam for feature destruction, e.g. patch size of shuffing in patch-shuffle')
    parser.add_argument('--beta', type=float, default=0.5, help='how much shape debias is applied, 0 degrade to original training scheme')
    parser.add_argument('--biloss', type=str, default="x'", help='what loss should be used for training biased model, for shape learner. Note, the final loss is also effected by arg bias_criterion') # runs with null should be considered as x'
    parser.add_argument('--deloss', type=str, default="w_", help='what loss should be used for training debiased model, for shape learner') # runs with null should be considered as b_ce-y+ce-null
    parser.add_argument('--infer', type=str, default="_", help='inference precedure when training bias ') # runs with null should be considered as b_ce-y+ce-null
    parser.add_argument('--his', action="store_true", help='whether to log histograms') # runs with null should be considered as b_ce-y+ce-null
    
    # for dfr last layer retraining 
    parser.add_argument('--llr', type=str, default="ub", help='which version of last layer retraining is used') # runs with null 
    
    # for dpr
    parser.add_argument('--tau',  help='temperature', default=1, type=float)
    parser.add_argument('--b_step',  help='how many steps to train biased model', default=800, type=int)
    parser.add_argument('--b_max_step',  help='max steps to train biased model', default=4000, type=int)
    parser.add_argument('--pretrained_bmodel',  help='Use pretrained biased model?', action='store_true')
    parser.add_argument('--pretrained_dmodel',  help='Use pretrained debiased model?', action='store_true')
    parser.add_argument('--save_sampling_prob',    action='store_true')
    parser.add_argument('--d_init',  help='how to init the debiased model', default="rand", type=str)
    
    # for multiple bias benchmarks
    parser.add_argument("--corr2", help="correlation between bias attribute and target attribute", default= "0.98pct", type=str)
    
    args = parser.parse_args()
    args.pretrained_bmodel = True
    if args.fix_randomseed:
        random_seed = args.seed
        torch.manual_seed(random_seed)
        torch.cuda.manual_seed(random_seed)
        torch.cuda.manual_seed_all(random_seed) # if use multi-GPU
        torch.backends.cudnn.deterministic = True
        torch.backends.cudnn.benchmark = False
        np.random.seed(random_seed)
        random.seed(random_seed)

    args.wandb = True
    # init learner

    # actual training
    print('Training starts ...')

    if args.alg == "lff":
        learner = learners.LffLearner(args)
    elif args.alg == "disent":
        learner = learners.DisentLearner(args)
    elif args.alg == "disentShape":
        learner = learners.DisentShapeLearner(args)
    elif args.alg == "vanilla":
        learner = learners.Learner(args)
    elif args.alg == "tar":
        learner = learners.TarLearner(args)
    elif args.alg == "shape":
        learner = learners.ShapeLearner(args)
    elif args.alg == "dfr":
        learner = learners.DfrLearner(args)
    elif args.alg == "denet":
        learner = learners.DenetLearner(args)
    elif args.alg == "dpr":
        learner = learners.DPRLearner(args)
    else:
        print('choose one of the two options ...')
        import sys
        sys.exit(0)
        
    learner.train(args)
        
    wandb.finish()
