import torch
import torch.nn as nn
import numpy as np
import random
from typing import Any, Callable, Dict, List, Optional, Union
from dataclasses import dataclass

from diffusers.pipelines.flux.pipeline_flux import calculate_shift, retrieve_timesteps, FluxPipeline
from diffusers.pipelines.flux.pipeline_output import FluxPipelineOutput

from diffusers.utils import is_torch_xla_available, logging
from utils import *


if is_torch_xla_available():
    import torch_xla.core.xla_model as xm # type: ignore

    XLA_AVAILABLE = True
else:
    XLA_AVAILABLE = False


logger = logging.get_logger(__name__)  # pylint: disable=invalid-name

def _get_clip_prompt_embeds(
    tokenizer,
    text_encoder,
    prompt: Union[str, List[str]],
    num_images_per_prompt: int = 1,
    device: Optional[torch.device] = None,
):
    device = device or text_encoder.device
    dtype = text_encoder.dtype

    prompt = [prompt] if isinstance(prompt, str) else prompt
    batch_size = len(prompt)

    text_inputs = tokenizer(
        prompt,
        padding="max_length",
        max_length=text_encoder.config.max_position_embeddings,
        truncation=True,
        return_overflowing_tokens=False,
        return_length=False,
        return_tensors="pt",
    )

    text_input_ids = text_inputs.input_ids
    prompt_embeds = text_encoder(text_input_ids.to(device), output_hidden_states=False)

    # Use pooled output of CLIPTextModel
    prompt_embeds = prompt_embeds.pooler_output
    prompt_embeds = prompt_embeds.to(dtype=dtype, device=device)

    # duplicate text embeddings for each generation per prompt, using mps friendly method
    prompt_embeds = prompt_embeds.repeat(1, num_images_per_prompt, 1)
    prompt_embeds = prompt_embeds.view(batch_size * num_images_per_prompt, -1)

    return prompt_embeds

def _get_t5_prompt_embeds(
    tokenizer,
    text_encoder,
    prompt: Union[str, List[str]] = None,
    num_images_per_prompt: int = 1,
    max_sequence_length: int = 512,
    device: Optional[torch.device] = None,
    dtype: Optional[torch.dtype] = None,
):
    device = device or text_encoder.device
    dtype = dtype or text_encoder.dtype

    prompt = [prompt] if isinstance(prompt, str) else prompt
    batch_size = len(prompt)

    text_inputs = tokenizer(
        prompt,
        padding="max_length",
        max_length=max_sequence_length,
        truncation=True,
        return_length=False,
        return_overflowing_tokens=False,
        return_tensors="pt",
    )
    
    text_input_ids = text_inputs.input_ids

    prompt_embeds = text_encoder(text_input_ids.to(device), output_hidden_states=False)[0]
    prompt_embeds = prompt_embeds.to(dtype=dtype, device=device)

    _, seq_len, _ = prompt_embeds.shape

    # duplicate text embeddings and attention mask for each generation per prompt, using mps friendly method
    prompt_embeds = prompt_embeds.repeat(1, num_images_per_prompt, 1)
    prompt_embeds = prompt_embeds.view(batch_size * num_images_per_prompt, seq_len, -1)

    return prompt_embeds


def encode_prompt(
    tokenizers,
    text_encoders,
    prompt: Union[str, List[str]],
    prompt_2: Union[str, List[str]] = None,
    num_images_per_prompt: int = 1,
    max_sequence_length: int = 512,
):

    tokenizer_1, tokenizer_2 = tokenizers
    text_encoder_1, text_encoder_2 = text_encoders
    device = text_encoder_1.device
    dtype = text_encoder_1.dtype

    prompt = [prompt] if isinstance(prompt, str) else prompt
    prompt_2 = prompt_2 or prompt
    prompt_2 = [prompt_2] if isinstance(prompt_2, str) else prompt_2
    batch_size = len(prompt)

    # We only use the pooled prompt output from the CLIPTextModel
    pooled_prompt_embeds = _get_clip_prompt_embeds(
        tokenizer=tokenizer_1,
        text_encoder=text_encoder_1,
        prompt=prompt,
        num_images_per_prompt=num_images_per_prompt,
    )
    prompt_embeds = _get_t5_prompt_embeds(
        tokenizer=tokenizer_2,
        text_encoder=text_encoder_2,
        prompt=prompt_2,
        num_images_per_prompt=num_images_per_prompt,
        max_sequence_length=max_sequence_length,
    )

    text_ids = torch.zeros(prompt_embeds.shape[1], 3).to(device=device, dtype=dtype)

    return prompt_embeds, pooled_prompt_embeds, text_ids



class NoiseCustomFluxPipeline(FluxPipeline):
    @torch.no_grad()
    def __call__(
        self,
        prompt: Union[str, List[str]] = None,
        prompt_2: Optional[Union[str, List[str]]] = None,
        height: Optional[int] = None,
        width: Optional[int] = None,
        num_inference_steps: int = 28,
        timesteps: List[int] = None,
        guidance_scale: float = 3.5,
        num_images_per_prompt: Optional[int] = 1,
        generator: Optional[Union[torch.Generator, List[torch.Generator]]] = None,
        latents: Optional[torch.FloatTensor] = None,
        prompt_embeds: Optional[torch.FloatTensor] = None,
        pooled_prompt_embeds: Optional[torch.FloatTensor] = None,
        output_type: Optional[str] = "pil",
        return_dict: bool = True,
        joint_attention_kwargs: Optional[Dict[str, Any]] = None,
        callback_on_step_end: Optional[Callable[[int, int, Dict], None]] = None,
        callback_on_step_end_tensor_inputs: List[str] = ["latents"],
        random_sample: bool = False,
        max_sequence_length: int = 512,
        bbox_info: Optional[List[List[List[float]]]] = None,  # [[x1,y1,x2,y2], [x1,y1,x2,y2], ...]
        gaussian_kernel: bool = False,
        anisotropic: bool = False,
        beta: float = 0,
        scale_factor: float = 0.8,
        gamma: float = 0.1,
        weight: float = 0.3,
        noise_custom: Optional[torch.FloatTensor] = None,  # 预定义的noise tensor
        label: Optional[int] = None,  # 当前sample的label
        **kwargs,
    ):
        # 基本参数设置
        height = height or self.default_sample_size * self.vae_scale_factor
        width = width or self.default_sample_size * self.vae_scale_factor

        # 检查必要参数
        if noise_custom is None and label is None:
            raise ValueError("Both noise_custom and label must be provided")

        # 检查其他输入参数
        self.check_inputs(
            prompt,
            prompt_2,
            height,
            width,
            prompt_embeds=prompt_embeds,
            pooled_prompt_embeds=pooled_prompt_embeds,
            callback_on_step_end_tensor_inputs=callback_on_step_end_tensor_inputs,
            max_sequence_length=max_sequence_length,
        )

        # 设置基本参数
        self._guidance_scale = guidance_scale
        self._joint_attention_kwargs = joint_attention_kwargs
        self._interrupt = False

        # 处理batch size
        if prompt is not None and isinstance(prompt, str):
            batch_size = 1
        elif prompt is not None and isinstance(prompt, list):
            batch_size = len(prompt)
        else:
            batch_size = prompt_embeds.shape[0]

        device = self._execution_device

        # 获取text embeddings
        lora_scale = (
            self.joint_attention_kwargs.get("scale", None) if self.joint_attention_kwargs is not None else None
        )
        (prompt_embeds, pooled_prompt_embeds, text_ids) = self.encode_prompt(
            prompt=prompt,
            prompt_2=prompt_2,
            prompt_embeds=prompt_embeds,
            pooled_prompt_embeds=pooled_prompt_embeds,
            device=device,
            num_images_per_prompt=num_images_per_prompt,
            max_sequence_length=max_sequence_length,
            lora_scale=lora_scale,
        )

        # 准备latents
        num_channels_latents = self.transformer.config.in_channels // 4
        
        height_now = 2 * (int(height) // self.vae_scale_factor)
        width_now = 2 * (int(width) // self.vae_scale_factor)
        # 使用指定label的custom noise

        if latents is None:
            if noise_custom is not None:
                num_noise_patterns = noise_custom.shape[0] # 50/7
                if num_noise_patterns == 1:
                    # fixed partial noise prior
                    if bbox_info is not None:
                        noise = torch.randn(batch_size * num_images_per_prompt, num_channels_latents, height_now, width_now, device=device, dtype=prompt_embeds.dtype)
                        # random layout
                        for batch_index in range(batch_size * num_images_per_prompt):
                        
                            for box in bbox_info:
                                # 针对一个样本再随机生成一个噪声,乘以0.1来控制噪声强度
                                x_min, y_min, x_max, y_max = box # 原始512x512坐标
                                x_min = int(x_min * width_now/width)
                                y_min = int(y_min * height_now/height)
                                x_max = int(x_max * width_now/width)
                                y_max = int(y_max * height_now/height)
                                
                                # 确保坐标在有效范围内
                                x_min = max(0, min(x_min, width_now))
                                x_max = max(0, min(x_max, width_now))
                                y_min = max(0, min(y_min, height_now))
                                y_max = max(0, min(y_max, height_now))
                                box_h = y_max - y_min
                                box_w = x_max - x_min
                                patch, box_patch_y1, box_patch_y2, box_patch_x1, box_patch_x2 = get_centered_patch_and_coords(noise_custom, box_h, box_w)
                                noise[batch_index][:,y_min + box_patch_y1 : y_min + box_patch_y2, x_min + box_patch_x1 : x_min + box_patch_x2] = patch

                    else:
                        # grid layout
                        noise = torch.randn(batch_size * num_images_per_prompt, num_channels_latents, height_now, width_now, device=device, dtype=prompt_embeds.dtype)
                        row = label // 7
                        col = label % 7
                        block_size = 64 // 7
                        start_x = col * block_size
                        start_y = row * block_size
                        end_x = start_x + block_size
                        end_y = start_y + block_size
                        noise[:,:,start_y:end_y,start_x:end_x] = noise_custom[0]
                elif num_noise_patterns > 50: # 500
                    noise_indices = []
                    for _ in range(batch_size * num_images_per_prompt):
                        if random_sample:
                            noise_index = random.randint(0, num_noise_patterns-1)
                        else:
                            item_per_cls = num_noise_patterns // 50     # 10
                            random_index = random.randint(0, item_per_cls-1)
                            noise_index = label * item_per_cls + random_index
                        noise_indices.append(noise_index)
                    # print(noise_indices) # debug
                    # 使用torch.stack一次性获取所有noise
                    noise = noise_custom[noise_indices]  # batch_size * num_images_per_prompt, 16, 64, 64
                else:
                    noise_index = label % num_noise_patterns  # number_item已经是从0开始的
                    noise = noise_custom[noise_index].unsqueeze(0)  # 添加batch维度 # 1,16,64,64
                    # 对于num_noise_patterns <= 50的情况，保持原来的repeat逻辑
                    noise = noise.repeat(batch_size * num_images_per_prompt, 1, 1, 1)
            else:
                noise = torch.randn(batch_size * num_images_per_prompt, num_channels_latents, height_now, width_now, device=device, dtype=prompt_embeds.dtype)
                if bbox_info is not None:
                # random layout
                    if gaussian_kernel:
                        for batch_index in range(batch_size * num_images_per_prompt):
                            if anisotropic:
                                centers, scales = bbox_to_centers_and_scales_anisotropic(bbox_info,image_size=(height, width),scale_factor=scale_factor)
                                if beta > 0:
                                    noise_gmm_anisotropic = generate_anisotropic_gmm_noise_bbox_normalized(
                                        noise=noise[batch_index].unsqueeze(0),
                                        centers = centers,
                                        sigmas = scales,
                                        weight = weight,
                                        beta = beta,
                                        energy_norm = True
                                    )
                                else:
                                    noise_gmm_anisotropic = generate_anisotropic_gmm_noise(
                                            noise=noise[batch_index].unsqueeze(0),
                                            centers = centers,
                                            sigmas = scales,
                                            weight = weight,
                                            energy_norm = True
                                        )
                                noise[batch_index] = noise_gmm_anisotropic[0]
                            
                            else:
                                centers, scales = bbox_to_centers_and_scales(bbox_info,image_size=(height, width),scale_factor=scale_factor)
                                mean_scale = torch.tensor(scales).mean().item()

                                noise_gmm = generate_uniform_gmm_noise(
                                    noise=noise[batch_index].unsqueeze(0),
                                    centers=centers,
                                    scale=mean_scale,
                                    weight=weight,
                                )
                                noise[batch_index] = noise_gmm[0]
                    else:
                        for batch_index in range(batch_size * num_images_per_prompt):
                            noise_index = torch.randn_like(noise[0])*gamma # 16,64,64 # 一副图片的layout分布是一致的
                            for box in bbox_info:
                                # 针对一个样本再随机生成一个噪声,乘以0.1来控制噪声强度
                                x_min, y_min, x_max, y_max = box # 原始512x512坐标
                                x_min = int(x_min * width_now/width)
                                y_min = int(y_min * height_now/height)
                                x_max = int(x_max * width_now/width)
                                y_max = int(y_max * height_now/height)
                                
                                # 确保坐标在有效范围内
                                x_min = max(0, min(x_min, width_now))
                                x_max = max(0, min(x_max, width_now))
                                y_min = max(0, min(y_min, height_now))
                                y_max = max(0, min(y_max, height_now))
                                # print(f"============= {batch_index} {x_min} {y_min} {x_max} {y_max} ==============") #debug
                                # 不同sample，但是同一位置的noise是相同的
                                noise[batch_index][:,y_min:y_max,x_min:x_max] = noise_index[:,y_min:y_max,x_min:x_max]
                else:
                # grid layout
                    for batch_index in range(batch_size * num_images_per_prompt):
                        number_item = label
                        row = number_item // 7
                        col = number_item % 7
                        block_size = width_now // 7
                        start_x = col * block_size
                        start_y = row * block_size
                        end_x = start_x + block_size
                        end_y = start_y + block_size
                        if gaussian_kernel:
                            bbox_info = [[start_x, start_y, end_x, end_y]]
                            centers, scales = bbox_to_centers_and_scales(bbox_info,image_size=(height_now, width_now),scale_factor=scale_factor)
                            mean_scale = torch.tensor(scales).mean().item()
                            
                            noise_gmm = generate_uniform_gmm_noise(
                                noise=noise[batch_index].unsqueeze(0),
                                centers=centers,
                                scale=mean_scale,
                                weight=weight,
                            )
                            noise[batch_index] = noise_gmm[0]
                        else:
                            noise_index = torch.randn_like(noise[0])*gamma # 16,64,64 # 一副图片的layout分布是一致的，但是不同sample的layout分布是不同的
                            # print(f"============= {batch_index} {start_x} {start_y} {end_x} {end_y} ==============") #debug
                            noise[batch_index][:,start_y:end_y,start_x:end_x] = noise_index[:,start_y:end_y,start_x:end_x]

            noise = noise.to(device=device, dtype=prompt_embeds.dtype)
            
            # Pack latents
            latents = self._pack_latents(
                noise,
                batch_size=batch_size * num_images_per_prompt,
                num_channels_latents=num_channels_latents,
                height=height_now,
                width=width_now,
            )
        

        # 准备image ids
        latent_image_ids = self._prepare_latent_image_ids(
            batch_size * num_images_per_prompt,
            height_now,
            width_now,
            device,
            prompt_embeds.dtype,
        )

        # 准备timesteps
        sigmas = np.linspace(1.0, 1 / num_inference_steps, num_inference_steps)
        image_seq_len = latents.shape[1]
        mu = calculate_shift(
            image_seq_len,
            self.scheduler.config.base_image_seq_len,
            self.scheduler.config.max_image_seq_len,
            self.scheduler.config.base_shift,
            self.scheduler.config.max_shift,
        )
        timesteps, num_inference_steps = retrieve_timesteps(
            self.scheduler,
            num_inference_steps,
            device,
            timesteps,
            sigmas,
            mu=mu,
        )
        num_warmup_steps = max(len(timesteps) - num_inference_steps * self.scheduler.order, 0)
        self._num_timesteps = len(timesteps)

        # 处理guidance
        if self.transformer.config.guidance_embeds:
            guidance = torch.full([1], guidance_scale, device=device, dtype=torch.float32)
            guidance = guidance.expand(latents.shape[0])
        else:
            guidance = None

        # 去噪循环
        with self.progress_bar(total=num_inference_steps) as progress_bar:
            for i, t in enumerate(timesteps):
                if self.interrupt:
                    continue

                timestep = t.expand(latents.shape[0]).to(latents.dtype)

                # 预测noise
                noise_pred = self.transformer(
                    hidden_states=latents,
                    timestep=timestep / 1000,
                    guidance=guidance,
                    pooled_projections=pooled_prompt_embeds,
                    encoder_hidden_states=prompt_embeds,
                    txt_ids=text_ids,
                    img_ids=latent_image_ids,
                    joint_attention_kwargs=self.joint_attention_kwargs,
                    return_dict=False,
                )[0]

                # 计算下一步
                latents_dtype = latents.dtype
                latents = self.scheduler.step(noise_pred, t, latents, return_dict=False)[0]

                if latents.dtype != latents_dtype:
                    if torch.backends.mps.is_available():
                        latents = latents.to(latents_dtype)

                if callback_on_step_end is not None:
                    callback_kwargs = {}
                    for k in callback_on_step_end_tensor_inputs:
                        callback_kwargs[k] = locals()[k]
                    callback_outputs = callback_on_step_end(self, i, t, callback_kwargs)

                    latents = callback_outputs.pop("latents", latents)
                    prompt_embeds = callback_outputs.pop("prompt_embeds", prompt_embeds)


                # 更新进度条
                if i == len(timesteps) - 1 or ((i + 1) > num_warmup_steps and (i + 1) % self.scheduler.order == 0):
                    progress_bar.update()

                if XLA_AVAILABLE:
                    xm.mark_step()

        # 解码生成图像
        if output_type == "latent":
            image = latents
        else:
            latents = self._unpack_latents(latents, height, width, self.vae_scale_factor)
            latents = (latents / self.vae.config.scaling_factor) + self.vae.config.shift_factor
            image = self.vae.decode(latents, return_dict=False)[0]
            image = self.image_processor.postprocess(image, output_type=output_type)

        # 清理显存
        self.maybe_free_model_hooks()

        if not return_dict:
            return (image,)

        return FluxPipelineOutput(images=image)