import os
import re
import sys
from pathlib import Path
import numpy as np
import matplotlib.pyplot as plt
import matplotlib as mpl
from matplotlib import font_manager

# font_path = "/usr/share/fonts/truetype/msttcorefonts/Times_New_Roman.ttf"
# font_prop = font_manager.FontProperties(fname=font_path)
# plt.rcParams['font.family'] = font_prop.get_name()


RE_LINE = re.compile(r"CNN top1 curve:\s*\[(.*?)\]")
RE_NUM  = re.compile(r"np\.float64\(\s*([0-9.+\-eE]+)\s*\)")


def parse_numbers(raw: str) -> list[float]:
    """
    Parse the string in square brackets into a list of floats.  
    Either 71.66 or np.float64(71.66) is supported.
    """
    # replace np.float64(...) 
    cleaned = RE_NUM.sub(r"\1", raw)
    
    return [float(x.strip()) for x in cleaned.split(",") if x.strip()]

def get_last_cnn_top1_curve(log_path: str | Path) -> list[float]:
    """
    Read the log file and return the array of the last CNN top1 curve that appeared.
    """
    last_curve = None
    with open(log_path, "r", encoding="utf-8") as log:
        for line in log:
            m = RE_LINE.search(line)
            if m:
                last_curve = parse_numbers(m.group(1))  
    if last_curve is None:
        raise ValueError("no found [CNN top1 curve]")
    return last_curve

if __name__ == "__main__":
    dataset_name = 'cifar100'
    methodslist = ['der','finetune','icarl','lwf','replay','tagfex']
    data = dict()
    data_dnm = dict()
    for method in methodslist:
        root_path = f'./results/compared/{method}'
        dataset_list = os.listdir(root_path)
        for dataset in dataset_list:
            if dataset_name in dataset:
                for x in os.listdir(os.path.join(root_path,dataset)):
                    if 'none' in x:
                        # DNM
                        log_file = os.path.join(root_path,dataset,x)
                        try:
                            curve = get_last_cnn_top1_curve(log_file)
                            print("Last CNN top1 curve:", curve)
                        except Exception as err:
                            print(f"error: {err}", log_file)
                            sys.exit(1)
                        data_dnm[f'{method}'] = curve
                    else:
                        # Original
                        log_file = os.path.join(root_path,dataset,x)
                        try:
                            curve = get_last_cnn_top1_curve(log_file)
                            print("Last CNN top1 curve:", curve)
                        except Exception as err:
                            print(f"error: {err}", log_file)
                            sys.exit(1)
                        data[f'{method}'] = curve
    print(data_dnm)
    method_dict ={'der':'DER',
                  'finetune':'Finetune',
                  'icarl': 'iCarL',
                  'lwf':'LwF',
                  'replay':'Replay',
                  'tagfex':'TagFex'                  
                  }
    print(data)
    print(plt.style.available)
    ['Solarize_Light2', '_classic_test_patch', '_mpl-gallery', 
     '_mpl-gallery-nogrid', 'bmh', 'classic', 'dark_background', 
     'fast', 'fivethirtyeight', 'ggplot', 'grayscale', 
     'petroff10', 'seaborn-v0_8', 'seaborn-v0_8-bright', 
     'seaborn-v0_8-colorblind', 'seaborn-v0_8-dark', 'seaborn-v0_8-dark-palette', 'seaborn-v0_8-darkgrid', 'seaborn-v0_8-deep', 'seaborn-v0_8-muted', 'seaborn-v0_8-notebook', 'seaborn-v0_8-paper', 'seaborn-v0_8-pastel', 'seaborn-v0_8-poster', 'seaborn-v0_8-talk', 'seaborn-v0_8-ticks', 'seaborn-v0_8-white', 'seaborn-v0_8-whitegrid', 'tableau-colorblind10']
    


    plt.style.use('seaborn-v0_8-dark')
  
    plt.rcParams["font.family"] = "Times New Roman"
    plt.rcParams["fontsize"] = 14
    plt.grid(True)
    print(plt.color_sequences)
    cmap = mpl.colormaps['tab10'].colors
    lenged_name = []

    # DeL ones
    for i,(a,b) in enumerate(zip(data_dnm.keys(),data_dnm.values())):
        plt.plot(b,color=cmap[i],linewidth=1,linestyle='-',marker='o',ms =3)
        lenged_name.append(f'{method_dict[a]} w/ DeL')
    # original ones
    for i,(a,b) in enumerate(zip(data.keys(),data.values())):
        plt.plot(b,color=cmap[i], linewidth=1,linestyle='--',marker='d',ms =3)
        lenged_name.append(f'{method_dict[a]} ')
    plt.legend(lenged_name,loc='upper right',ncols=2,fontsize='x-small')
    plt.xlabel('Number of Classes')
    plt.ylabel('Accuracy')
    plt.savefig('performancecurve.png',format='png',dpi=300)
    

