
from collections import OrderedDict
import numpy as np
import pandas as pd
import os

from dataloader.dataset_short_term_forecasting import M4Dataset, M4Meta

def group_values(values, groups, group_name):

    idx = np.where(groups == group_name)[0]
    seqs = []
    for i in idx:
        v = np.asarray(values[i]).ravel()
        if np.isnan(v).any():
            v = v[~np.isnan(v)]
        seqs.append(v)
    return np.array(seqs, dtype=object)

def mase(forecast, insample, outsample, frequency, eps=1e-8):
    n = min(len(forecast), len(outsample))
    f = np.asarray(forecast[:n], dtype=float)
    y = np.asarray(outsample[:n], dtype=float)
    ins = np.asarray(insample, dtype=float)
    if len(ins) <= frequency:
        denom = eps
    else:
        diffs = np.abs(ins[frequency:] - ins[:-frequency])
        denom = np.mean(diffs) if diffs.size else eps
    num = np.mean(np.abs(f - y))
    return num / max(denom, eps)

def smape_2(forecast, target):
    denom = np.abs(target) + np.abs(forecast)
    denom[denom == 0.0] = 1.0
    return 200 * np.abs(forecast - target) / denom

def mape(forecast, target):
    denom = np.abs(target)
    denom[denom == 0.0] = 1.0
    return 100 * np.abs(forecast - target) / denom

class M4Summary:
    def __init__(self, file_path, root_path):
        self.file_path = file_path  
        self.training_set = M4Dataset.load(training=True, dataset_file=root_path)
        self.test_set = M4Dataset.load(training=False, dataset_file=root_path)
        self.naive_path = os.path.join(root_path, 'submission-Naive2.csv')

    def evaluate(self, target_groups=None):

        grouped_owa = OrderedDict()

        naive2_raw = pd.read_csv(self.naive_path).values[:, 1:].astype(np.float32)
        naive2_forecasts = np.array([row[~np.isnan(row)] for row in naive2_raw], dtype=object)

        all_groups = list(M4Meta.seasonal_patterns)
        if target_groups is None:
            available = []
            for g in all_groups:
                fp = os.path.join(self.file_path, f"{g}_forecast.csv")
                if os.path.exists(fp):
                    available.append(g)
            target_groups = available
        else:
            target_groups = [g for g in target_groups if os.path.exists(os.path.join(self.file_path, f"{g}_forecast.csv"))]

        if not target_groups:
            raise FileNotFoundError("No forecast files found for any target groups.")

        model_mases, naive2_smapes, naive2_mases = {}, {}, {}
        grouped_smapes, grouped_mapes = {}, {}

        for group_name in target_groups:
            file_name = os.path.join(self.file_path, f"{group_name}_forecast.csv")

            arr = pd.read_csv(file_name).values
            H = M4Meta.horizons_map[group_name]
            if arr.shape[1] == H + 1:
                arr = arr[:, 1:]
            arr = arr.astype(np.float32)
            model_forecast = np.array([row[~np.isnan(row)] for row in arr], dtype=object)

            naive2_forecast = group_values(naive2_forecasts, self.test_set.groups, group_name)
            target = group_values(self.test_set.values, self.test_set.groups, group_name)
            insample = group_values(self.training_set.values, self.test_set.groups, group_name)
            frequency = self.training_set.frequencies[self.test_set.groups == group_name][0]

            n = min(len(model_forecast), len(naive2_forecast), len(target), len(insample))
            if n == 0:
                continue

            model_mases[group_name] = np.mean([
                mase(model_forecast[i], insample[i], target[i], frequency) for i in range(n)
            ])
            naive2_mases[group_name] = np.mean([
                mase(naive2_forecast[i], insample[i], target[i], frequency) for i in range(n)
            ])
            grouped_smapes[group_name] = np.mean([
                np.mean(smape_2(model_forecast[i], target[i])) for i in range(n)
            ])
            naive2_smapes[group_name] = np.mean([
                np.mean(smape_2(naive2_forecast[i], target[i])) for i in range(n)
            ])
            grouped_mapes[group_name] = np.mean([
                np.mean(mape(model_forecast[i], target[i])) for i in range(n)
            ])

        grouped_smapes = self.summarize_groups(grouped_smapes, target_groups)
        grouped_mapes = self.summarize_groups(grouped_mapes, target_groups)
        grouped_model_mases = self.summarize_groups(model_mases, target_groups)
        grouped_naive2_smapes = self.summarize_groups(naive2_smapes, target_groups)
        grouped_naive2_mases = self.summarize_groups(naive2_mases, target_groups)

        for k in grouped_model_mases.keys():
            grouped_owa[k] = (grouped_model_mases[k] / grouped_naive2_mases[k] +
                              grouped_smapes[k] / grouped_naive2_smapes[k]) / 2

        def round_all(d):
            return dict(map(lambda kv: (kv[0], np.round(kv[1], 3)), d.items()))

        return (round_all(grouped_smapes),
                round_all(grouped_owa),
                round_all(grouped_mapes),
                round_all(grouped_model_mases))

    def summarize_groups(self, scores, present_groups):
    
        scores_summary = OrderedDict()

        def group_count(group_name):
            return len(np.where(self.test_set.groups == group_name)[0])

        weighted_score = {}
        total_count = len(self.test_set.groups)

        for g in ['Yearly', 'Quarterly', 'Monthly']:
            if g in present_groups and g in scores:
                weighted_score[g] = scores[g] * group_count(g)
                scores_summary[g] = scores[g]

        others_groups = ['Weekly', 'Daily', 'Hourly']
        if all(g in present_groups and g in scores for g in others_groups):
            others_score = sum(scores[g] * group_count(g) for g in others_groups)
            others_count = sum(group_count(g) for g in others_groups)
            scores_summary['Others'] = others_score / others_count
            weighted_score['Others'] = others_score

        if len(weighted_score) == 1:
            g = next(iter(weighted_score))
            scores_summary['Average'] = weighted_score[g] / group_count(g)
            return scores_summary

        total_weight = sum(group_count(k) for k in scores_summary.keys() if k != 'Others')
        if 'Others' in scores_summary:
            total_weight += sum(group_count(g) for g in others_groups)
        weighted_sum = sum(weighted_score.values())
        scores_summary['Average'] = weighted_sum / total_count if total_count > 0 else np.nan

        return scores_summary
